#!/usr/bin/env python
# -*- coding: utf-8 -*-

import sys
import re
import os
import glob
import logging
import xml.etree.ElementTree as ET
from xml.dom import minidom

from config import config
from io_util import *

CTRLpath = config.get('data', 'control_path')
DAQpath = config.get('data', 'daq_path')
MONH1path = config.get('data', 'monitor1_path')
MONH2path = config.get('data', 'monitor2_path')
MONH3path = config.get('data', 'monitor3_path')
MONE1path = config.get('data', 'monitor1_event_path')
MONE2path = config.get('data', 'monitor2_event_path')
MONE3path = config.get('data', 'monitor3_event_path')
Nexuspath = config.get('data', 'nexus_path')
softwarePath = config.get('data', 'software_path')
rawDataPrefix = config.get('data', 'raw_data_prefix')
nexusDataPrefix = config.get('data', 'nexus_data_prefix')
datasetLocationPrefix = config.get('data', 'dataset_location_prefix')

rodsbackup = config.get('irods', 'rods_backup')
rodsworkspace = config.get('irods', 'rods_public')

def getRunString(runNo):
    return "RUN" + str(runNo).zfill(7)

def getRunNoList(directory='.', pattern='^RUN[0-9]{7}'):
    runNo=[]

    for dirpath, dirnames, files in os.walk(directory):
        for dirname in dirnames:
            if re.match(pattern, dirname):
                runNo.append(int(dirname[3:]))

    runNo.sort()

    return runNo

def getControlPath(runNo):
    return CTRLpath + "RUN" + str(runNo).zfill(7) + "/"

def getDetectorPath(runNo):
    _tmp1 = DAQpath + "RUN" + str(runNo).zfill(7) + "/"
    _tmp2 = DAQpath + "RUN" + str(runNo) + "/"
    if os.path.exists(_tmp1):
        _file = _tmp1
    else:
        if os.path.exists(_tmp2):
            _file = _tmp2
        else:
            logger.info('RUN %i: no detector data found', runNo)
            _file = None

    return _file

def getDetectorFiles(runNo):
    _path = getDetectorPath(runNo)

    _files=[]

    if _path is None:
        return []

    try:
        _file1 = glob.glob(_path + "*.xml")
        _file2 = glob.glob(_path + "*.dat")
        _files = _file1 + _file2
    except:
        logger.info('RUN %i: no detector data found', runNo)
        pass

    filelist=[]
    for f in _files:
        _filename = getFileName(f)
        _filetime = getFileCreationDate(f)
        _filesize = getFileSize(f)
        _fromlocation = f
        _tolocation = rawDataPrefix + getRunString(runNo) + "/Detector/" + _filename
        _dataDescription = "detector-raw"
        _datafileFormatName = "raw_detector"
        filelist.append([_filename, _filetime, str(_filesize), _fromlocation, _tolocation, _dataDescription, _datafileFormatName])
    return filelist

def getDAQFileList(runNo):
    #ns={'d': 'http://csns.ihep.ac.cn/xml/ns/descriptor'}
    _file = getDetectorSummaryFile(runNo)
    
    if _file is None:
        return []

    with open(_file) as f:
        xmlstring = f.read()

    if 'descriptor' in xmlstring:
        #xmlstring=re.sub(' xmlns="[^"]+"', '', xmlstring, count=1)
        xmlstring=re.sub('\<daqDescriptor.+', '<root>', xmlstring)
        xmlstring=re.sub('\<\/daqDescriptor\>', '</root>', xmlstring, count=1)
    else:
        xmlstring = re.sub('\<\?xml version="1.0" encoding="UTF-8"\?\>', '<?xml version="1.0" encoding="UTF-8"?>\n<root>', xmlstring)
        xmlstring += "</root>"

    _files = []

    try:
        root = ET.fromstring(xmlstring)
        _modules=root.findall("module")
        for a in _modules:
            _child = a.getchildren()[0].find(".//file")
            for _f in _child:
                _files.append(_f.text)
    except Exception as ex:
        logger.error(ex)

    return _files

def getDetectorSummaryFile(runNo):
    path = getDetectorPath(runNo)
    if path is None:
        return None

    try:
        _file = glob.glob(path + "*summary*.xml")
        _file = _file[-1]
    except Exception as ex:
        logger.error('RUN %i: no DAQ summary file found', runNo)
        _file = None

    return _file

def getDetectorData(runNo):
    path = getDetectorPath(runNo)
    if path is None:
        return []

    try:
        _file = glob.glob(path + "BL01*.dat")
    except Exception as ex:
        logger.error(ex)
        _file=[]

    return _file

def hasDetectorPath(runNo):
    return getDetectorPath(runNo) is not None

def hasNexusPath(runNo):
    _tmp = Nexuspath + getRunString(runNo) + "/"
    return os.path.exists(_tmp)

def getMonHisPath(runNo, i):
    if i == 1:
        _path = MONH1path 
    elif i == 2:
        _path = MONH2path
    elif i == 3:
        _path = MONH3path
    else:
        return None

    _tmp = _path + "RUN" + str(runNo).zfill(7) + "/"
    if os.path.exists(_tmp):
        _file = _tmp
    else:
        logger.info('RUN %i: no histogram data found for Monitor %i', runNo, i)
        _file = None
    return _file

def hasMonitorHisPath(runNo, i):
    return getMonHisPath(runNo, i) is not None

def getMonitorHisFiles(runNo, i):
    _path = getMonHisPath(runNo, i)

    if _path is None:
        return []

    _files=[]
    try:
        _files = glob.glob( _path + "*.bin")
    except Exception as ex:
        logger.error(ex)
        return []

    filelist = []
    for f in _files:
        _filename = getFileName(f)
        
        if len(_filename) == 34:   # Old fashion file name, run <= 690
            _fileDBname = _filename[0:16] + "M" + str(i) + "_" + _filename[16:]
        else:
            _fileDBname = _filename

        _filetime = getFileCreationDate(f)
        _filesize = getFileSize(f)
        _fromlocation = f
        _tolocation = rawDataPrefix + getRunString(runNo) + "/Monitor" + str(i) + "/Histogram/" + _filename
        _dataDescription = "monitor-histogram"
        _datafileFormatName = "raw_monitor"
        filelist.append([_fileDBname, _filetime, str(_filesize), _fromlocation, _tolocation, _dataDescription, _datafileFormatName])
    return filelist

def getMonEvtPath(runNo, i):
    if i == 1:
        _path = MONE1path
    elif i == 2:
        _path = MONE2path
    elif i == 3:
        _path = MONE3path
    else:
        return None

    _tmp = _path + "RUN" + str(runNo).zfill(7) + "/"
    if os.path.exists(_tmp):
        _file = _tmp
    else:
        logger.info('RUN %i: no event data found for Monitor %i', runNo, i)
        _file = None
    return _file

def hasMonitorEvtPath(runNo, i):
    return getMonEvtPath(runNo, i) is not None

def getMonitorEvtFiles(runNo, i):
    _path = getMonEvtPath(runNo, i)

    if _path is None:
        return []

    _files = []
    try:
        _files = glob.glob(_path + "*.h5")
    except Exception as ex:
        logger.error(ex)
        return []

    filelist = []
    for f in _files:
        _filename = getFileName(f)
        if len(_filename) == 24:
            _fileDBname = _filename[0:7] + "M" + str(i) + "_" + _filename[7:]
        elif len(_filename) == 41:  # Old fashion file name, run <= 690
            _fileDBname = _filename[0:16] + "M" + str(i) + "_" + _filename[16:]
        else:
            _fileDBname = _filename

        _filetime = getFileCreationDate(f)
        _filesize = getFileSize(f)
        _fromlocation = f
        _tolocation = rawDataPrefix + getRunString(runNo) + "/Monitor" + str(i) + "/Event/" + _filename
        _dataDescription = "monitor-nexus"
        _datafileFormatName = "nexus_monitor"
        filelist.append([_fileDBname, _filetime, str(_filesize), _fromlocation, _tolocation, _dataDescription, _datafileFormatName])
    return filelist

def getNexusPath(runNo):
    _tmp = Nexuspath + getRunString(runNo) + "/"
    if os.path.exists(_tmp):
        return _tmp
    else:
        return None

def getSoftwarePath(runNo):
    _tmp = softwarePath + getRunString(runNo) + "/"
    if os.path.exists(_tmp):
        return _tmp
    else:
        return None

def makeSoftwarePath(runNo):
    if getSoftwarePath(runNo) is None:
        _tmp = softwarePath + getRunString(runNo) + "/"
        logger.info("Making path: %s", _tmp)
        os.makedirs(_tmp)

def getNexusFiles(runNo):
    _path = getNexusPath(runNo)
    
    if _path is None:
        return []

    _files=[]
    try:
        _files = glob.glob( _path + "*.nxs")
    except Exception as ex:
        logger.error(ex)
        return []

    filelist=[]
    for f in _files:
        _filename = getFileName(f)
        _filetime = getFileCreationDate(f)
        _filesize = getFileSize(f)
        _fromlocation = f
        _tolocation = nexusDataPrefix + getRunString(runNo) + "/" + _filename
        _dataDescription = "nexus"
        _datafileFormatName = "nexus"
        filelist.append([_filename, _filetime, str(_filesize), _fromlocation, _tolocation, _dataDescription, _datafileFormatName])

    return filelist

def getSoftwareFiles(runNo):
    _path = getSoftwarePath(runNo)

    if _path is None:
        return []

    _files=[]
    try:
        _files = glob.glob( _path + "*.xml")
    except Exception as ex:
        logger.error(ex)
        return []

    filelist=[]
    for f in _files:
        _filename = getFileName(f)
        _filetime = getFileCreationDate(f)
        _filesize = getFileSize(f)
        _fromlocation = f
        _tolocation = rawDataPrefix + getRunString(runNo) + "/Software/" + _filename
        _dataDescription = "software summary"
        _datafileFormatName = "software_summary"
        filelist.append([_filename, _filetime, str(_filesize), _fromlocation, _tolocation, _dataDescription, _datafileFormatName])

    return filelist

def getControlSummaryFile(runNo):
    _path = getControlPath(runNo)

    if _path is None:
        return None
    try:
        _files = glob.glob(_path + "CSNS-TS1-BL01-CTRL-Summary-*.xml")
    except Exception as ex:
        logger.error(ex)
        return None

    if _files:
        _file = _files[0]
    else:
        _file = None

    return _file

def hasControlSummaryFile(runNo):
    return getControlSummaryFile(runNo) is not None

def getFirstElementValue(element, path, strip=True):
    _tmp = element.findall(path)
    if len(_tmp) > 0:
        _v = _tmp[0].text
        if _v is None:
            return _v

        if strip:
            return _v.strip()
        else:
            return _v
    else:
        return None

def parseControlSummaryFile(runNo):
    _file = getControlSummaryFile(runNo)

    # has control summary file
    if _file is not None:
        try:
            root = ET.parse(_file).getroot()

            _tmp = getFirstElementValue(root, "NXentry/start_time_utc")
            if _tmp is not None and len(_tmp) > 0:
                startDatetime = re.sub(' ', 'T', _tmp) + '.000+08:00' 
            else:
                startDatetime = "unknown"

            _tmp = getFirstElementValue(root, "NXentry/end_time_utc")
            if _tmp is not None and len(_tmp) > 0:
                endDatetime = re.sub(' ', 'T', _tmp) + '.000+08:00' 
            else:
                endDatetime = "unknown"

            _v = ""
            _tmp1 = getFirstElementValue(root, "NXentry/description")
            _tmp2 = getFirstElementValue(root, "NXentry/measurement_type")
            if _tmp1 is not None and len(_tmp1) > 0:
                _v = _v + _tmp1
            if _tmp2 is not None and len(_tmp2) > 0:
                _v = _v + ":" + _tmp2
            _v = _v.strip(' :')
            if len(_v) > 0:
                datasetDescription = _v
            else:
                datasetDescription = 'unknown'

            user_id = getFirstElementValue(root, "NXentry/user_id")
            if user_id is None or len(user_id) == 0:
                user_id = "BL01-user"

            proposal_id = getFirstElementValue(root, "NXentry/proposal_id")
            if proposal_id is None or len(proposal_id) == 0:
                if int(runNo) <= 847:
                    proposal_id = "BL01-obsolete"
                else:
                    proposal_id = "BL01-commissioning"

            _tmp = getFirstElementValue(root, "NXentry/NXinstrument/NXsample/name")
            if _tmp is not None and len(_tmp) > 0:
                sampleTypeName = _tmp 
            else:
                sampleTypeName = "unknown"

            sampleName = sampleTypeName

            _tmp = getFirstElementValue(root, "NXentry/NXinstrument/NXsample/chemical_formula")
            if _tmp is not None and len(_tmp) > 0:
                sampleTypeMolecularFormula = _tmp 
            else:
                sampleTypeMolecularFormula = "unknown"

        # error
        except Exception as ex:
            logger.warning(str(ex))
            user_id = "BL01-user"
            if int(runNo) <= 847:
                proposal_id = "BL01-obsolete"
            else:
                proposal_id = "BL01-commissioning"
            startDatetime = "unknown"
            endDatetime = "unknown"
            datasetDescription = "unknown"
            sampleTypeName = "unknown"
            sampleName = "unknown"
            sampleTypeMolecularFormula = "unknown"

    # no control summary file
    else:
        logger.warning('No control summary file found for run %s' % (runNo))
        user_id = "BL01-user"
        if int(runNo) <= 847:
            proposal_id = "BL01-obsolete"
        else:
            proposal_id = "BL01-commissioning"
        startDatetime = "unknown"
        endDatetime = "unknown"
        datasetDescription = "unknown"
        sampleTypeName = "unknown"
        sampleName = "unknown"
        sampleTypeMolecularFormula = "unknown"

    # Run meta data
    datasetName = '__csns__' + getRunString(runNo)
    dataLocation = datasetLocationPrefix + getRunString(runNo)

    _dict = {}
    _dict['startDatetime'] = startDatetime
    _dict['endDatetime'] = endDatetime
    _dict['userName'] = user_id
    _dict['investigationName'] = proposal_id
    _dict['datasetName'] = datasetName
    _dict['dataLocation'] = dataLocation
    _dict['datasetDescription'] = datasetDescription
    _dict['sampleTypeName'] = sampleTypeName
    _dict['sampleName'] = sampleName
    _dict['sampleTypeMolecularFormula'] = sampleTypeMolecularFormula

    return _dict

def getControlFiles(runNo):
    _path = getControlPath(runNo)

    if _path is None:
        return []

    _files = []
    try:
        _files = glob.glob(_path + "*.xml")
    except Exception as ex:
        logger.error(ex)
        return [] 

    filelist = []
    for f in _files:
        _filename = getFileName(f)
        _filetime = getFileCreationDate(f)
        _filesize = getFileSize(f)
        _fromlocation = f
        _tolocation = rawDataPrefix + getRunString(runNo) + "/Control/" + _filename
        _dataDescription = "control-raw"
        _datafileFormatName = "control_summary"
        filelist.append([_filename, _filetime, str(_filesize), _fromlocation, _tolocation, _dataDescription, _datafileFormatName])

    return filelist

def getRunTime(runNo):
    _file = getControlSummaryFile(runNo)

    if _file is None:
        return "",""

    tree = ET.parse(_file)
    root = tree.getroot()

    _tmp = root.findall("NXentry/start_time_utc")[0].text
    if _tmp:
        _tmp = re.sub(' ', 'T', _tmp)
        _tmp += ".000+08:00"
        beginDate = _tmp
    else:
        beginData = ""

    _tmp = root.findall('NXentry/end_time_utc')[0].text
    if _tmp:
        _tmp = re.sub(' ', 'T', _tmp)
        _tmp += ".000+08:00"
        endDate = _tmp
    else:
        endData = ""

    return beginDate, endDate

def writeXml(runNo, beginTime, endTime, controlFiles, detectorFiles, monitor1hisFiles, monitor2hisFiles, monitor3hisFiles, monitor1evtFiles, monitor2evtFiles, monitor3evtFiles, nexusFiles):
    runString = getRunString(runNo)
    try:
        makeSoftwarePath(runNo)
    except Exception as ex:
        logger.error(ex)
        return False

    top = ET.Element('root')

    child = ET.SubElement(top, 'run')
    child.text = runString

    child = ET.SubElement(top, 'begin')
    child.text = beginTime

    child = ET.SubElement(top, 'end')
    child.text = endTime

    _num = len(controlFiles)
    control = ET.SubElement(top, 'contorl')
    control.set('number',str(_num))
    for filename, filecreation, filesize, fromlocation, tolocation, _, _ in controlFiles:
        _file = ET.SubElement(control, 'file')
        _child = ET.SubElement(_file, 'filename')
        _child.text = filename

        _child = ET.SubElement(_file, 'filesize')
        _child.text = filesize

        _child = ET.SubElement(_file, 'creationtime')
        _child.text = filecreation

        _child = ET.SubElement(_file, 'from')
        _child.text = fromlocation

        _child = ET.SubElement(_file, 'to')
        _child.text = tolocation

    _num = len(detectorFiles)
    detector = ET.SubElement(top, 'detector')
    detector.set('number',str(_num))
    for filename, filecreation, filesize, fromlocation, tolocation, _, _ in detectorFiles:
        _file = ET.SubElement(detector, 'file')
        _child = ET.SubElement(_file, 'filename')
        _child.text=filename

        _child = ET.SubElement(_file, 'filesize')
        _child.text=filesize

        _child = ET.SubElement(_file, 'creationtime')
        _child.text=filecreation

        _child = ET.SubElement(_file, 'from')
        _child.text=fromlocation

        _child = ET.SubElement(_file, 'to')
        _child.text=tolocation

    for k in range(3):
        if k == 0:
            _list = monitor1hisFiles
            _name = 'monitor1_histogram'
        elif k == 1:
            _list = monitor2hisFiles
            _name = 'monitor2_histogram'
        elif k == 2:
            _list = monitor3hisFiles
            _name = 'monitor3_histogram'
        else:
            pass

        _num = len(_list)
        _monitor = ET.SubElement(top, _name)
        _monitor.set('number',str(_num))
        for filename, filecreation, filesize, fromlocation, tolocation, _, _ in _list:
            _file = ET.SubElement(_monitor, 'file')
            _child = ET.SubElement(_file, 'filename')
            _child.text=filename

            _child = ET.SubElement(_file, 'filesize')
            _child.text=filesize

            _child = ET.SubElement(_file, 'creationtime')
            _child.text=filecreation

            _child = ET.SubElement(_file, 'from')
            _child.text=fromlocation

            _child = ET.SubElement(_file, 'to')
            _child.text=tolocation

    for k in range(3):
        if k == 0:
            _list = monitor1evtFiles
            _name = 'monitor1_event'
        elif k == 1:
            _list = monitor2evtFiles
            _name = 'monitor2_event'
        elif k == 2:
            _list = monitor3evtFiles
            _name = 'monitor3_event'
        else:
            pass

        _num=len(_list)
        _monitor = ET.SubElement(top, _name)
        _monitor.set('number',str(_num))
        for filename, filecreation, filesize, fromlocation, tolocation, _, _ in _list:
            _file = ET.SubElement(_monitor, 'file')
            _child = ET.SubElement(_file, 'filename')
            _child.text = filename

            _child = ET.SubElement(_file, 'filesize')
            _child.text = filesize

            _child = ET.SubElement(_file, 'creationtime')
            _child.text = filecreation

            _child = ET.SubElement(_file, 'from')
            _child.text = fromlocation

            _child = ET.SubElement(_file, 'to')
            _child.text = tolocation

    _num = len(nexusFiles)
    nexus = ET.SubElement(top, 'nexus')
    nexus.set('number',str(_num))
    for filename, filecreation, filesize, fromlocation, tolocation, _, _ in nexusFiles:
        _file = ET.SubElement(nexus, 'file')
        _child = ET.SubElement(_file, 'filename')
        _child.text = filename

        _child = ET.SubElement(_file, 'filesize')
        _child.text = filesize

        _child = ET.SubElement(_file, 'creationtime')
        _child.text = filecreation

        _child = ET.SubElement(_file, 'from')
        _child.text = fromlocation

        _child = ET.SubElement(_file, 'to')
        _child.text=tolocation

    xmlstr = minidom.parseString(ET.tostring(top)).toprettyxml(indent="   ")
    #print(xmlstr)
    with open(getSoftwarePath(runNo) + runString + ".xml", "w") as f:
        f.write(xmlstr)

    return True

def setControlLocation(runNo, _filename):
    backup= rawDataPrefix + getRunString(runNo) + "/Control/" + _filename
    return backup

def setDetectorLocation(runNo, _filename):
    backup= rawDataPrefix + getRunString(runNo) + "/Detector/" + _filename
    return backup

def setMonitorHisLocation(runNo, k, _filename):
    backup= rawDataPrefix + getRunString(runNo) + "/Monitor" + str(k) + "/Histogram/" + _filename
    return backup

def setMonitorEvtLocation(runNo, k, _filename):
    backup= rawDataPrefix + getRunString(runNo) + "/Monitor" + str(k) + "/Event/" + _filename
    return backup

def setNexusLocation(runNo, k, _filename):
    backup= nexusDataPrefix + getRunString(runNo) + "/" + _filename
    return backup

