#!/usr/bin/env python
# -*- coding: utf-8 -*-

import re
from datetime import datetime, timedelta
import icat
from config import config
from io_util import *

icatHost = config.get('icat', 'server')
icatUser = config.get('icat', 'user')
icatPasswd = config.get('icat', 'password')
icatAuth = config.get('icat', 'auth')

def getIcatSession():
    icatClient=icat.ICAT(icatHost)
    credentials={"credentials": [{"username": icatUser}, {"password": icatPasswd}]}
    return icatClient.login(icatAuth, credentials)

def isIcatSessionValid(icatSession):
    try:
        _tmp = icatSession.getUserName()
        if _tmp is None or _tmp == "":
            return False
        else:
            return True
    except:
        return False

def getFacilityId(icatSession, facility='CSNS'):
    _query = "SELECT e.id From Facility e WHERE e.name='%s'" % facility
    _tmp = icatSession.search(_query)
    if len(_tmp) == 0:
        return None
    return _tmp[0]

def getInstrumentId(icatSession, instrument='BL01', facility='CSNS'):
    _query = "SELECT e.id From Instrument e WHERE e.name='%s' and e.facility.name='%s'" % (instrument, facility)
    _tmp = icatSession.search(_query)
    if len(_tmp) == 0:
        return None
    return _tmp[0]

def getInvestigationId(icatSession, investigation, visit=None, facility='CSNS'):
    if visit is None:
        visit = investigation

    _query = "SELECT e.id From Investigation e WHERE e.name='%s' and e.visitId='%s' and e.facility.name='%s'" % (investigation, visit, facility)
    _tmp = icatSession.search(_query)
    if len(_tmp) == 0:
        return None
    return _tmp[0]

def getInvestigationDate(icatSession, investigationId):
    _query = "SELECT e.startDate, e.endDate From Investigation e WHERE e.id=%s" % investigationId
    _tmp = icatSession.search(_query)
    if len(_tmp) == 0:
        return [None, None]
    return _tmp[0]

def getSampleTypeId(icatSession, sampleTypeName, molecularFormula, facility='CSNS'):
    _query = "SELECT e.id From SampleType e WHERE e.name='%s' and e.molecularFormula='%s' and e.facility.name='%s'" % (sampleTypeName, molecularFormula, facility)
    _tmp = icatSession.search(_query)
    if len(_tmp) == 0:
        return None
    return _tmp[0]

def getSampleId(icatSession, investigationId, sampleName):
    _query = "SELECT e.id From Sample e WHERE e.investigation.id=%s and e.name='%s'" % (investigationId, sampleName)
    _tmp = icatSession.search(_query)
    if len(_tmp) == 0:
        return None
    return _tmp[0]

def getDatasetTypeId(icatSession, name, facility='CSNS'):
    _query = "SELECT e.id From DatasetType e WHERE e.name='%s' and e.facility.name='%s'" % (name, facility)
    _tmp = icatSession.search(_query)
    if len(_tmp) == 0:
        return None
    return _tmp[0]

def getDatasetId(icatSession, investigationId, datasetName):
    _query = "SELECT e.id From Dataset e WHERE e.name='%s' and e.investigation.id=%s" % (datasetName, investigationId)
    _tmp = icatSession.search(_query)
    if len(_tmp) == 0:
        return None
    return _tmp[0]

def getDatafileFormatId(icatSession, formatName, formatVersion, facility='CSNS'):
    _query = "SELECT e.id From DatafileFormat e WHERE e.name='%s' and e.version='%s' and e.facility.name='%s'" % (formatName, formatVersion, facility)
    _tmp = icatSession.search(_query)
    if len(_tmp) == 0:
        return None
    return _tmp[0]

def getDatafileId(icatSession, datasetId, name):
    _query = "SELECT e.id From Datafile e WHERE e.name='%s' and e.dataset.id=%s" % (name, datasetId)
    _tmp = icatSession.search(_query)
    if len(_tmp) == 0:
        return None
    return _tmp[0]

"""
Add sample to a investigation.
"""
def setSample(icatSession, investigationId, sampleTypeId, sampleName='unknown'):
    entities = [{"Sample": {"type":{"id":sampleTypeId},"investigation":{"id":investigationId},"name":sampleName}}]
    _tmp = icatSession.write(entities)
    return _tmp[0]

def setSampleType(icatSession, facilityId, sampleTypeName="unknown", molecularFormula="unknown"):
    entities = [{"SampleType":{"facility":{"id":facilityId},"molecularFormula":molecularFormula,"name":sampleTypeName}}]
    _tmp = icatSession.write(entities)
    return _tmp[0]

def setDataset(icatSession, investigationId, datasetTypeId, sampleId, datasetName, location, startDatetime, endDatetime, description, complete=True, doi=None):
    _ds = {"investigation": {"id": investigationId}, "sample": {"id": sampleId}, "type": {"id": datasetTypeId},"complete": complete, "location": location, "name": datasetName, "description": description}

    if startDatetime is not None and startDatetime.strip().lower() != 'unknown':
        _ds['startDate'] = startDatetime
    if endDatetime is not None and endDatetime.strip().lower() != 'unknown':
        _ds['endDate'] = endDatetime

    if doi is not None and len(doi.strip()) > 0:
        _ds['doi'] = doi
    
    entities = [{"Dataset": _ds}]
    _tmp = icatSession.write(entities)
    return _tmp[0]

def updateDataset(icatSession, datasetId, investigationId=None, datasetTypeId=None, sampleId=None, name=None, startDateTime=None, description=None, complete=None, location=None, endDateTime=None, doi=None):
    _df = {"id": datasetId}
    if investigationId is not None:
        _df['investigation'] = {"id": investigationId}
    if datasetTypeId is not None:
        _df['type'] = {"id": datasetTypeId}
    if sampleId is not None:
        _df['sample'] = {"id": sampleId}
    if name is not None and len(name.strip()) > 0:
        _df['name'] = name
    if startDateTime is not None and len(startDateTime.strip()) > 0:
        _df['startDate'] = startDateTime
    if description is not None and len(description.strip()) > 0:
        _df['description'] = description
    if complete is not None:
        _df['complete'] = complete
    if location is not None and len(location.strip()) > 0:
        _df['location'] = location
    if endDateTime is not None and len(endDateTime.strip()) > 0:
        _df['endDate'] = endDateTime
    if doi is not None and len(doi.strip()) > 0:
        _df['doi'] = doi

    if len(_df.keys()) <= 1:
        return False
    entities = [{"Dataset": _df}]
    _tmp = icatSession.write(entities)
    return True

def setDatafile(icatSession, datasetId, datafileFormatId, name, createTime, description, size, location, modTime=None, doi=None, checksum=None):
    _df = {"dataset": {"id": datasetId}, "datafileFormat": {"id": datafileFormatId}, "datafileCreateTime": createTime, "description": description, "fileSize": int(size), "location": location, "name": name}
    if modTime is not None and len(modTime.strip()) > 0:
        _df['datafileModTime'] = modTime
    if doi is not None and len(doi.strip()) > 0:
        _df['doi'] = doi
    if checksum is not None and len(checksum.strip()) > 0:
        _df['checksum'] = checksum

    entities = [{"Datafile": _df}]
    _tmp = icatSession.write(entities)
    return _tmp[0]

def updateDatafile(icatSession, datafileId, datasetId=None, datafileFormatId=None, name=None, createTime=None, description=None, size=None, location=None, modTime=None, doi=None, checksum=None):
    _df = {"id": datafileId}
    if datasetId is not None > 0:
        _df['dataset'] = {"id": datasetId}
    if datafileFormatId is not None:
        _df['datafileFormat'] = {"id": datafileFormatId}
    if name is not None and len(name.strip()) > 0:
        _df['name'] = name
    if createTime is not None and len(createTime.strip()) > 0:
        _df['datafileCreateTime'] = createTime
    if description is not None and len(description.strip()) > 0:
        _df['description'] = description
    if size is not None:
        _df['fileSize'] = int(size)
    if location is not None and len(location.strip()) > 0:
        _df['location'] = location
    if modTime is not None and len(modTime.strip()) > 0:
        _df['datafileModTime'] = modTime
    if doi is not None and len(doi.strip()) > 0:
        _df['doi'] = doi
    if checksum is not None and len(checksum.strip()) > 0:
        _df['checksum'] = checksum

    if len(_df.keys()) <= 1:
        return False
    entities = [{"Datafile": _df}]
    _tmp = icatSession.write(entities)
    return True

####### TODO: Update works for User, Dataset, etc. But not for Investigation, why?
def setInvestigationDate(icatSession, conf, investigationId):
    try:
        oldStartDate, oldEndDate = getInvestigationDate(icatSession, investigationId)
        if oldStartDate is None:
            oldStartDate = ""
        if oldEndDate is None:
            oldEndDate = ""
    except:
        oldStartDate=""
        oldEndDate=""

    newStartDate = conf['startDatetime']
    newEndDate = conf['endDatetime']

    boolStart = False
    startDate = oldStartDate
    if newStartDate != "unknown":
        _newDate = datetime.strptime(newStartDate, '%Y-%m-%dT%H:%M:%S.%f+08:00')
        if oldStartDate == "":
            startDate = newStartDate
            boolStart = True
        else:
            _oldDate = datetime.strptime(oldStartDate, '%Y-%m-%dT%H:%M:%S.%f+08:00')
            if (_oldDate - _newDate) > timedelta(seconds = 1):
                startDate = newStartDate
                boolStart = True

    boolEnd = False
    endDate = oldEndDate
    if newEndDate != "unknown":
        _newDate = datetime.strptime(newEndDate, '%Y-%m-%dT%H:%M:%S.%f+08:00')
        if oldEndDate == "":
            endDate = newEndDate
            boolEnd = True
        else:
            _oldDate = datetime.strptime(oldEndDate, '%Y-%m-%dT%H:%M:%S.%f+08:00')
            if (_newDate - _oldDate) > timedelta(seconds = 1):
                endDate = newEndDate
                boolEnd=True

    if boolStart and boolEnd:
        entities=[{"Investigation":{"id": investigationId, "startDate": startDate, "endDate": endDate}}]
    if boolStart and not boolEnd:
        entities=[{"Investigation":{"id": investigationId, "startDate": startDate}}]
    if not boolStart and boolEnd:
        entities=[{"Investigation":{"id": investigationId, "endDate": endDate}}]
    if not boolStart and not boolEnd:
        return None

    _tmp = icatSession.write(entities)
    return _tmp[0]

"""
Catalog a dataset into ICAT.
"""
def catalogDataset(icatSession, facilityId, investigationId, sampleTypeName, sampleName, molecularFormula, datasetName, dataLocation, startDatetime, endDatetime, dataDescription):
    # sampletype id
    try:
        sampleTypeId = setSampleType(icatSession, facilityId, sampleTypeName = sampleTypeName, molecularFormula = molecularFormula)
    except icat.IcatException as ex:
        if ex.getType() == 'OBJECT_ALREADY_EXISTS':
            sampleTypeId = getSampleTypeId(icatSession, sampleTypeName, molecularFormula)
        else:
            raise

    # sample id
    try:
        sampleId = setSample(icatSession, investigationId, sampleTypeId, sampleName=sampleName)
    except icat.IcatException as ex:
        if ex.getType() == 'OBJECT_ALREADY_EXISTS':
            sampleId = getSampleId(icatSession, investigationId, sampleName)
        else:
            raise

    # datasettype id
    datasetTypeId = getDatasetTypeId(icatSession, 'raw')

    try:
        datasetId = setDataset(icatSession, investigationId, datasetTypeId, sampleId, datasetName, dataLocation, startDatetime, endDatetime, dataDescription)
    except icat.IcatException as ex:
        if ex.getType() == 'OBJECT_ALREADY_EXISTS':
            datasetId = getDatasetId(icatSession, investigationId, datasetName) 
        else:
            raise
    
    return datasetId

"""
Catalog a datafile into ICAT.
"""
def catalogDatafile(icatSession, datasetId, datafileFormatName, datafileName, dataCreateTime, dataDescription, dataSize, dataLocation):
    datafileFormatId = getDatafileFormatId(icatSession, datafileFormatName, '1.0')
    try:
        datafileId = setDatafile(icatSession, datasetId, datafileFormatId, datafileName, dataCreateTime, dataDescription, dataSize, dataLocation)
    except icat.IcatException as ex:
        if ex.getType() == 'OBJECT_ALREADY_EXISTS':
            datafileId = getDatafileId(icatSession, datasetId, datafileName)
            updateDatafile(icatSession, datafileId, size=dataSize, location=dataLocation)
        else:
            raise
            
    return datafileId

"""
Catalog data into ICAT, both for dataset and datafile.
"""
def catalogData(icatSession, conf, facilityId, investigationId, files):
    datasetId = catalogDataset(icatSession, facilityId, investigationId, conf['sampleTypeName'], conf['sampleName'], conf['sampleTypeMolecularFormula'], conf['datasetName'], conf['dataLocation'], conf['startDatetime'], conf['endDatetime'], conf['datasetDescription'])

    for datafileName, dataCreateTime, dataSize,  _, dataLocation, dataDescription, datafileFormatName in files:
        catalogDatafile(icatSession, datasetId, datafileFormatName, datafileName, dataCreateTime, dataDescription, int(dataSize), dataLocation)
    
    return True

