#!/usr/bin/env python
# -*- coding: utf-8 -*-

import logging
from sqlalchemy import Column, String, create_engine
from sqlalchemy.orm import sessionmaker
from sqlalchemy.ext.declarative import declarative_base
from sqlalchemy import create_engine, Column, Integer, SmallInteger, String, DateTime, text
from config import config

field_text = text

Base = declarative_base()

class Record(Base):

    __tablename__ = 'BL01'

    id = Column(Integer, primary_key=True)
    beamline = Column(String(64), nullable=False)
    run = Column(String(64), nullable=False, index=True)
    ctrl = Column(Integer, nullable=False)
    daq = Column(Integer, nullable=False)
    monhis1 = Column(Integer, nullable=False)
    monhis2 = Column(Integer, nullable=False)
    monhis3 = Column(Integer, nullable=False)
    monevt1 = Column(Integer, nullable=False)
    monevt2 = Column(Integer, nullable=False)
    monevt3 = Column(Integer, nullable=False)
    nexus = Column(Integer, nullable=False)
    begin = Column(String(64), nullable=True)
    end = Column(String(64), nullable=True)
    stat = Column(Integer, nullable=False, index=True)
    create_time = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"))
    update_time = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"), server_onupdate=text("CURRENT_TIMESTAMP"))

    def __repr__(self):
        return '%s(%r)' % (self.__class__.__name__, self.run)

class UserNexusRecord(Base):

    __tablename__ = 'user_nexus'

    id = Column(Integer, primary_key=True)
    beamline = Column(String(4), nullable=False)
    run = Column(String(10), nullable=False, index=True)
    file_name = Column(String(255), nullable=False)
    from_location = Column(String(255), nullable=False)
    to_location = Column(String(255), nullable=False)
    stat = Column(SmallInteger, nullable=False, index=True, default=0)
    create_time = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"))
    update_time = Column(DateTime, nullable=False, server_default=text("CURRENT_TIMESTAMP"), server_onupdate=text("CURRENT_TIMESTAMP"))
    
    def __repr__(self):
        return '%s(%r, %r, %r)' % (self.__class__.__name__, self.beamline, self.run, self.file_name)

def getDBsession():
    host = config.get('mysqldb','server')
    db = config.get('mysqldb','database')
    user = config.get('mysqldb','user')
    passwd = config.get('mysqldb','password')

    engine = create_engine('mysql+mysqlconnector://' + user + ':' + passwd + '@' + host + '/' + db)
    return sessionmaker(bind=engine)()

def persist_user_nexus_record(session, record):
    if not isinstance(record, UserNexusRecord):
        logging.getLogger(__name__).error("The record must be an instantance of UserNexusRecord")
        return False
    
    try:
        t = session.query(UserNexusRecord).filter(UserNexusRecord.beamline == record.beamline, UserNexusRecord.run == record.run, UserNexusRecord.file_name == record.file_name).one_or_none()
        if t is None:
            session.add(record)
        else:
            t.from_location = record.from_location
            t.to_location = record.to_location
            if record.stat is not None:
                t.stat = record.stat
            t.update_time = text("CURRENT_TIMESTAMP")
        try:
            session.commit()
            return True
        except Exception as ex:
            logging.getLogger(__name__).error(str(ex))
            session.rollback()
            return False
    except Exception as ex:
        logging.getLogger(__name__).error(str(ex))
        return False
