#!/usr/bin/python

import sys
import os
import epics
import redis
import neon
import datetime
import time
import random
import numpy as np
import warnings
import json
import threading
import subprocess
import shlex

#import warnings
#warnings.filterwarnings("ignore")

import logging
from logging.handlers import RotatingFileHandler
logger = logging.getLogger('SANScockpit')
hdlr = RotatingFileHandler(os.getcwd() + '/SANScockpit.log', maxBytes=100
                           * 1024 * 1024, backupCount=1)
formatter = logging.Formatter('%(asctime)s %(levelname)s %(message)s')
hdlr.setFormatter(formatter)
logger.addHandler(hdlr)
logger.setLevel(logging.DEBUG)

SANSgroup=[]
SANSgroup.append('module01')
SANSgroup.append('module02')
SANSgroup.append('module03')
SANSgroup.append('module04')
SANSgroup.append('module05')
SANSgroup.append('module06')
SANSgroup.append('module07')
SANSgroup.append('module08')
SANSgroup.append('module09')
SANSgroup.append('module10')
SANSgroup.append('monitor1')
SANSgroup.append('monitor2')
SANSgroup.append('monitor3')
SANSdroneInfo=[]
for i in range(len(SANSgroup)):
    SANSdroneInfo.append({})
    SANSdroneInfo[i]['host']='10.1.34.118'
    SANSdroneInfo[i]['module']=SANSgroup[i]
    SANSdroneInfo[i]['pid']=-1
    SANSdroneInfo[i]['running']='0'
    SANSdroneInfo[i]['rpc']='0'
    SANSdroneInfo[i]['pobox']='0'
    SANSdroneInfo[i]['heartbeat']='/SANS/heartbeat/detector/'+SANSgroup[i]

fh = 0


def run_once():
    import fcntl
    global fh
    fh = open(os.path.realpath(__file__), 'r')
    try:
        fcntl.flock(fh, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except:
        os._exit(1)


def run_this():
    mypid = str(os.getpid())
    myname = os.path.basename(__file__)

    # _cmd="ps -ef | grep "+myname+" | grep -v grep | awk '{print $2}'"

    _cmd = 'ps -C ' + myname + ' -o pid='
    pid = subprocess.check_output(_cmd, shell=True)
    pid = shlex.split(pid)
    for p in pid:
        if p != mypid:
            try:
                _cmd = '/usr/bin/kill -9 ' + str(p)

                # _cmd = shlex.split(_cmd)

                subprocess.check_call(_cmd, shell=True)
                logger.warning('Last process killed')
            except:
                logger.warning('Last process not killed')
        else:
            pass

def kill_all():
    mypid = str(os.getpid())
    myname = os.path.basename(__file__)

    _cmd = 'ps -ef | grep ' + myname \
        + " | grep -v grep | awk '{print $2}'"

    # _cmd="ps -C "+myname+" -o pid="

    pid = subprocess.check_output(_cmd, shell=True)
    pid = shlex.split(pid)
    for p in pid:
        if p != mypid:
            try:
                _cmd = '/usr/bin/kill -9 ' + str(p)

                # _cmd = shlex.split(_cmd)

                subprocess.check_call(_cmd, shell=True)
                logger.warning('Last process killed')
            except:
                logger.warning('Last process not killed')
        else:
            pass

    _cmd = '/usr/bin/kill -9 ' + str(mypid)
    subprocess.call(_cmd, shell=True)

def getTime():
    return time.strftime('%Y-%m-%d %H:%M:%S')

class getRedisServer():

   def __init__(
        self,
        ip,
        passwd,
        timeout,
        ):

        from redis.sentinel import Sentinel

        self.status=False
        self.redisWrite=None
        self.redisRead=None
        begin=time.time()
        while True:
            if self.redisWrite is None:
                sentinel = Sentinel(ip, socket_timeout=0.1)
                try:
                    self.redisWrite=sentinel.master_for('neonmaster', socket_timeout=30, password=passwd)
                    self.redisRead=sentinel.slave_for('neonmaster', socket_timeout=30, password=passwd)
                except (redis.exceptions.ConnectionError):
                    logger.error("Redis exception")
                if self.redisWrite is not None:
                    if self.redisRead is None: self.redisRead=self.redisWrite
                    self.status = True
                    logger.info("Redis connected")
                    break
                else:
                    logger.error("Redis falied")
                if time.time()-begin>timeout:
                    logger.warning("Reids timeout")
                    break

   def getStatus(self):
        return self.status

   def getRedisWrite(self):
        return self.redisWrite

   def getRedisRead(self):
        return self.redisRead

class getEpicsServer():
    def __init__(
        self,
        retry,
        ):
        epics.ca.use_initial_context()

        pvCommand = "EXP_IB2_SQ:soft:ctrl_cmd"
        pvStatus = "EXP_IB2_SQ:soft:online_stat"

        self.status=False

        self.epicsCommand = None
        self.epicsStatus = None
        
        for i in range(int(retry)):
            if not self.epicsCommand:
                try:
                    self.epicsCommand = epics.PV(pvCommand)
                    self.epicsStatus = epics.PV(pvStatus)
                    self.status=True
                    logger.info("EPICS connected")
                    break
                except:
                    logger.warning("EPICS retry")
            time.sleep(0.2)

    def getStatus(self):
        return self.status

    def getServer(self):
        return self.epicsCommand, self.epicsStatus

class getDroneServer():
    def __init__(self, droneInfo):
        import neon

        self.droneInfo=droneInfo
        self.nModule=len(self.droneInfo)

        self.m_neonRedis = neon.Neon.NeonRedisSentinel([('10.1.34.116', 9001),('10.1.34.116', 9011)], i_socket_timeout=10, \
                                                           master_name='neonmaster', master_timeout = 120, master_password='sanlie;123', \
                                                           idb = 0, isWritable = True)
        
        #self.killAll()
        self.createAll()

    def create(self, n):
        clitpob= neon.Neon.NeonService.POBox(self.m_neonRedis, '/SANS/process/detector', 'client:'+self.droneInfo[n]['module'])
        clitrpc = neon.Neon.NeonService.NeonRPC(sendPOBox = clitpob, recvPOBox = clitpob)
        self.droneInfo[n]['pobox']=clitrpc
        self.droneInfo[n]['rpc']='server:'+self.droneInfo[n]['module']
        
        _pid=self.getPid(n)
        if _pid == -1:
            while True:
                _cmd="ssh drone@"+self.droneInfo[n]['host']+" cd /home/drone/workspace/drone/ && ./"+self.droneInfo[n]['module']
                _cmd = shlex.split(_cmd)
                subprocess.Popen(_cmd, shell=False)
                time.sleep(0.1)
                _pid=self.getPid(n)
                if _pid !=-1: break
        else:
            pass

        self.droneInfo[n]['running']='1'
        self.droneInfo[n]['pid']=_pid

    def createAll(self):
        for i in range(self.nModule): self.create(i)

    def kill(self, n):
        while True:
            _cmd="ssh drone@"+self.droneInfo[n]['host']+" ps -ef | grep "+self.droneInfo[n]['module']+".py | awk '{print $2}' | head -1"
            pid=subprocess.check_output(_cmd, shell=True)
            if pid!='':
                _cmd="ssh drone@"+self.droneInfo[n]['host']+" /usr/bin/kill -9 "+pid
                _cmd = shlex.split(_cmd)
                subprocess.check_output(_cmd, shell=False)
            else:
                logger.info("The module killed, %s" % self.droneInfo[n]['module'])
                break

    def killAll(self):
        for i in range(self.nModule): self.kill(i)

    def isExist(self, _droneInfo):
        exist=False
        _cmd="ssh drone@"+_droneInfo['host']+" ps -ef | grep "+_droneInfo['module']+".py | awk '{print $2}' | head -1"
        pid=subprocess.check_output(_cmd, shell=True)
        if pid!='':
            _cmd="ssh drone@"+_droneInfo['host']+" /usr/bin/kill -9 "+pid
            _cmd = shlex.split(_cmd)
            exist=True
            logger.info("The module exist, %s : %s" % (_droneInfo['module'], pid))
        else:
            exist=False
            logger.info("The module not exist, %s" % _droneInfo['module'])
        return exist

    def getPid(self, n):
        _cmd="ssh drone@"+self.droneInfo[n]['host']+" ps -ef | grep "+self.droneInfo[n]['module']+".py | awk '{print $2}' | head -1"
        pid=subprocess.check_output(_cmd, shell=True)
        if pid!='':
            pid=int(pid)
        else:
            pid = -1
        return pid

    def getPidAll(self):
        for i in range(self.nModule):
            print self.droneInfo[i]['module'], self.getPid(i)
        
    def start(self, n):
        #_droneInfo['pobox'].execRPC(_droneInfo['rpc'], 'Start', {'string':'start', 'times':3}, 'Print', -1.0)
        self.clear(n)

    def startAll(self):
        for i in range(self.nModule):
            self.start(i)

    def configure(self, n):
        pass

    def configureAll(self):
        pass

    def stop(self, n):
        self.clear(n)

    def stopAll(self):
        for i in range(self.nModule):
            self.stop(n)

    def abort(self, n):
        self.clear(n)

    def abortAll(self):
        for i in range(self.nModule):
            self.abort(i)

    def clear(self, n):
        try:
            self.droneInfo[n]['pobox'].execRPC(self.droneInfo[n]['rpc'], 'Clear', {'string':'clear', 'times':3}, 'Print', -1.0)
        except:
            self.kill(n)
            time.sleep(1.0)
            self.create(n)

    def clearAll(self):
        for i in range(self.nModule):
            self.clear(i)

    def exit(self):
        self.killAll()

class setHeartbeat(threading.Thread):

    def __init__(
        self,
        refreshtime,
        ):
        threading.Thread.__init__(self)
        pvHeart = "EXP_IB2_SQ:soft:analyse_hb"
        self.pvname=epics.PV(pvHeart)
        self.refreshtime=refreshtime

    def setStatus(self, _status):
        self.status=_status

    def getStatus(self):
        return self.status

    def process(self):
        self.pvname.put(getTime(), use_complete=True)

    def run(self):
        while True:
            try:
                self.process()
            except:
                pass
            time.sleep(self.refreshtime)

class setStatus(threading.Thread):

   def __init__(
        self,
        pvname,
        refreshtime,
        ):
        threading.Thread.__init__(self)

        self.status='WAITING'
        self.pvname=epics.PV(pvname)
        self.refreshtime=refreshtime

   def setStatus(self, _status):
        self.status=_status

   def getStatus(self):
        return self.status

   def process(self):
        self.pvname.put(self.getStatus(), use_complete=True)

   def run(self):
        while True:
            try:
                self.process()
            except:
                pass
            time.sleep(self.refreshtime)

class getEpicsCommand(threading.Thread):
    def __init__(self, redisServer, epicsStatus, droneServer):
        threading.Thread.__init__(self)
        epics.ca.use_initial_context()

        self.redisServer = redisServer
        self.epicsStatus = epicsStatus
        self.droneServer = droneServer

        self.pvCommand=epics.PV('EXP_IB2_SQ:soft:ctrl_cmd')
        self.redisPath="/SANS/control/command"
        
        self.SANSstatusList=['INITIALIZED', 'READY', 'RUNNING']
        self.SANScommandList=["CONF", "START", "STOP", "EXIT", "CANCEL"]
        self.MYstatusList=["waiting", "unconfigured", "configuring", "ready", "running", "paused", "error"]
        self.MYcommandList=["configure", "unconfigure", "start", "pause", "resume", "stop", "abort"]

        self.mantidHeartBeat='/SANS/heartbeat/mantid'
        self.pilotHeartBeat='/SANS/heartbeat/pilot'

        logger.debug("Initializing...")
        try:
            self.getEpicsConfigure()
        except:
            pass
        logger.debug("Initialized!!!")
       
        self.lastCmd=-1024

        self.setCommand(6, 1, 0, -1)

    def setCommand(self, cmd, stat, epics, retry):
        success=False

        cmd=int(cmd)
        stat=int(stat)
        epics=int(epics)
        retry=int(retry)
        if epics!=-1: _status=self.SANSstatusList[epics]
        self.redisServer.set("/SANS/control/command", json.dumps(cmd))

        _count=-1
        while True:
            if retry!=-1:
                _count+=1
                if _count > retry:
                    success=False
                    break

            mantidStatus = self.getMantidStatus()
            pilotStatus = self.getPilotStatus()
            if pilotStatus==stat and mantidStatus==stat:
                logger.debug("pilot & mantid, %s" % self.MYstatusList[pilotStatus])
                if epics!=-1:
                    self.epicsStatus.put(_status)
                    logger.info("Tell epics, %s" % self.SANSstatusList[epics])
                
                success=True
                break
            else:
                logger.debug("Waiting for, %s" % self.MYstatusList[stat])
                logger.debug("   pilot,  %s" % self.MYstatusList[pilotStatus])
                logger.debug("   mantid, %s" % self.MYstatusList[mantidStatus])
            time.sleep(0.5)

        return success

    def getDroneStatus(self):
        droneStatus=4
        _statusList=[]
        try:
            for i in range(SANSdroneInfo):
                _status = self.redisServer.get(self.droneHeartBeat)
                _statusList.append(json.load(_status)['status'])
            for i in range(SANSdroneInfo-1):
                if _statusList[i+1]!=_statusList[i]:
                    droneStatus=6
                    break
        except:
            droneStatus=6
        return droneStatus

    def getMantidStatus(self):
        try:
            mantidStatus = self.redisServer.get(self.mantidHeartBeat)
            mantidStatus=json.loads(mantidStatus)['status']
        except:
            mantidStatus=0

        return mantidStatus

    def getPilotStatus(self):
        try:
            pilotStatus = self.redisServer.get(self.pilotHeartBeat)
            pilotStatus=json.loads(pilotStatus)['status']
        except:
            pilotStatus=0

        return pilotStatus

    def getEpicsConfigure(self):
        try:
            run_no=int(epics.caget('EXP_IB2_SQ:soft:run_no'))
        except:
            run_no="unknown"

        try:
            user_id=epics.caget('EXP_IB2_SQ:soft:user_id')
        except:
            user_id="unknown"

        try:
            proposal_id=epics.caget('EXP_IB2_SQ:soft:proposal_id')
        except:
            proposal_id="unknown"

        _configure={}
        _configure['runNo']=run_no
        _configure['userID']=user_id
        _configure['proposalID']=proposal_id
        _configure=json.dumps(_configure, ensure_ascii=False)
        self.redisServer.set("/SANS/control/configure", _configure)
        logger.debug("Set configure, %d" % run_no)

    def run(self):
        while True:
            mantidStatus = self.getMantidStatus()
            pilotStatus = self.getPilotStatus()
            if pilotStatus==6:
                logger.error("pilot failed")
                self.epicsStatus.put("ERROR")
                continue
            elif mantidStatus==6:
                logger.error("mantid failed")
                self.epicsStatus.put("ERROR")
                continue
            else:
                pass

            try:
                _value=int(self.pvCommand.get())
            except:
                logger.warning("Epics Connected failed")
                continue

            if _value==self.lastCmd:
                continue
            elif _value<0 and _value>4:
                logger.info("Invalid command %d" % _value)
                continue
            else:
                self.lastCmd=_value
                logger.info("Receive command %s" % self.SANScommandList[_value])

            if self.lastCmd == 0:#CONF
                self.getEpicsConfigure()
                _success=self.setCommand(0, 3, 1, 100)
                if not _success:
                    self.setCommand(6, 1, -1, -1)
                    self.setCommand(0, 3, 1, -1)

            elif self.lastCmd == 1:#START
                #start
                try:
                    self.droneServer.clearAll()
                    logger.info("Drone cleared")
                except:
                    logger.error("Drone clear failed")
                
                time.sleep(5)
                
                _success = self.setCommand(2, 4, 2, 100)
                if not _success:
                    #abort
                    self.setCommand(6, 1, -1, -1)
                    #conf
                    self.getEpicsConfigure()
                    time.sleep(0.1)
                    self.setCommand(0, 3, -1, 100)
                    #start
                    self.setCommand(2, 4, 2, 100)

            elif self.lastCmd == 2:#STOP
                #stop
                _success = self.setCommand(5, 3, 0, 100)
                if not _success:
                    #abort
                    self.setCommand(6, 1, -1, -1)
                    #conf
                    self.getEpicsConfigure()
                    time.sleep(0.2)
                    self.setCommand(0, 3, 0, 100)
            elif self.lastCmd == 3 or self.lastCmd == 4:#EXIT or CANCEL
                #abort
                self.setCommand(6, 1, 0, -1)
            else:
                logger.info("Invalid command %d" % self.lastCmd)

            time.sleep(0.5)

if __name__=="__main__":
    try:
        if sys.argv[1] == 'stop':
            kill_all()
    except:
        run_this()


    logger.info("===================")
    logger.info("Welcome to CockPit!")
    logger.info("        SANS       ")
    logger.info("===================")
    # connect Redis
    try:
        #myredis=getRedisServer("10.1.33.141", 9000, 10)
        myredis=getRedisServer([('10.1.34.116', 9001), ('10.1.34.116',9011)], "sanlie;123", 10)
        if myredis.getStatus():
            redisServer=myredis.getRedisWrite()
            logger.info("Redis Connected")
    except:
        redisServer=None
        logger.error("Redis not available")
    #"==================="
    # connect Drone
    try:
        droneServer=getDroneServer(SANSdroneInfo)
        logger.info("Drone Created")
    except:
        logger.error("Drone Failed")

    #"==================="
    # connect Epics
    try:
        myepics=getEpicsServer(10)
        epicsCommand, epicsStatus = myepics.getServer()
        logger.info("EPICS Connected")
    except:
        epicsCommand=None
        epicsStatus=None
        logger.error("EPICS Failed")

    try:
        _cmd=int(epicsCommand.get())
        if _cmd < 0 or _cmd>5:
            redisServer.set("/SANS/control/command", json.dumps(-1))
    except:
        pass

    #"==================="
    try:
        threadGetEpics = getEpicsCommand(redisServer, epicsStatus, droneServer)
        threadGetEpics.setDaemon(True)
        threadGetEpics.start()
        logger.info("Command thread created")
    except:
        logger.error("Command thread Failed")
    time.sleep(0.1)
    # "==================="
    logger.info('===================')
    logger.info('    RMcockpit Go   ')
    logger.info('===================')
    try:
        threadGetEpics.join()
    except:

        pass
    # "==================="

    try:
        epicsStatus.put('ERROR')
        logger.info('===================')
        logger.info('   RMcockpit bye!  ')
        logger.info('===================')
    except:
        pass
    sys.exit()
