from io_util import logger
import data
import database
import icat_util
import transfer
import time
import sys
import threading

class main(threading.Thread):
    def __init__(self):
        threading.Thread.__init__(self)
        self.dbSession = self.getDBSession()
        self.icatSession = icat_util.getIcatSession()

    def closeIcatSession(self):
        try:
            self.icatSession.logout()
        except Exception as ex:
            logger.warning('Logout icat session failed: %s' % (str(ex)))

    def getDBSession(self):
        _r = False
        while True:
            try:
                _dbSession = database.getDBsession()
                _r = True
            except Exception as ex:
                logger.error(str(ex))
                time.sleep(5)

            if _r:
                break
        return _dbSession

    def updateDBSession(self):
        self.dbSession = self.getDBSession()

    def closeDBSession(self):
        try:
            self.dbSession.close()
        except Exception as ex:
            logger.warning('Logout database session failed: %s' % (str(ex)))

    def queryRecord(self, runNo):
        runString = data.getRunString(runNo)

        try:
            _tmp = self.dbSession.query(database.Record).filter_by(run=runString).first()
        except Exception as ex:
            logger.warning(str(ex))
            self.updateDBSession()
            _tmp = self.dbSession.query(database.Record).filter_by(run=runString).first()

        if _tmp is not None:
            _result = _tmp.stat
        else:
            _result = -1

        return _result

    def insertRecord(self, record):
        try:
            self.dbSession.add(record)
            self.dbSession.commit()
            return True
        except Exception as ex:
            logger.error(str(ex))
            self.dbSession.rollback()
            return False

    def updateRecord(self, runNo, stat):
        runString = data.getRunString(runNo)
        try:
            _tmp = self.dbSession.query(database.Record).filter_by(run=runString).first()
            _tmp.stat = stat
            #self.dbSession.add(_tmp)
            self.dbSession.commit()
            return True
        except Exception as ex:
            logger.error(str(ex))
            self.dbSession.rollback()
            return False

    def createRecord(self, runNo, conf):
        hasControl = data.hasControlSummaryFile(runNo)
        hasDetector = data.hasDetectorPath(runNo)
        hasMonitorHis = data.hasMonitorHisPath(runNo)
        hasMonitorEvt = data.hasMonitorEvtPath(runNo)
        hasNexus = data.hasNexusPath(runNo)

        runString = data.getRunString(runNo)

        _record = database.Record(beamline='BL02', run=runString, ctrl=hasControl, daq=hasDetector, monhis=hasMonitorHis, monevt=hasMonitorEvt, nexus=hasNexus, begin=conf['startDatetime'], end=conf['endDatetime'], stat=0)

        return _record

    def getRunMetaData(self, runNo):
        return data.parseControlSummaryFile(runNo)

    # 0: begin
    # 1: nexus transfer to Public
    # 2: write icat
    # 3: raw transfer to Public
    # 4: write icat
    # 5: nexus transfer to Backup
    # 6: raw transfer to Backup
    # 7: nexus transfer to Cloud
    def process(self):
        runNoList = data.getRunNoList(data.Nexuspath)

        for runNo in runNoList:
            if runNo < 39500:  # prompt the speed
            #if runNo < 12997:  # not checked
                continue
            try:
                stat = self.queryRecord(runNo)
            except Exception as ex:
                logger.error(str(ex))
                continue

            if stat == 6:
                continue
            else:
                logger.info("Deal with run %s: %s" % (runNo, stat))

                try:
                    conf = self.getRunMetaData(runNo)
                except Exception as ex:
                    logger.error(str(ex))
                    continue

                investigationName = conf['investigationName']
                visitName = investigationName
                if investigationName == 'BL02-obsolete':
                    visitName = 'commissioning-BL02-0'

                if not icat_util.isIcatSessionValid(self.icatSession):
                    logger.debug('Update ICAT session')
                    self.icatSession = icat_util.getIcatSession()

                try:
                    investigationId = icat_util.getInvestigationId(self.icatSession, investigationName, visit=visitName)
                    if investigationId is None:
                        _temp = 'P' + investigationName
                        logger.info('Proposal %s not found, try %s' % (investigationName, _temp))
                        investigationId = icat_util.getInvestigationId(self.icatSession, _temp)
                        if investigationId is not None:
                            logger.info('Proposal %s found.' % (_temp))
                            investigationName = _temp
                            conf['investigationName'] = investigationName
                        else:
                            logger.info('Proposal %s not found.' % (_temp))

                    if investigationId is None:
                        logger.error('Proposal %s for run %s not found, skip.' % (investigationName, runNo))
                        continue
                except Exception as ex:
                    logger.error('Run %s: %s' % (runNo, str(ex)))
                    continue

            record = self.createRecord(runNo, conf)
            try:
                facilityId = icat_util.getFacilityId(self.icatSession)
                if facilityId is None:
                    logger.error('No facility id found.')
                    continue
            except Exception as ex:
                logger.error('Get facility id failed: %s.' % (str(ex)))
                continue

            # new record
            if stat == -1:
                logger.info('Create transfer record for new run ...')
                _r = self.insertRecord(record)
                if _r:
                    stat = 0
                else:
                    logger.error('Create record for run %s failed' % (runNo))
                    continue

            # transfer nexus data
            _nexusFiles = data.getNexusFiles(runNo)
            if stat == 0:
                logger.info('Transfer NeXus data to public zone and cloud ...')
                try:
                    success1 = False
                    success2 = False
                    nexusPath = data.getNexusPath(runNo)
                    success2 = transfer.transferNexusToCloud(nexusPath, runNo, conf['investigationName'])
                    success1 = transfer.transferNexusToPublic(runNo, _nexusFiles)
                    if success1 and success2:
                        _r = self.updateRecord(runNo, 1)
                        if _r:
                            stat = 1
                        else:
                            continue
                    else:
                        logger.error('Nexus transfer to public for run %s failed' % (runNo))
                        continue
                except Exception as ex:
                    logger.error('Nexus transfer to public for run %s failed: %s' % (runNo, str(ex)))
                    continue

            # Catalog nexus data
            if stat == 1:
                logger.info('Catalog NeXus data ...')
                try:
                    _r = icat_util.catalogData(self.icatSession, conf, facilityId, investigationId, _nexusFiles)
                    if _r:
                        _t = self.updateRecord(runNo, 2)
                        if _t:
                            stat = 2
                        else:
                            continue
                    else:
                        logger.error('Nexus catalogue for run %s failed.' % (runNo))
                        continue
                except Exception as ex:
                    logger.error('Nexus catalogue for run %s failed: %s' % (runNo, str(ex)))
                    continue

            _controlFiles = data.getControlFiles(runNo)
            _detectorFiles = data.getDetectorFiles(runNo)
            _monitorhisFiles = data.getMonitorHisFiles(runNo)
            _monitorevtFiles = data.getMonitorEvtFiles(runNo)

            # write data xml
            try:
                logger.info('Write data XML ...')
                _r = data.writeXml(runNo, conf['startDatetime'], conf['endDatetime'], _controlFiles, _detectorFiles, _monitorhisFiles, _monitorevtFiles, _nexusFiles)
            except Exception as ex:
                logger.error('Write data XML for run %s failed: %s' % (runNo, str(ex)))

            _softwareFiles = data.getSoftwareFiles(runNo)

            _rawFiles = []
            for _item in [_controlFiles, _detectorFiles, _monitorhisFiles, _monitorevtFiles, _softwareFiles]:
                if len(_item) > 0:
                    _rawFiles = _rawFiles + _item

            # transfer raw data to public zone
            if stat == 2:
                logger.info('Transfer raw data to public zone ...')
                try:
                    _r = transfer.transferRawToPublic(runNo, _controlFiles, _detectorFiles, _monitorhisFiles, _monitorevtFiles)
                    if _r:
                        _t = self.updateRecord(runNo, 3)
                        if _t:
                            stat = 3
                        else:
                            continue
                    else:
                        logger.error('Raw transfer to pulic for run %s failed' % (runNo))
                        continue
                except Exception as ex:
                    logger.error('Transfer raw data to public zone for run %s failed: %s' % (runNo, str(ex)))
                    continue

            # Catalog raw data
            if stat == 3:
                logger.info('Catalog raw data ...')
                try:
                    _r = icat_util.catalogData(self.icatSession, conf, facilityId, investigationId, _rawFiles)
                    if _r:
                        _t = self.updateRecord(runNo, 4)
                        if _t:
                            stat = 4
                        else:
                            continue
                    else:
                        logger.error('Raw catalogue for run %s failed.' % (runNo))
                        continue
                except Exception as ex:
                    logger.error('Raw catalogue for run %s failed: %s' % (runNo, str(ex)))
                    continue

                try:
                    logger.info('Update investigation date ...')
                    #icat_util.setInvestigationDate(self.icatSession, conf, investigationId, **{"timeout": 10})
                except Exception as ex:
                    logger.error('Update investigation date for run %s failed: %s' % (runNo, str(ex)))
                    pass
                    
            # transfer nexus data to backup zone
            if stat == 4:
                logger.info('Transfer NeXus data to backup zone ...')
                try:
                    _success = False
                    _success = transfer.transferNexusToBackup(runNo, _nexusFiles)
                    if _success:
                        _r = self.updateRecord(runNo, 5)
                        if _r:
                            stat = 5
                        else:
                            continue
                    else:
                        logger.error('Nexus transfer to backup for run %s failed' % (runNo))
                        continue
                except Exception as ex:
                    logger.error('Nexus transfer to backup for run %s failed: %s' % (runNo, str(ex)))
                    continue

            # transfer raw data to backup zone
            if stat == 5:
                logger.info('Transfer raw data to backup zone ...')
                try:
                    _r = transfer.transferRawToBackup(runNo, _controlFiles, _detectorFiles, _monitorhisFiles, _monitorevtFiles, _softwareFiles)
                    if _r:
                        _t = self.updateRecord(runNo, 6)
                        if _t:
                            stat = 6
                            logger.info('Done.')
                        else:
                            continue
                    else:
                        logger.error('Transfer raw data to backup zone for run %s failed' % (runNo))
                        continue
                except Exception as ex:
                    logger.error('Transfer raw data to backup zone for run %s failed: %s' % (runNo, str(ex)))
                    continue

    def run(self):
        while True:
            self.process()
            self.closeDBSession()
            time.sleep(10.0)
            self.updateDBSession()

if __name__ == '__main__':
    threadDaemon = main()
    threadDaemon.setDaemon(True)
    threadDaemon.start()
    threadDaemon.join()
    threadDaemon.closeIcatSession()
    threadDaemon.closeDBSession()
    sys.exit()
