#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys
import os
import epics
from epics.ca import CAThread, withInitialContext
import redis
import datetime
import time
import random
import numpy as np
from scipy import exp
from scipy.optimize import leastsq
from scipy.optimize import curve_fit
import matplotlib.pyplot as plt
import matplotlib.animation as anim
from drawnow import drawnow
import warnings
import signal
import json
from Queue import Queue
from threading import Thread
import threading
import logging
import multiprocessing
import atexit
import jsonArray
import neon
import fcntl
import logging
import subprocess
import shlex

logger = logging.getLogger('RMcockpit')
hdlr = logging.FileHandler(os.getcwd() + '/RMcockpit.log')
formatter = logging.Formatter('%(asctime)s %(levelname)s %(message)s')
hdlr.setFormatter(formatter)
logger.addHandler(hdlr)
logger.setLevel(logging.DEBUG)

RMgroup = []
RMgroup.append('module01')
RMgroup.append('monitor1')
RMdroneInfo = []
for i in range(len(RMgroup)):
    RMdroneInfo.append({})
    RMdroneInfo[i]['host'] = '10.1.31.118'
    RMdroneInfo[i]['module'] = RMgroup[i]
    RMdroneInfo[i]['pid'] = -1
    RMdroneInfo[i]['running'] = '0'
    RMdroneInfo[i]['rpc'] = '0'
    RMdroneInfo[i]['pobox'] = '0'
    RMdroneInfo[i]['heartbeat'] = '/MR/heartbeat/detector/10.1.31.118:' \
        + RMgroup[i]

fh = 0


def run_once():
    global fh
    fh = open(os.path.realpath(__file__), 'r')
    try:
        fcntl.flock(fh, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except:
        os._exit(1)


def run_this():
    mypid = str(os.getpid())
    myname = os.path.basename(__file__)
    while True:
        _cmd = 'ps -ef | grep ' + myname \
            + " | awk '{print $2}' | head -1"
        pid = subprocess.check_output(_cmd, shell=True)
        if pid != '' and pid != mypid:
            _cmd = '/usr/bin/kill -9 ' + pid
            _cmd = shlex.split(_cmd)
            subprocess.check_output(_cmd, shell=False)
            logger.warning('This killed')
        else:
            break


def getTime():
    return time.strftime('%Y-%m-%d %H:%M:%S')


class getRedisServer:

    def __init__(
        self,
        ip,
        port,
        retry,
        ):

        self.status = False
        client = False
        self.server = None
        for i in range(int(retry)):
            if not self.server:
                self.server = redis.Redis(host=ip, port=port,
                        password='sanlie;123', db=0,
                        socket_connect_timeout=1.0)
            else:
                try:
                    client = self.server.client_list()
                except redis.exceptions.ConnectionError:
                    logger.warning('Redis failed')
                if client:
                    self.status = True
                    break
                else:
                    logger.warning('Redis retry')
            time.sleep(0.2)

    def getStatus(self):
        return self.status

    def getServer(self):
        return self.server


class getEpicsServer:

    def __init__(self, retry):

        epics.ca.use_initial_context()

        pvCommand = 'EXP_IB2_RM:soft:ctrl_cmd'
        pvStatus = 'EXP_IB2_RM:soft:analyse_stat'

        self.status = False

        self.epicsCommand = None
        self.epicsStatus = None

        for i in range(int(retry)):
            if not self.epicsCommand:
                try:
                    _epicsCommand = epics.PV(pvCommand)
                    _epicsStatus = epics.PV(pvStatus)
                    if _epicsCommand.get() and _epicsStatus.get():
                        self.epicsCommand = _epicsCommand
                        self.epicsStatus = _epicsStatus
                        self.status = True
                        logger.info('EPICS connected')
                        break
                except:
                    logger.warning('EPICS retry')
            time.sleep(0.2)

    def getStatus(self):
        return self.status

    def getServer(self):
        return (self.epicsCommand, self.epicsStatus)


class getDroneServer:

    def __init__(self, droneInfo):
        import neon

        self.droneInfo = droneInfo
        self.nModule = len(self.droneInfo)

        self.m_neonRedis = neon.Neon.NeonRedisSentinel(
            [('10.1.31.116', 9001), ('10.1.31.116', 9011)],
            i_socket_timeout=0.1,
            master_name='neonmaster',
            master_timeout=200,
            master_password='sanlie;123',
            idb=0,
            isWritable=True,
            )

        self.killAll()
        self.createAll()

        # self.clearAll()

    def create(self, n):
        clitpob = neon.Neon.NeonService.POBox(self.m_neonRedis,
                '/MR/process/detector', self.droneInfo[n]['host']
                + ':client:' + self.droneInfo[n]['module'])
        clitrpc = neon.Neon.NeonService.NeonRPC(sendPOBox=clitpob,
                recvPOBox=clitpob)
        self.droneInfo[n]['pobox'] = clitrpc
        self.droneInfo[n]['rpc'] = self.droneInfo[n]['host'] \
            + ':server:' + self.droneInfo[n]['module']
        _pid = self.getPid(n)
        if _pid == -1:
            while True:
                _cmd = 'ssh drone@' + self.droneInfo[n]['host'] \
                    + ' cd /home/drone/workspace/drone/cockpit/ && ./' \
                    + self.droneInfo[n]['module']
                _cmd = shlex.split(_cmd)
                subprocess.Popen(_cmd, shell=False)
                time.sleep(1.0)
                _pid = self.getPid(n)
                if _pid != -1:
                    self.droneInfo[n]['running'] = '1'
                    self.droneInfo[n]['pid'] = _pid
                    break
        else:
            self.droneInfo[n]['running'] = '1'
            self.droneInfo[n]['pid'] = _pid

    def createAll(self):
        for i in range(self.nModule):
            self.create(i)

    def kill(self, n):
        while True:
            _cmd = 'ssh drone@' + self.droneInfo[n]['host'] \
                + ' ps -ef | grep ' + self.droneInfo[n]['module'] \
                + ".py | awk '{print $2}' | head -1"
            pid = subprocess.check_output(_cmd, shell=True)
            if pid != '':
                _cmd = 'ssh drone@' + self.droneInfo[n]['host'] \
                    + ' /usr/bin/kill -9 ' + pid
                _cmd = shlex.split(_cmd)
                subprocess.check_output(_cmd, shell=False)
            else:
                logger.info('The module killed, %s'
                            % self.droneInfo[n]['module'])
                break

    def killAll(self):
        for i in range(self.nModule):
            self.kill(i)

    def isExist(self, _droneInfo):
        exist = False
        _cmd = 'ssh drone@' + _droneInfo['host'] + ' ps -ef | grep ' \
            + _droneInfo['module'] + ".py | awk '{print $2}' | head -1"
        pid = subprocess.check_output(_cmd, shell=True)
        if pid != '':
            _cmd = 'ssh drone@' + _droneInfo['host'] \
                + ' /usr/bin/kill -9 ' + pid
            _cmd = shlex.split(_cmd)
            exist = True
            logger.info('The module exist, %s : %s'
                        % (_droneInfo['module'], pid))
        else:
            exist = False
            logger.info('The module not exist, %s' % _droneInfo['module'
                        ])
        return exist

    def getPid(self, n):
        _cmd = 'ssh drone@' + self.droneInfo[n]['host'] \
            + ' ps -ef | grep ' + self.droneInfo[n]['module'] \
            + ".py | awk '{print $2}' | head -1"
        pid = subprocess.check_output(_cmd, shell=True)
        if pid != '':
            pid = int(pid)
        else:
            pid = -1
        return pid

    def start(self, n):

        # _droneInfo['pobox'].execRPC(_droneInfo['rpc'], 'Start', {'string':'start', 'times':3}, 'Print', -1.0)

        self.clear(n)

    def startAll(self):
        for i in range(self.nModule):
            self.start(i)

    def configure(self, n):
        pass

    def configureAll(self):
        pass

    def stop(self, n):
        self.clear(n)

    def stopAll(self):
        for i in range(self.nModule):
            self.stop(n)

    def abort(self, n):
        self.clear(n)

    def abortAll(self):
        for i in range(self.nModule):
            self.abort(i)

    def clear(self, n):
        try:
            self.droneInfo[n]['pobox'].execRPC(self.droneInfo[n]['rpc'
                    ], 'Clear', {'string': 'clear', 'times': 3}, 'Print'
                    , -1.0)
        except:
            self.kill(n)
            time.sleep(1.0)
            self.create(n)

        # relist =  self.droneInfo['rpc'].getRSLT()
        # for item in relist:
        #    leadtime, callbackfunc, result, error = item
        #    print result

    def clearAll(self):
        for i in range(self.nModule):
            self.clear(i)

    def exit(self):
        self.killAll()


class setHeartbeat(threading.Thread):

    def __init__(self, refreshtime):

        threading.Thread.__init__(self)

        pvHeart = 'EXP_IB2_RM:soft:analyse_hb'
        self.pvname = epics.PV(pvHeart)
        self.refreshtime = refreshtime

    def setStatus(self, _status):
        self.status = _status

    def getStatus(self):
        return self.status

    def run(self):
        while True:
            try:
                self.pvname.put(getTime(), use_complete=True)
            except:
                logger.warning('Heartbeat failed')
            time.sleep(self.refreshtime)


class setStatus(threading.Thread):

    def __init__(self, refreshtime):

        threading.Thread.__init__(self)

        self.status = 'WAITING'
        self.pvname = epics.PV('EXP_IB2_RM:soft:analyse_stat')
        self.refreshtime = refreshtime

    def setStatus(self, _status):
        self.status = _status

    def getStatus(self):
        return self.status

    def run(self):
        while True:
            try:
                self.pvname.put(self.getStatus(), use_complete=True)
            except:
                logger.warning('Status setting failed')
            time.sleep(self.refreshtime)


class getEpicsCommand(threading.Thread):

    def __init__(
        self,
        threadHeartbeat,
        redisServer,
        epicsCommand,
        epicsStatus,
        droneServer,
        ):
        threading.Thread.__init__(self)
        epics.ca.use_initial_context()
        self.redisServer = redisServer
        self.epicsCommand = epicsCommand
        self.epicsStatus = epicsStatus
        self.droneServer = droneServer
        self.threadHeartbeat = threadHeartbeat
        self.pvCommand = epics.PV('EXP_IB2_RM:soft:ctrl_cmd')
        self.pvStatus = epics.PV('EXP_IB2_RM:soft:analyse_stat')
        self.pvProton = epics.PV('EXP_IB2_RM:soft:prcharge')
        self.pvNeutron = epics.PV('EXP_IB2_RM:soft:neutron_counts')

        self.MRstatusList = [
            'WAITING',
            'INITIALIZED',
            'READY',
            'ANALYSING',
            'ANALYSED',
            'ENDED',
            'ERROR',
            ]
        self.MRcommandList = ['CONF', 'STARTANALYSE', 'NEXTPOINT',
                              'STOP', 'RESET']
        self.MYstatusList = [
            'waiting',
            'unconfigured',
            'configuring',
            'ready',
            'running',
            'paused',
            'error',
            ]
        self.MYcommandList = [
            'configure',
            'unconfigure',
            'start',
            'pause',
            'resume',
            'stop',
            'abort',
            ]

        self.mantidHeartBeat = '/MR/heartbeat/mantid'
        self.pilotHeartBeat = '/MR/heartbeat/pilot'

        logger.debug('Initializing...')
        try:
            self.getEpicsConfigure()
        except:
            pass
        logger.debug('Initialized!!!')
        self.setCommand(6, 1, 1, -1)

        self.lastCmd = '---'
        self.startTime = time.time()
        self.setStartTime(False)
        self.setEndTime(False)
        self.protoncharge = [[], []]
        self.detectorCounts = [[], []]

    def setProton(self):
        _time = time.time() - self.startTime
        _charge = float(epics.caget('EXP_IB2_RM:soft:prcharge'))
        self.protoncharge[0].append(_time)
        self.protoncharge[1].append(_charge)
        self.redisServer.set('/MR/drone/proton_charge',
                             json.dumps(self.protoncharge))

    def setProtonNeutron(self, redisData=True):
        try:
            _tmp = float(self.pvProton.get())
            _time = time.time() - self.startTime
            if len(self.protoncharge) > 1000:
                self.protoncharge[0][-1] = _time
                self.protoncharge[1][-1] = _tmp
            else:
                self.protoncharge[0].append(_time)
                self.protoncharge[1].append(_tmp)
            self.redisServer.set('/MR/drone/proton_charge',
                                 json.dumps(self.protoncharge))
        except:
            pass

        try:
            if redisData:
                _tmp = \
                    int(json.loads(self.redisServer.get('/MR/workspace/MantidData/neutroncounts'
                        )))
            else:
                _tmp = 0
            self.pvNeutron.put(_tmp)
            _time = time.time() - self.startTime
            if len(self.detectorCounts) > 1000:
                self.detectorCounts[0][-1] = _time
                self.detectorCounts[1][-1] = _tmp
            else:
                self.detectorCounts[0].append(_time)
                self.detectorCounts[1].append(_tmp)
            self.redisServer.set('/MR/drone/neutron_counts',
                                 json.dumps(self.detectorCounts))
        except:
            pass

    def getStartTime(self):
        try:
            _tmp = \
                json.loads(self.redisServer.get('/MR/drone/startTimeFloat'
                           ))
        except:
            _tmp = time.time()

        if _tmp == 0:
            _str = '---'
        else:
            _str = time.strftime('%Y-%m-%d %H:%M:%S',
                                 time.localtime(int(_tmp)))

        self.redisServer.set('/MR/drone/startTime', json.dumps(_str))

    def getEndTime(self):
        try:
            _tmp = \
                json.loads(self.redisServer.get('/MR/drone/endTimeFloat'
                           ))
        except:
            _tmp = time.time()

        if _tmp == 0:
            _str = '---'
        else:
            _str = time.strftime('%Y-%m-%d %H:%M:%S',
                                 time.localtime(int(_tmp)))

        self.redisServer.set('/MR/drone/endTime', json.dumps(_str))

    def setStartTime(self, now=True):
        if now:
            _tmp = time.time()
            self.startTime = _tmp
            _str = time.strftime('%Y-%m-%d %H:%M:%S',
                                 time.localtime(_tmp))
            self.redisServer.set('/MR/drone/startTime',
                                 json.dumps(_str))
            self.redisServer.set('/MR/drone/startTimeFloat',
                                 json.dumps(_tmp))
        else:
            self.redisServer.set('/MR/drone/startTime', json.dumps('---'
                                 ))
            self.redisServer.set('/MR/drone/startTimeFloat',
                                 json.dumps(0))

    def setEndTime(self, now=True):
        if now:
            _tmp = time.time()
            _str = time.strftime('%Y-%m-%d %H:%M:%S',
                                 time.localtime(_tmp))
            self.redisServer.set('/MR/drone/endTime', json.dumps(_str))
            self.redisServer.set('/MR/drone/endTimeFloat',
                                 json.dumps(_tmp))
        else:
            self.redisServer.set('/MR/drone/endTime', json.dumps('---'))
            self.redisServer.set('/MR/drone/endTimeFloat',
                                 json.dumps(0))

    def setCommand(
        self,
        cmd,
        stat,
        epics,
        retry,
        ):
        cmd = int(cmd)
        stat = int(stat)
        epics = int(epics)
        retry = int(retry)
        if epics != -1:
            _status = self.MRstatusList[epics]
        self.redisServer.set('/MR/control/command', json.dumps(cmd))

        _count = -1
        while True:
            if retry != -1:
                _count += 1
                if _count > retry:
                    success = False
                    break

            mantidStatus = self.getMantidStatus()
            pilotStatus = self.getPilotStatus()
            if pilotStatus == stat and mantidStatus == stat:
                logger.debug('pilot & mantid, %s'
                             % self.MYstatusList[pilotStatus])
                if epics != -1:
                    self.threadHeartbeat.setStatus(_status)
                    self.epicsStatus.put(_status)
                    logger.info('Tell epics, %s'
                                % self.MRstatusList[epics])

                success = True
                break
            else:
                logger.debug('Waiting for, %s'
                             % self.MYstatusList[stat])
                logger.debug('   pilot,  %s'
                             % self.MYstatusList[pilotStatus])
                logger.debug('   mantid, %s'
                             % self.MYstatusList[mantidStatus])
            time.sleep(0.5)

        return success

    def getDroneStatus(self):
        for i in range(len(RMgroup)):
            _tmp = self.redisServer.get(RMdroneInfo[i]['heartbeat'])
            _tmp = json.loads(_tmp)
            print _tmp

        # droneStatus = self.redisServer.get(self.droneHeartBeat)
        # droneStatus=json.load(droneStatus)['status']
        # droneStatus=json.load(droneStatus)['timestamp']
        # return droneStatus

    def getMantidStatus(self):
        try:
            mantidStatus = self.redisServer.get(self.mantidHeartBeat)
            mantidStatus = json.loads(mantidStatus)['status']
        except:
            mantidStatus = 0

        return mantidStatus

    def getPilotStatus(self):
        try:
            pilotStatus = self.redisServer.get(self.pilotHeartBeat)
            pilotStatus = json.loads(pilotStatus)['status']
        except:
            pilotStatus = 0

        return pilotStatus

    def getEpicsConfigure(self):

        # run_type=epics.caget('EXP_IB2_RM:soft:run_type')

        try:
            run_no = epics.caget('EXP_IB2_RM:soft:run_no')
        except:
            run_no = '---'

        # start_pos=epics.caget('EXP_IB2_RM:soft:start_pos')
        # end_pos=epics.caget('EXP_IB2_RM:soft:end_pos')
        # scan_point=epics.caget('EXP_IB2_RM:soft:scan_points')

        try:
            proton_charge = epics.caget('EXP_IB2_RM:soft:proton_charge')
        except:
            proton_charge = 0

        # end_time=epics.caget('EXP_IB2_RM:soft:end_time')

        try:
            userID = epics.caget('EXP_IB2_RM:soft:user_id')
        except:
            userID = '---'
        try:
            proposalID = epics.caget('EXP_IB2_RM:soft:proposal_id')
        except:
            proposalID = '---'
        try:
            detectorAngle = epics.caget('EXP_IB2_RM:soft:2theta')
        except:
            detectorAngle = 0

        # _points=[]
        # _step=(float(end_pos)-float(start_pos))/(float(scan_point)-1)
        # for i in range(int(scan_point)):
        # _points.append(float(start_pos)+i*_step)

        _configure = {}
        _configure['runNo'] = run_no
        _configure['userID'] = userID
        _configure['proposalID'] = proposalID
        _configure['detectorAngle'] = detectorAngle

        # _configure['runType']=run_type
        # _configure['positionPoint']=_points
        # _configure['protonCharge']=proton_charge
        # _configure['tof']=self.tofmantid.tolist()

        _configure = json.dumps(_configure, ensure_ascii=False)
        self.redisServer.set('/MR/control/configure', _configure)
        logger.info('Set configure, %d' % run_no)

    def run(self):
        while True:
            mantidStatus = self.getMantidStatus()
            pilotStatus = self.getPilotStatus()
            if pilotStatus == 6:
                logger.error('pilot failed')
                self.epicsStatus.put('ERROR')
                continue
            elif mantidStatus == 6:
                logger.error('mantid failed')
                self.epicsStatus.put('ERROR')
                continue
            else:
                pass

            try:
                _value = self.pvCommand.get()
                if _value == 'STARTANALYSE' or _value == 'START':
                    self.setProtonNeutron(True)

                if _value == self.lastCmd:
                    continue
                elif self.lastCmd == 'STARTANALYSE' and _value \
                    == 'START':
                    self.lastCmd = 'START'
                    continue
                else:
                    char_value = str(_value)
                    self.lastCmd = char_value
                    logger.info('Receive command %s' % char_value)
            except:
                continue

            # self.getDroneStatus()

            if char_value == 'CONF':
                self.getEpicsConfigure()
                _success = self.setCommand(0, 3, 2, 100)
                if not _success:
                    self.setCommand(6, 1, -1, -1)
                    self.setCommand(0, 3, 2, -1)
            elif char_value == 'STARTANALYSE' or char_value == 'START':

                self.getEpicsConfigure()

                # clear drone

                try:
                    self.droneServer.clearAll()
                    logger.info('Drone cleared')
                except:
                    logger.error('Drone clear failed')

                time.sleep(10)

                self.setStartTime(True)
                self.setEndTime(False)
                self.protoncharge = [[], []]
                self.detectorCounts = [[], []]

                self.setProtonNeutron(False)
                self.setCommand(2, 4, 3, -1)
            elif char_value == 'NEXTPOINT':

                self.setCommand(5, 3, 2, -1)
            elif char_value == 'RESET':

                self.setCommand(6, 1, 1, -1)
            elif char_value == 'FINISH':

                self.setCommand(5, 3, 4, -1)
                self.setEndTime(True)
            elif char_value == 'STOP':

                _success = self.setCommand(5, 3, -1, 100)
                _success = self.setCommand(1, 1, 5, 100)
                if not _success:
                    self.setCommand(6, 1, 5, -1)
            else:
                logger.warning('Invalid command %s' % char_value)

            time.sleep(0.5)


if __name__ == '__main__':

    # run_this()

    logger.info('===================')
    logger.info('Welcome to CockPit!')
    logger.info('         RM        ')
    logger.info('===================')

    # connect Redis

    try:
        myredis = getRedisServer('10.1.31.116', 9000, 10)
        if myredis.getStatus():
            redisServer = myredis.getServer()
            logger.debug('Redis Connected')
    except:
        redisServer = None
        logger.error('Redis Failed')

    # "==================="
    # connect Epics

    try:
        myepics = getEpicsServer(10)
        if myepics.getStatus():
            (epicsCommand, epicsStatus) = myepics.getServer()
            logger.debug('EPICS Connected')
    except:
        epicsCommand = None
        epicsStatus = None
        logger.error('EPICS Failed')

    # "==================="
    # connect Drone

    try:
        droneServer = getDroneServer(RMdroneInfo)
        logger.debug('Drone Created')
    except:
        logger.error('Drone Failed')

    # start heartbeat
    # "==================="

    try:
        threadHeartbeat = setHeartbeat(1.0)
        threadHeartbeat.setDaemon(True)
        threadHeartbeat.start()
        logger.debug('Heartbeat connected')
    except:
        logger.error('Heartbeat Failed')

    time.sleep(0.1)

    # set status
    # "==================="

    try:
        threadSetStatus = setStatus(1.0)
        threadSetStatus.setDaemon(True)
        threadSetStatus.start()
        logger.debug('State thread created')
    except:
        logger.error('State thread Failed')
    time.sleep(0.1)

    # receive epics command
    # "==================="

    try:
        threadGetEpics = getEpicsCommand(threadSetStatus, redisServer,
                epicsCommand, epicsStatus, droneServer)
        threadGetEpics.setDaemon(True)
        threadGetEpics.start()
        logger.debug('Command thread created')
    except:
        logger.error('Command thread Failed')
    time.sleep(0.1)

    # "==================="

    logger.info('RMcockpit started')
    try:
        threadGetEpics.join()
        threadSetStatus.join()
        threadHeartbeat.join()
    except:
        pass

    # "==================="

    try:
        threadSetStatus.setStatus('ERROR')
        epicsStatus.put('ERROR')
        logger.info('RMcockpit quit')
    except:
        pass
