#!/usr/bin/python
# -*- coding: utf-8 -*-

import sys
import os
import epics
import redis
import time
import numpy as np
import json
import threading
import atexit
import neon
import subprocess
import shlex
import random

import logging
from logging.handlers import RotatingFileHandler
logger = logging.getLogger('RMcockpit')

# hdlr = logging.FileHandler(os.getcwd()+'/RMcockpit.log', maxBytes=100, backupCount=5)

hdlr = RotatingFileHandler(os.getcwd() + '/RMcockpit.log', maxBytes=100
                           * 1024 * 1024, backupCount=1)
formatter = logging.Formatter('%(asctime)s %(levelname)s %(message)s')
hdlr.setFormatter(formatter)
logger.addHandler(hdlr)
logger.setLevel(logging.DEBUG)

RMdrone = ['module01', 'monitor1']
webInfo = {}
_group = ['pilot', 'mantid']
for g in RMdrone:
    _group.append(g)
webInfo['name'] = _group

_group = ['pilot']
for _g in _group:
    webInfo[_g] = {}
    webInfo[_g]['host'] = '10.1.31.184'
    webInfo[_g]['pid'] = '---'
    webInfo[_g]['heartbeat'] = '---'
    webInfo[_g]['pobox'] = '---'
    webInfo[_g]['rpc'] = '---'
    webInfo[_g]['state'] = 'waiting'
    webInfo[_g]['level'] = 'warning'

_group = ['mantid']
for _g in _group:
    webInfo[_g] = {}
    webInfo[_g]['host'] = '10.1.31.117'
    webInfo[_g]['pid'] = '---'
    webInfo[_g]['heartbeat'] = '---'
    webInfo[_g]['pobox'] = '---'
    webInfo[_g]['rpc'] = '---'
    webInfo[_g]['state'] = 'waiting'
    webInfo[_g]['level'] = 'warning'

droneInfo = []
for i in range(len(RMdrone)):
    droneInfo.append({})
    droneInfo[i]['host'] = '10.1.31.118'
    droneInfo[i]['module'] = RMdrone[i]
    droneInfo[i]['pid'] = '---'
    droneInfo[i]['pobox'] = '---'
    droneInfo[i]['rpc'] = '---'
    droneInfo[i]['state'] = 'waiting'
    droneInfo[i]['timestamp'] = '---'
    droneInfo[i]['heartbeat'] = '/MR/heartbeat/detector/' \
        + RMdrone[i]

for i in range(len(droneInfo)):
    _g = droneInfo[i]['module']
    webInfo[_g] = {}
    webInfo[_g]['host'] = droneInfo[i]['host']
    webInfo[_g]['pid'] = str(droneInfo[i]['pid'])
    webInfo[_g]['heartbeat'] = '---'
    webInfo[_g]['pobox'] = 'client:' + _g
    webInfo[_g]['rpc'] = 'server:' + _g
    webInfo[_g]['state'] = droneInfo[i]['state']
    webInfo[_g]['level'] = 'warning'

fh = 0


def run_once():
    import fcntl
    global fh
    fh = open(os.path.realpath(__file__), 'r')
    try:
        fcntl.flock(fh, fcntl.LOCK_EX | fcntl.LOCK_NB)
    except:
        os._exit(1)


def run_this():
    mypid = str(os.getpid())
    myname = os.path.basename(__file__)

    # _cmd="ps -ef | grep "+myname+" | grep -v grep | awk '{print $2}'"
    pid1=[]
    pid2=[]

    _s=False
    try:
        _cmd = 'ps -C ' + myname + ' -o pid='
        pid1 = subprocess.check_output(_cmd, shell=True)
        pid1 = shlex.split(pid)
        _s=True
    except:
        pid1=[]

    if not _s:
        try:
            _tmp="\"python "+myname+"\""
            _cmd = "ps -C " + _tmp + " -o pid="
            pid2 = subprocess.check_output(_cmd, shell=True)
            pid2 = shlex.split(pid)
        except:
            pid2=[]

    pid=[]
    for p in pid1:pid.append(p)
    for p in pid2:pid.append(p)

    for p in pid:
        if p != mypid:
            try:
                _cmd = '/usr/bin/kill -9 ' + str(p)

                # _cmd = shlex.split(_cmd)

                subprocess.check_call(_cmd, shell=True)
                logger.warning('Last process killed')
            except:
                logger.warning('Last process not killed')
        else:
            pass


def kill_all():
    mypid = str(os.getpid())
    myname = os.path.basename(__file__)

    _cmd = 'ps -ef | grep ' + myname \
        + " | grep -v grep | awk '{print $2}'"

    # _cmd="ps -C "+myname+" -o pid="

    pid = subprocess.check_output(_cmd, shell=True)
    pid = shlex.split(pid)
    for p in pid:
        if p != mypid:
            try:
                _cmd = '/usr/bin/kill -9 ' + str(p)

                # _cmd = shlex.split(_cmd)

                subprocess.check_call(_cmd, shell=True)
                logger.warning('Last process killed')
            except:
                logger.warning('Last process not killed')
        else:
            pass

    _cmd = '/usr/bin/kill -9 ' + str(mypid)
    subprocess.call(_cmd, shell=True)


def getTime():
    return time.strftime('%Y-%m-%d %H:%M:%S')


class startRedisServer:

    def __init__(self):
        pid = self.getPid('redis-server')
        if len(pid) != 0:
            pass
        else:
            _cmd = \
                '/opt/shared/library/redis-4.0.1/bin/redis-server /home/neon/workspace/redis/node0/redis_9000.conf'
            subprocess.call(_cmd, shell=True)

        pid = self.getPid('redis-sentinel')
        if len(pid) != 0:
            pass
        else:
            _cmd = \
                '/opt/shared/library/redis-4.0.1/bin/redis-sentinel /home/neon/workspace/redis/node0/sentinel_9000.conf'
            subprocess.call(_cmd, shell=True)

    def getPid(self, name):
        PID = []
        try:

            # PID=map(int,subprocess.check_output(["pidof", name]).split())

            PID = map(int, subprocess.check_output(['pgrep', '-f',
                      name]).split())
        except:
            pass

        return PID


class getRedisServer:

    def __init__(
        self,
        ip,
        port,
        retry,
        ):

        self.status = False
        client = False
        self.server = None
        for i in range(int(retry)):
            if not self.server:
                self.server = redis.Redis(host=ip, port=port,
                        password='sanlie;123', db=0,
                        socket_connect_timeout=1.0)
            else:
                try:
                    client = self.server.client_list()
                except redis.exceptions.ConnectionError:
                    logger.warning('Redis failed')
                if client:
                    self.status = True
                    break
                else:
                    logger.warning('Redis retry')
            time.sleep(0.2)

    def getStatus(self):
        return self.status

    def getServer(self):
        return self.server


class getEpicsServer:

    def __init__(self, retry):

        epics.ca.use_initial_context()

        self.status = False

        for i in range(int(retry)):
            try:

                # _tmp = epics.cainfo('EXP_IB2_RM:soft:ctrl_cmd', print_out=False)

                _tmp = epics.caget('EXP_IB2_RM:soft:ctrl_cmd')
                if _tmp is not None:
                    self.status = True
                    break
            except:
                pass
            time.sleep(0.1)

    def getStatus(self):
        return self.status


class getDroneServer:

    def __init__(self):
        import neon

        self.nModule = len(droneInfo)

        self.m_neonRedis = neon.Neon.NeonRedisSentinel(
            [('10.1.31.116', 9001), ('10.1.31.116', 9011)],
            i_socket_timeout=0.1,
            master_name='neonmaster',
            master_timeout=200,
            master_password='sanlie;123',
            idb=0,
            isWritable=True,
            )

        self.state=[]
        for i in range(self.nModule):self.state.append(False)

        self.createAll()

        # self.clearAll()

    def getState(self, n):
       return self.state[n]

    def getStateAll(self, n):
       return self.state

    def create(self, n):
        clitpob = neon.Neon.NeonService.POBox(self.m_neonRedis,
                '/SANS/process/detector', 'client:' + droneInfo[n]['module'])
        clitrpc = neon.Neon.NeonService.NeonRPC(sendPOBox=clitpob,
                recvPOBox=clitpob)
        droneInfo[n]['pobox'] = clitrpc
        droneInfo[n]['rpc'] = 'server:' + droneInfo[n]['module']
        _pid = self.getPid(n)
        if _pid == -1:
            for i in range(5):
                _cmd = 'ssh drone@' + droneInfo[n]['host'] \
                    + ' cd /home/drone/workspace/drone/ && ./' \
                    + droneInfo[n]['module']
                _cmd = shlex.split(_cmd)
                subprocess.Popen(_cmd, shell=False)
                time.sleep(1.0)
                _pid = self.getPid(n)
                if _pid != -1:
                    droneInfo[n]['pid'] = str(_pid)
                    self.state[n]=True
                    break
                else:
                    self.state[n]=False
                    logger.error('%s create failed' % (droneInfo[n]['module']))
        else:
            droneInfo[n]['pid'] = str(_pid)
            self.state[n]=True
        
        webInfo[droneInfo[n]['module']]['pid'] = str(droneInfo[n]['pid'
                ])

    def createAll(self):
        for i in range(self.nModule):
            self.create(i)

    def kill(self, n):
        while True:
            pid = self.getPid(n)
            if pid != -1:
                try:
                    _cmd = 'ssh drone@' + droneInfo[n]['host'] \
                        + ' /usr/bin/kill -9 ' + str(pid)

                    # _cmd = shlex.split(_cmd)

                    subprocess.check_call(_cmd, shell=True)
                except:
                    pass
            else:
                logger.debug('%s killed' % droneInfo[n]['module'])
                break

    def killAll(self):
        for i in range(self.nModule):
            self.kill(i)

    def getPid(self, n):
        _cmd = 'ssh drone@' + droneInfo[n]['host'] + ' ps -ef | grep ' \
            + droneInfo[n]['module'] \
            + ".py | grep -v grep | awk '{print $2}' | head -1"
        pid = subprocess.check_output(_cmd, shell=True)
        if pid != '':
            pid = int(pid)
        else:
            pid = -1
        return pid

    def start(self, n):
        self.clear(n)

    def startAll(self):
        for i in range(self.nModule):
            self.start(i)

    def configure(self, n):
        pass

    def configureAll(self):
        pass

    def stop(self, n):
        self.clear(n)

    def stopAll(self):
        for i in range(self.nModule):
            self.stop(n)

    def abort(self, n):
        self.clear(n)

    def abortAll(self):
        for i in range(self.nModule):
            self.abort(i)

    def clear(self, n):
        _cleared = False

        try:
            _timeout = 5.0
            droneInfo[n]['pobox'].execRPC(droneInfo[n]['rpc'], 'Clear',
                    {'string': droneInfo[n]['module']}, 'setDroneStatus'
                    , _timeout)

            # _rpc=RPCMethods()

            for i in range(int(_timeout / 0.5)):
                relist = droneInfo[n]['pobox'].getRSLT()

                for item in relist:

                    # result = droneInfo[n]['module']

                    (leadtime, callbackfunc, result, error) = item
                    if result == 'Ready':
                        _cleared = True

                    # params = ()
                    # _cleared = _rpc.execute(callbackfunc, params)

                    if _cleared:
                        logger.debug('%s, cleared'
                                % droneInfo[n]['module'])
                        break
                    webInfo[droneInfo[n]['module']]['state'] = \
                        'clearing'
                    webInfo[droneInfo[n]['module']]['level'] = 'warning'
                time.sleep(0.5)
        except:

            logger.warning('%s, not cleared' % droneInfo[n]['module'])

        if _cleared:
            webInfo[droneInfo[n]['module']]['state'] = 'cleared'
            webInfo[droneInfo[n]['module']]['level'] = 'normal'
        else:
            self.kill(n)
            time.sleep(1.0)
            self.create(n)
            webInfo[droneInfo[n]['module']]['state'] = 'restarted'
            webInfo[droneInfo[n]['module']]['level'] = 'warning'
            logger.debug('%s, restarted' % droneInfo[n]['module'])

    def clearAll(self):
        for i in range(self.nModule):
            self.clear(i)

    def exit(self):
        self.killAll()


class RPCMethods(neon.Neon.NeonService.NeonRPC.MethodCall):

    def setDroneStatus(self, params):
        return True


class setGlassfish(threading.Thread):

    def __init__(self, redisServer, refreshtime):

        threading.Thread.__init__(self)

        self.redisServer = redisServer
        self.refreshtime = refreshtime

        self.mantidHeartBeat = '/MR/heartbeat/mantid'
        self.pilotHeartBeat = '/MR/heartbeat/pilot'

        self.MYstatusList = [
            'waiting',
            'unconfigured',
            'configuring',
            'ready',
            'running',
            'paused',
            'error',
            ]

    def getDroneStatus(self, n):
        _name = droneInfo[n]['module']
        try:
            _tmp = \
                json.loads(self.redisServer.get(droneInfo[n]['heartbeat'
                           ]))
            _time = _tmp['timestamp']
            if _time != webInfo[_name]['heartbeat']:
                webInfo[_name]['heartbeat'] = _time
                webInfo[_name]['state'] = 'running'
                webInfo[_name]['level'] = 'normal'
            else:
                pass
        except:
            webInfo[_name]['state'] = 'error'
            webInfo[_name]['level'] = 'error'

    def getMantidStatus(self):
        try:
            mantid = self.redisServer.get(self.mantidHeartBeat)
            mantidStatus = json.loads(mantid)['status']
            mantidTime = json.loads(mantid)['timestamp']
            mantidPid = json.loads(mantid)['pid']
        except:
            mantidStatus = '0'
            mantidTime = '---'
            mantidPid = '0'

        return [mantidStatus, mantidTime, mantidPid]

    def getPilotStatus(self):
        try:
            pilot = self.redisServer.get(self.pilotHeartBeat)
            pilotStatus = json.loads(pilot)['status']
            pilotTime = json.loads(pilot)['timestamp']
            pilotPid = json.loads(pilot)['pid']
        except:
            pilotStatus = '0'
            pilotTime = '---'
            pilotPid = '0'

        return [pilotStatus, pilotTime, pilotPid]

    def run(self):
        while True:
            _r = self.getMantidStatus()
            webInfo['mantid']['heartbeat'] = _r[1]
            webInfo['mantid']['state'] = self.MYstatusList[int(_r[0])]
            webInfo['mantid']['pid'] = _r[2]
            _state = int(_r[0])
            if _state == 1 or _state == 3 or _state == 4:
                webInfo['mantid']['level'] = 'normal'
            elif _state == 0:
                webInfo['mantid']['level'] = 'warning'
            elif _state == 6:
                webInfo['mantid']['level'] = 'error'
            else:
                webInfo['mantid']['level'] = 'warning'

            _r = self.getPilotStatus()
            webInfo['pilot']['heartbeat'] = _r[1]
            webInfo['pilot']['state'] = self.MYstatusList[int(_r[0])]
            webInfo['pilot']['pid'] = _r[2]
            _state = int(_r[0])
            if _state == 1 or _state == 3 or _state == 4:
                webInfo['pilot']['level'] = 'normal'
            elif _state == 0:
                webInfo['pilot']['level'] = 'warning'
            elif _state == 6:
                webInfo['pilot']['level'] = 'error'
            else:
                webInfo['pilot']['level'] = 'warning'

            for i in range(len(droneInfo)):
                self.getDroneStatus(i)

            self.redisServer.set('/MR/cockpit/web', json.dumps(webInfo))
            time.sleep(self.refreshtime)


class setHeartbeat(threading.Thread):

    def __init__(self, refreshtime):

        threading.Thread.__init__(self)

        pvHeart = 'EXP_IB2_RM:soft:analyse_hb'
        self.pvname = epics.PV(pvHeart)
        self.refreshtime = refreshtime

    def setStatus(self, _status):
        self.status = _status

    def getStatus(self):
        return self.status

    def run(self):
        while True:
            try:
                self.pvname.put(getTime(), use_complete=True)
            except:
                logger.warning('Heartbeat failed')
            time.sleep(self.refreshtime)


class setStatus(threading.Thread):

    def __init__(self, refreshtime):

        threading.Thread.__init__(self)

        self.status = 'WAITING'
        self.pvname = epics.PV('EXP_IB2_RM:soft:analyse_stat')
        self.refreshtime = refreshtime

    def setStatus(self, _status):
        self.status = _status

    def getStatus(self):
        return self.status

    def run(self):
        while True:
            try:
                self.pvname.put(self.getStatus(), use_complete=True)
            except:
                logger.warning('Status setting failed')
            time.sleep(self.refreshtime)


class main(threading.Thread):

    def __init__(
        self,
        threadHeartbeat,
        threadSetStatus,
        redisServer,
        droneServer,
        ):
        threading.Thread.__init__(self)
        self.stop = threading.Event()

        epics.ca.use_initial_context()
        self.redisServer = redisServer
        self.droneServer = droneServer
        self.threadHeartbeat = threadHeartbeat
        self.threadSetStatus = threadSetStatus

        self.pvCommand = epics.PV('EXP_IB2_RM:soft:ctrl_cmd')
        self.pvStatus = epics.PV('EXP_IB2_RM:soft:analyse_stat')
        self.pvProton = epics.PV('EXP_IB2_RM:soft:prcharge')
        self.pvNeutron = epics.PV('EXP_IB2_RM:soft:neutron_counts')
        self.pvRunNo = epics.PV('EXP_IB2_RM:soft:run_no')
        self.pvUserId = epics.PV('EXP_IB2_RM:soft:user_id')
        self.pvProposalId = epics.PV('EXP_IB2_RM:soft:proposal_id')
        self.pvDetectorAngle = epics.PV('EXP_IB2_RM:soft:2theta')
        self.pvDetectorY = epics.PV('EXP_IB2_RM:soft:DetectorY')
        self.pvDetectorZ = epics.PV('EXP_IB2_RM:soft:DetectorZ')
        self.pvEndtype = epics.PV('EXP_IB2_RM:soft:Endtype')
        self.pvEndvalue = epics.PV('EXP_IB2_RM:soft:Endvalue')

        self.MRstatusList = [
            'WAITING',
            'INITIALIZED',
            'READY',
            'RUNNING',
            'ENDED',
            'ERROR',
            ]
        self.MRcommandList = ['CONF', 'STARTANALYSE', 'FINISH', 'STOP',
                              'RESET']
        self.MYstatusList = [
            'waiting',
            'unconfigured',
            'configuring',
            'ready',
            'running',
            'paused',
            'error',
            ]
        self.MYcommandList = [
            'configure',
            'unconfigure',
            'start',
            'pause',
            'resume',
            'stop',
            'abort',
            ]

        self.mantidHeartBeat = '/MR/heartbeat/mantid'
        self.pilotHeartBeat = '/MR/heartbeat/pilot'

        self.iRetry = 10
        self.lastTimeStamp = '---'
        self.lastCmd = '---'
        self.myState="unconfigured"

        self.setInitial()

    def setInitial(self):
        try:
            self.endType = self.pvEndtype.get()
        except:
            self.endType = 'time'

        self.resetProtonChargeList()
        self.resetMonitorCountList()
        self.resetDetectorCountList()

        try:
            _pvState = str(self.pvStatus.get())
        except:
            _pvState = 'INITIALIZED'

        if _pvState == 'WAITING' or _pvState == 'INITIALIZED' \
            or _pvState == 'ERROR':
            if int(self.getMantidStatus()[0]) != 1:
                _result = self.setCommand('mantid', 6, 1, self.iRetry)
                if not _result:
                    logger.error('mantid abort failed')
                    self.setEpicsState('ERROR')
                    self.stop.set()
            if int(self.getPilotStatus()[0]) != 1:
                _result = self.setCommand('pilot', 6, 1, self.iRetry)
                if not _result:
                    logger.error('pilot abort failed')
                    self.setEpicsState('ERROR')
                    self.stop.set()

            self.setEpicsState('INITIALIZED')
            self.myState="unconfigured"
        elif _pvState == 'READY':

            _r = self.getMantidStatus()
            _state = int(_r[0])
            if _state != 3:
                if _state == 1:
                    self.getEpicsConfigure()

                    _result = self.setCommand('mantid', 0, 3,
                            self.iRetry)
                    if not _result:
                        logger.error('mantid configure failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()
                else:

                    _result = self.setCommand('mantid', 6, 1,
                            self.iRetry)
                    if not _result:
                        logger.error('mantid abort failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

                    self.getEpicsConfigure()

                    _result = self.setCommand('mantid', 0, 3,
                            self.iRetry)
                    if not _result:
                        logger.error('mantid configure failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

            _r = self.getPilotStatus()
            _state = int(_r[0])
            if _state != 3:
                if _state == 1:
                    self.getEpicsConfigure()

                    _result = self.setCommand('pilot', 0, 3,
                            self.iRetry)
                    if not _result:
                        logger.error('pilot configure failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()
                else:

                    _result = self.setCommand('pilot', 6, 1,
                            self.iRetry)
                    if not _result:
                        logger.error('pilot abort failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

                    self.getEpicsConfigure()

                    _result = self.setCommand('pilot', 0, 3,
                            self.iRetry)
                    if not _result:
                        logger.error('pilot configure failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

            self.setEpicsState('READY')
            self.myState="ready"
        elif _pvState == 'RUNNING':
            _r = self.getMantidStatus()
            _state = int(_r[0])
            if _state != 4:
                if _state == 3:
                    _result = self.setCommand('mantid', 2, 4,
                            self.iRetry)
                    if not _result:
                        logger.error('mantid start failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()
                else:
                    _result = self.setCommand('mantid', 6, 1,
                            self.iRetry)
                    if not _result:
                        logger.error('mantid abort failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

                    self.getEpicsConfigure()

                    _result = self.setCommand('mantid', 0, 3,
                            self.iRetry)
                    if not _result:
                        logger.error('mantid configure failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

                    _result = self.setCommand('mantid', 2, 4,
                            self.iRetry)
                    if not _result:
                        logger.error('mantid start failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

            _r = self.getPilotStatus()
            _state = int(_r[0])
            if _state != 4:
                if _state == 3:
                    _result = self.setCommand('pilot', 2, 4,
                            self.iRetry)
                    if not _result:
                        logger.error('pilot start failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()
                else:
                    _result = self.setCommand('pilot', 6, 1,
                            self.iRetry)
                    if not _result:
                        logger.error('pilot abort failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

                    self.getEpicsConfigure()

                    _result = self.setCommand('pilot', 0, 3,
                            self.iRetry)
                    if not _result:
                        logger.error('pilot configure failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

                    _result = self.setCommand('pilot', 2, 4,
                            self.iRetry)
                    if not _result:
                        logger.error('pilot start failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

            self.getStartTime()
            self.getProtonChargeList()
            self.getDetectorCountList()
            self.setEpicsState('RUNNING')
            self.myState="running"
        elif _pvState == 'ENDED':

            _r = self.getMantidStatus()
            _state = int(_r[0])
            if _state != 3:
                if _state == 4:
                    _result = self.setCommand('mantid', 5, 3,
                            self.iRetry)
                    if not _result:
                        logger.error('mantid stop failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()
                else:
                    _result = self.setCommand('mantid', 6, 1,
                            self.iRetry)
                    if not _result:
                        logger.error('mantid abort failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

                    self.getEpicsConfigure()

                    _result = self.setCommand('mantid', 0, 3,
                            self.iRetry)
                    if not _result:
                        logger.error('mantid configure failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

            _r = self.getPilotStatus()
            _state = int(_r[0])
            if _state != 3:
                if _state == 4:
                    _result = self.setCommand('pilot', 5, 3,
                            self.iRetry)
                    if not _result:
                        logger.error('Pilot stop failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()
                else:
                    _result = self.setCommand('pilot', 6, 1,
                            self.iRetry)
                    if not _result:
                        logger.error('Pilot abort failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

                    self.getEpicsConfigure()

                    _result = self.setCommand('pilot', 0, 3,
                            self.iRetry)
                    if not _result:
                        logger.error('Pilot configure failed')
                        self.setEpicsState('ERROR')
                        self.stop.set()

            self.setEpicsState('ENDED')
            self.myState="ready"
        else:

            if int(self.getMantidStatus()[0]) != 1:
                _result = self.setCommand('mantid', 6, 1, self.iRetry)
            if int(self.getPilotStatus()[0]) != 1:
                _result = self.setCommand('pilot', 6, 1, self.iRetry)

            self.setEpicsState('INITIALIZED')
            self.myState="unconfigured"

    def getProtonChargeList(self):
        try:
            _tmp = self.redisServer.get('/MR/drone/proton_charge')
            _tmp = json.loads(_tmp)
            self.protonChargeList = _tmp
        except:
            self.protonChargeList = [[], []]

    def resetProtonChargeList(self):
        self.protonChargeList = [[], []]

    def getDetectorCountList(self):
        try:
            _tmp = self.redisServer.get('/MR/drone/neutron_counts')
            _tmp = json.loads(_tmp)
            self.detectorCountList = _tmp
        except:
            self.detectorCountList = [[], []]

    def resetDetectorCountList(self):
        self.detectorCountList = [[], []]

    def getMonitorCountList(self):
        try:
            _tmp = self.redisServer.get('/MR/drone/monitor_counts')
            _tmp = json.loads(_tmp)
            self.monitorCountList = _tmp
        except:
            self.monitorCountList = [[], []]

    def resetMonitorCountList(self):
        self.monitorCountList = [[], []]

    def setProtonNeutron(self):
        try:
            _tmp = float(self.pvProton.get())
        except:
            _tmp = 0.0

        _time = time.time() - self.startTime
        if len(self.protonChargeList) > 1000:
            self.protonChargeList[0][-1] = _time
            self.protonChargeList[1][-1] = _tmp
        else:
            self.protonChargeList[0].append(_time)
            self.protonChargeList[1].append(_tmp)

        self.redisServer.set('/MR/drone/proton_charge',
                             json.dumps(self.protonChargeList))

        try:
            _tmp = \
                int(json.loads(self.redisServer.get('/MR/workspace/MantidData/neutroncounts'
                    )))
        except:
            _tmp = 0

        _time = time.time() - self.startTime
        if len(self.detectorCountList) > 1000:
            self.detectorCountList[0][-1] = _time
            self.detectorCountList[1][-1] = _tmp
        else:
            self.detectorCountList[0].append(_time)
            self.detectorCountList[1].append(_tmp)

        if _tmp != 0:
            _rates = float(_tmp) / float(_time)
        else:
            _rates = 0.0
        self.redisServer.set('/MR/drone/detector_rates',
                             json.dumps(_rates))
        self.redisServer.set('/MR/drone/detector_counts',
                             json.dumps(self.detectorCountList))
        
        if _tmp !=0:
            try:
                self.pvNeutron.put(_tmp)
            except:
                pass

        try:
            _tmp = \
                int(json.loads(self.redisServer.get('/MR/workspace/MantidData/monitor/neutroncounts'
                    )))
        except:
            _tmp = 0

        _time = time.time() - self.startTime
        if len(self.monitorCountList) > 1000:
            self.monitorCountList[0][-1] = _time
            self.monitorCountList[1][-1] = _tmp
        else:
            self.monitorCountList[0].append(_time)
            self.monitorCountList[1].append(_tmp)

        if _tmp != 0:
            _rates = float(_tmp) / float(_time)
        else:
            _rates = 0.0
        self.redisServer.set('/MR/drone/monitor_rates',
                             json.dumps(_rates))
        self.redisServer.set('/MR/drone/monitor_counts',
                             json.dumps(self.monitorCountList))

        progressPath = '/MR/cockpit/progress'
        if self.endType == 'protonCharge':
            self.redisServer.set(progressPath,
                                 json.dumps(self.protonChargeList[-1]))
        elif self.endType == 'detectorCount':
            self.redisServer.set(progressPath,
                                 json.dumps(self.detectorCountList[-1]))
        elif self.endType == 'monitorCount':
            self.redisServer.set(progressPath,
                                 json.dumps(self.monitorCountList[-1]))
        elif self.endType == 'time':
            self.redisServer.set(progressPath, json.dumps(_time))
        else:
            self.redisServer.set(progressPath, json.dumps(_time))

    def getEndTime(self):
        try:
            _tmp = \
                json.loads(self.redisServer.get('/MR/drone/endTimeFloat'
                           ))
        except:
            _tmp = time.time()

        if _tmp == 0:
            _str = '---'
        else:
            _str = time.strftime('%Y-%m-%d %H:%M:%S',
                                 time.localtime(int(_tmp)))

        self.redisServer.set('/MR/drone/endTime', json.dumps(_str))

    def getStartTime(self):
        try:
            _tmp = \
                json.dumps(self.redisServer.get('/MR/drone/startTimeFloat'
                           ))
            _tmp = float(_tmp)
            _str = time.strftime('%Y-%m-%d %H:%M:%S',
                                 time.localtime(_tmp))
            self.startTime = _tmp
            self.redisServer.set('/MR/drone/startTime',
                                 json.dumps(_str))
        except:
            _tmp = time.time()
            self.startTime = _tmp
            _str = time.strftime('%Y-%m-%d %H:%M:%S',
                                 time.localtime(_tmp))
            self.redisServer.set('/MR/drone/startTime',
                                 json.dumps(_str))
            self.redisServer.set('/MR/drone/startTimeFloat',
                                 json.dumps(_tmp))

    def setStartTime(self, now=True):
        if now:
            _tmp = time.time()
            self.startTime = _tmp
            _str = time.strftime('%Y-%m-%d %H:%M:%S',
                                 time.localtime(_tmp))
            self.redisServer.set('/MR/drone/startTime',
                                 json.dumps(_str))
            self.redisServer.set('/MR/drone/startTimeFloat',
                                 json.dumps(_tmp))
        else:
            self.redisServer.set('/MR/drone/startTime', json.dumps('---'
                                 ))
            self.redisServer.set('/MR/drone/startTimeFloat',
                                 json.dumps(0))

    def setEndTime(self, now=True):
        if now:
            _tmp = time.time()
            _str = time.strftime('%Y-%m-%d %H:%M:%S',
                                 time.localtime(_tmp))
            self.redisServer.set('/MR/drone/endTime', json.dumps(_str))
            self.redisServer.set('/MR/drone/endTimeFloat',
                                 json.dumps(_tmp))
        else:
            self.redisServer.set('/MR/drone/endTime', json.dumps('---'))
            self.redisServer.set('/MR/drone/endTimeFloat',
                                 json.dumps(0))

    def setEpicsState(self, _status):
        self.threadHeartbeat.setStatus(_status)
        self.threadSetStatus.setStatus(_status)

    def setCommand(
        self,
        com,
        cmd,
        stat,
        retry,
        ):
        com = str(com)
        cmd = int(cmd)
        pvState = int(stat)
        retry = int(retry)

        if com == 'pilot':
            self.redisServer.set('/MR/control/command/pilot',
                                 json.dumps(cmd))
        elif com == 'mantid':
            self.redisServer.set('/MR/control/command/mantid',
                                 json.dumps(cmd))
        else:
            return False

        _count = -1
        while True:
            if retry != -1:
                _count += 1
                if _count > retry:
                    success = False
                    break

            if com == 'pilot':
                _r = self.getPilotStatus()
                _state = int(_r[0])
            elif com == 'mantid':
                _r = self.getMantidStatus()
                _state = int(_r[0])
            else:
                return False

            if _state == pvState:
                logger.debug('%s, %s' % (com,
                             self.MYstatusList[_state]))
                success = True
                break
            else:
                logger.debug('Waiting %s, from %s to %s' % (com,
                             self.MYstatusList[_state],
                             self.MYstatusList[pvState]))
            time.sleep(0.5)

        return success

    def getDroneStatus(self, n):
        try:
            _tmp = \
                json.loads(self.redisServer.get(droneInfo[n]['heartbeat'
                           ]))
            _time = _tmp['timestamp']
            if _time != droneInfo[n]['timestamp']:
                droneInfo[n]['state'] = 'running'
            else:
                pass
        except:
            droneInfo[n]['state'] = 'error'

    def getMantidStatus(self):
        try:
            mantid = self.redisServer.get(self.mantidHeartBeat)
            mantidStatus = json.loads(mantid)['status']
            mantidTime = json.loads(mantid)['timestamp']
            mantidPid = json.loads(mantid)['pid']
        except:
            mantidStatus = '0'
            mantidTime = '---'
            mantidPid = '0'

        return [mantidStatus, mantidTime, mantidPid]

    def getPilotStatus(self):
        try:
            pilot = self.redisServer.get(self.pilotHeartBeat)
            pilotStatus = json.loads(pilot)['status']
            pilotTime = json.loads(pilot)['timestamp']
            pilotPid = json.loads(pilot)['pid']
        except:
            pilotStatus = '0'
            pilotTime = '---'
            pilotPid = '0'

        return [pilotStatus, pilotTime, pilotPid]

    def getEpicsConfigure(self):

        try:
            run_no = self.pvRunNo.get()
        except:
            run_no = '---'
        try:
            userID = self.pvUserId.get()
        except:
            userID = '---'
        try:
            proposalID = self.pvProposalId.get()
        except:
            proposalID = '---'
        try:
            detectorAngle = self.pvDetectorAngle.get()
        except:
            detectorAngle = 0
        try:
            detectorY = self.pvDetectorY.get()
        except:
            detectorY = 0
        try:
            detectorZ = self.pvDetectorZ.get()
        except:
            detectorZ = 0
        try:
            self.endType = self.pvEndtype.get()
        except:
            self.endType = 'time'
        try:
            endValue = self.pvEndvalue.get()
        except:
            endValue = 0.0

        _configure = {}
        _configure['runNo'] = str(run_no)
        _configure['userID'] = str(userID)
        _configure['proposalID'] = str(proposalID)
        _configure['detectorAngle'] = str(detectorAngle)
        _configure['detectorY'] = str(detectorY)
        _configure['detectorZ'] = str(detectorZ)
        _configure['progressMin'] = '0'

        # _configure['progressMax']=str(endValue)

        _configure['progressMax'] = str(20.0)

        self.redisServer.set('/MR/control/configure',
                             json.dumps(_configure))
        logger.info('Set configure, %s' % run_no)

    def seekCommand(self, module, toState):
        if module=="mantid":
            _r = self.getMantidStatus()
        elif module=="pilot":
            _r = self.getPilotStatus()
        else:
            return

        try:
            fromState = self.MYstatusList[int(_r[0])]
        except:
            return

        if fromState == toState:
            pass
        elif toState == "error":
            pass
        else:
            if fromState == "unconfigured":
                if toState == "ready":
                    _success = self.setCommand(module, 0, 3,
                            self.iRetry)
                elif toState == "running":
                    _success = self.setCommand(module, 0, 3,
                            self.iRetry)
                    _success = self.setCommand(module, 2, 4,
                            self.iRetry)
                else:
                    pass

            elif fromState == "ready":
                if toState == "unconfigured":
                    _success = self.setCommand(module, 1, 3,
                            self.iRetry)
                elif toState == "running":
                    _success = self.setCommand(module, 2, 4,
                            self.iRetry)
                else:
                    pass
            elif fromState == "running":
                if toState == "unconfigured":
                    _success = self.setCommand(module, 5, 3,
                            self.iRetry)
                    _success = self.setCommand(module, 1, 1,
                            self.iRetry)
                elif toState == "ready":
                    _success = self.setCommand(module, 5, 3,
                            self.iRetry)
                else:
                    pass
            else:
                pass

    def run(self):
        while True:
            self.seekCommand("mantid", self.myState)
            self.seekCommand("pilot", self.myState)

            time.sleep(1.0)

            try:
                _value = str(self.pvCommand.get())
                _time = str(self.pvCommand.timestamp)
                if _value == 'STARTANALYSE' or _value == 'START':
                    try:
                        self.setProtonNeutron()
                    except:
                        pass
                
                if _time == self.lastTimeStamp and _value==self.lastCmd:
                    continue
                else:
                    newCmd = str(_value)
                    self.lastTimeStamp = _time
                    self.lastCmd = newCmd
                    logger.info('Receiving epics command %s' % newCmd)
            except:
                logger.debug('Receive epics command failed')
            
            if newCmd == 'CONF':
                self.getEpicsConfigure()

                self.seekCommand("mantid", "ready")
                self.seekCommand("pilot", "ready")

                if int(self.getMantidStatus()[0]) == 3 \
                    and int(self.getPilotStatus()[0]) == 3:
                    self.setEpicsState('READY')
                    self.myState="ready"
                else:
                    if int(self.getMantidStatus()[0]) != 3:
                        logger.error('mantid fatal')
                    elif int(self.getPilotStatus()[0]) != 3:
                        logger.error('pilot fatal')
                    else:
                        pass

                    self.setEpicsState('ERROR')
                    self.myState="error"
                    logger.error('%s failed' % (newCmd))

            elif newCmd == 'STARTANALYSE':
                # clear drone

                try:
                    self.droneServer.clearAll()
                except:
                    logger.error('Drone not cleared')

                self.resetProtonChargeList()
                self.resetDetectorCountList()
                self.resetMonitorCountList()

                self.seekCommand("mantid", "running")

                self.setStartTime(True)
                self.setEndTime(False)

                self.seekCommand("pilot", "running")

                if int(self.getMantidStatus()[0]) == 4 \
                    and int(self.getPilotStatus()[0]) == 4:
                    self.setProtonNeutron()
                    self.setEpicsState('RUNNING')
                    self.myState="running"
                else:
                    if int(self.getMantidStatus()[0]) != 4:
                        logger.error('mantid fatal')
                    elif int(self.getPilotStatus()[0]) != 4:
                        logger.error('pilot fatal')
                    else:
                        pass
                    self.pvStatus.put('ERROR')
                    self.myState="error"
                    logger.error('%s failed' % (newCmd))
            
            elif newCmd == 'FINISH':
                self.seekCommand("mantid", "ready")
                self.seekCommand("pilot", "ready")
           
                if int(self.getMantidStatus()[0]) == 3 \
                    and int(self.getPilotStatus()[0]) == 3:
                    self.setEndTime(True)
                    self.setEpicsState('READY')
                    self.myState="ready"
                else:
                    if int(self.getMantidStatus()[0]) != 3:
                        logger.error('mantid fatal')
                    elif int(self.getPilotStatus()[0]) != 3:
                        logger.error('pilot fatal')
                    else:
                        pass

                    self.threadSetStatus.setStatus('ERROR')
                    self.pvStatus.put('ERROR')
                    self.myState="error"
                    logger.error('%s failed' % (newCmd))
            
            elif newCmd == 'STOP':
                self.seekCommand("mantid", "ready")
                self.seekCommand("pilot", "ready")
        
                if int(self.getMantidStatus()[0]) == 3 \
                    and int(self.getPilotStatus()[0]) == 3:
                    self.setEpicsState('ENDED')
                    self.myState="ready"
                else:
                    if int(self.getMantidStatus()[0]) != 3:
                        logger.error('mantid fatal')
                    elif int(self.getPilotStatus()[0]) != 3:
                        logger.error('pilot fatal')
                    else:
                        pass

                    self.threadSetStatus.setStatus('ERROR')
                    self.pvStatus.put('ERROR')
                    self.myState="error"
                    logger.error('%s failed' % (newCmd))
            
            elif newCmd == 'RESET':

                _success = self.setCommand('mantid', 6, 1, self.iRetry)
                _success = self.setCommand('pilot', 6, 1, self.iRetry)

                if int(self.getMantidStatus()[0]) == 1 \
                    and int(self.getPilotStatus()[0]) == 1:
                    self.setEpicsState('INITIALIZED')
                    self.myState="unconfigured"
                else:
                    if int(self.getMantidStatus()[0]) != 1:
                        logger.error('mantid fatal')
                    elif int(self.getPilotStatus()[0]) != 1:
                        logger.error('pilot fatal')
                    else:
                        pass

                    self.threadSetStatus.setStatus('ERROR')
                    self.pvStatus.put('ERROR')
                    self.myState="error"
                    logger.error('%s failed' % (newCmd))
            else:

                logger.warning('Invalid command %s' % newCmd)

        self.stop.set()

if __name__ == '__main__':
    try:
        if sys.argv[1] == 'stop':
            kill_all()
    except:
        run_this()

    logger.info('===================')
    logger.info('Welcome to CockPit!')
    logger.info('         RM        ')
    logger.info('===================')

    # "==================="
    # start Redis

    try:
        _redis = startRedisServer()
        logger.debug('Redis Started')
    except:
        logger.error('Redis Start Failed')
        sys.exit()

    # "==================="
    # connect Redis

    try:
        myredis = getRedisServer('10.1.31.116', 9000, 10)
        if myredis.getStatus():
            redisServer = myredis.getServer()
            logger.debug('Redis Connected')
    except:
        redisServer = None
        logger.error('Redis Failed')
        sys.exit()

    # "==================="
    # connect Epics

    try:
        myepics = getEpicsServer(5)
        if myepics.getStatus():
            logger.debug('Epics Connected')
        else:
            logger.error('Epics Failed')
    except:
        logger.error('Epics Failed')

    # "==================="
    # connect Drone
    try:
        droneServer = getDroneServer()
        logger.debug('Drone Created')
    except:
        logger.error('Drone Failed')

    # start heartbeat
    # "==================="

    try:
        threadHeartbeat = setHeartbeat(1.0)
        threadHeartbeat.setDaemon(True)
        threadHeartbeat.start()
        logger.debug('Heartbeat connected')
    except:
        logger.error('Heartbeat Failed')

    time.sleep(0.1)

    # set status
    # "==================="

    try:
        threadSetStatus = setStatus(1.0)
        #threadSetStatus = setStatus(1.0)
        threadSetStatus.setDaemon(True)
        threadSetStatus.start()
        logger.debug('Statethread created')
    except:
        logger.error('Statethread Failed')
    time.sleep(0.1)

    # receive epics command
    # "==================="

    try:
        threadGetEpics = main(threadHeartbeat,
                threadSetStatus, redisServer, droneServer)
        threadGetEpics.setDaemon(True)
        threadGetEpics.start()
        logger.debug('Commandthread created')
    except:
        logger.error('Commandthread Failed')
    time.sleep(0.1)

    # set web state
    # "==================="

    try:
        threadGlassfish = setGlassfish(redisServer, 1.0)
        threadGlassfish.setDaemon(True)
        threadGlassfish.start()
        logger.debug('Omsthread created')
    except:
        logger.error('Omsthread Failed')
    time.sleep(0.1)

    # "==================="

    logger.info('===================')
    logger.info('     Cockpit Go    ')
    logger.info('===================')
    try:
        threadGetEpics.join()
    except:

        # threadSetStatus.join()
        # threadHeartbeat.join()

        pass

    # "==================="

    try:
        threadSetStatus.setStatus('ERROR')
        epicsStatus.put('ERROR')
        logger.info('===================')
        logger.info('   Cockpit bye!    ')
        logger.info('===================')
    except:
        pass
    sys.exit()
