/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package cn.ac.csns.microscope.ejb;

import cn.ac.csns.microscope.exception.TransferException;
import cn.ac.csns.microscope.exception.TransferException.TransferExceptionType;
import java.util.List;
import java.util.Map;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.NoResultException;
import javax.persistence.PersistenceException;
import javax.persistence.TransactionRequiredException;
import javax.persistence.TypedQuery;
import javax.persistence.criteria.CriteriaQuery;

/**
 * Abstract class for entity facades.
 *
 * @author tangm <a href="mailto:tangm@ihep.ac.cn">Tang Ming</a>
 */
public abstract class AbstractFacade<T> {

    private final Class<T> entityClass;

    public AbstractFacade(Class<T> entityClass) {
        this.entityClass = entityClass;
    }

    protected abstract EntityManager getEntityManager();

    /**
     * Synchronize the persistence context to the underlying database.
     *
     * @throws IpsException if the flush fails
     */
    public void flush() throws TransferException {
        try {
            getEntityManager().flush();
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (PersistenceException ex) {
            throw new TransferException(TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Refresh the state of the instance from the database, overwriting changes
     * made to the entity, if any.
     *
     * @param entity the entity
     * @throws IpsException if the refresh fails
     */
    public void refresh(T entity) throws TransferException {
        try {
            getEntityManager().refresh(entity);
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (EntityNotFoundException ex) {
            throw new TransferException(TransferExceptionType.NO_SUCH_OBJECT_FOUND,
                    ex.getLocalizedMessage());
        } catch (TransactionRequiredException ex) {
            throw new TransferException(TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Make an instance managed and persistent.
     * 
     * @param entity the entity
     * @throws TransferException if the create fails
     */
    public void create(T entity) throws TransferException {
        try {
            getEntityManager().persist(entity);
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (EntityExistsException ex) {
            throw new TransferException(TransferExceptionType.OBJECT_ALREADY_EXISTS,
                    ex.getLocalizedMessage());
        } catch (TransactionRequiredException ex) {
            throw new TransferException(TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Merge the state of the given entity into the current persistence context.
     * 
     * @param entity the entity
     * @throws TransferException if edit or create fails
     */
    public void edit(T entity) throws TransferException {
        try {
            getEntityManager().merge(entity);
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (TransactionRequiredException ex) {
            throw new TransferException(TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Remove the entity instance.
     * 
     * @param entity the entity
     * @throws TransferException if the remove fails
     */
    public void remove(T entity) throws TransferException {
        try {
            getEntityManager().remove(getEntityManager().merge(entity));
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (TransactionRequiredException ex) {
            throw new TransferException(TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Find by primary key. Search for an entity of the specified class and
     * primary key. If the entity instance is contained in the persistence
     * context, it is returned from there.
     * 
     * @param id the primary key
     * @return the found entity instance or null if the entity does not exist
     * @throws TransferException if id is not a valid primary key
     */
    public T find(Object id) throws TransferException {
        try {
            return getEntityManager().find(entityClass, id);
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Find all entity instances.
     * 
     * @return all entity instances
     * @throws TransferException if the find fails
     */
    public List<T> findAll() throws TransferException {
        try {
            CriteriaQuery cq = getEntityManager().getCriteriaBuilder().createQuery();
            cq.select(cq.from(entityClass));
            return getEntityManager().createQuery(cq).getResultList();
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (Exception ex) {
            throw new TransferException(TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Find entity instances in a specified range.
     * 
     * @param range the range
     * @return the entity instances between range[0] and range[1]
     * @throws TransferException if the find fails
     */
    public List<T> findRange(int[] range) throws TransferException {
        try {
            CriteriaQuery cq = getEntityManager().getCriteriaBuilder().createQuery();
            cq.select(cq.from(entityClass));
            javax.persistence.Query q = getEntityManager().createQuery(cq);
            q.setMaxResults(range[1] - range[0] + 1);
            q.setFirstResult(range[0]);
            return q.getResultList();
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (Exception ex) {
            throw new TransferException(TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Count the number of the entity instances.
     * 
     * @return the count
     * @throws TransferException if the count fails
     */
    public int count() throws TransferException {
        try {
            CriteriaQuery cq = getEntityManager().getCriteriaBuilder().createQuery();
            javax.persistence.criteria.Root<T> rt = cq.from(entityClass);
            cq.select(getEntityManager().getCriteriaBuilder().count(rt));
            javax.persistence.Query q = getEntityManager().createQuery(cq);
            return ((Long) q.getSingleResult()).intValue();
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (Exception ex) {
            throw new TransferException(TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Find a single result by a named query.
     *
     * @param queryName the name of the named query
     * @param parameters the parameters of the named query
     * @return the result
     * @throws TransferException if the find fails
     */
    protected T findSingleResultByNamedQuery(String queryName, Map<String, Object> parameters)
            throws TransferException {
        try {
            TypedQuery<T> query = getEntityManager().createNamedQuery(queryName, entityClass);
            parameters.entrySet().forEach(entry -> {
                query.setParameter(entry.getKey(), entry.getValue());
            });
            return query.getSingleResult();
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferException.TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (NoResultException ex) {
            throw new TransferException(TransferException.TransferExceptionType.NO_SUCH_OBJECT_FOUND,
                    ex.getLocalizedMessage());
        } catch (Exception ex) {
            throw new TransferException(TransferException.TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Find a list of results by a named query.
     *
     * @param queryName the name of the named query
     * @param parameters the parameters of the named query
     * @return the results
     * @throws TransferException if the find fails
     */
    protected List<T> findResultListByNamedQuery(String queryName, Map<String, Object> parameters)
            throws TransferException {
        try {
            TypedQuery<T> query = getEntityManager().createNamedQuery(queryName, entityClass);
            parameters.entrySet().forEach(entry -> {
                query.setParameter(entry.getKey(), entry.getValue());
            });
            return query.getResultList();
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferException.TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (Exception ex) {
            throw new TransferException(TransferException.TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Find a list of results by an JPQL query.
     *
     * @param jpqlQueryString the JPQL query string
     * @param parameters the parameters of the query
     * @return the results
     * @throws TransferException if the find fails
     */
    protected List<T> findResultListByQuery(String jpqlQueryString, Map<String, Object> parameters)
            throws TransferException {
        try {
            TypedQuery<T> query = getEntityManager().createQuery(jpqlQueryString, entityClass);
            parameters.entrySet().forEach(entry -> {
                query.setParameter(entry.getKey(), entry.getValue());
            });
            return query.getResultList();
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferException.TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (Exception ex) {
            throw new TransferException(TransferException.TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Find an single result by a JPQL query.
     *
     * @param jpqlQueryString the JPQL query string
     * @param parameters the parameters of the named query
     * @return the result
     * @throws TransferException if the find fails
     */
    protected T findSingleResultByQuery(String jpqlQueryString, Map<String, Object> parameters)
            throws TransferException {
        try {
            TypedQuery<T> query = getEntityManager().createQuery(jpqlQueryString, entityClass);
            parameters.entrySet().forEach(entry -> {
                query.setParameter(entry.getKey(), entry.getValue());
            });
            return query.getSingleResult();
        } catch (IllegalArgumentException ex) {
            throw new TransferException(TransferException.TransferExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (NoResultException ex) {
            throw new TransferException(TransferException.TransferExceptionType.NO_SUCH_OBJECT_FOUND,
                    ex.getLocalizedMessage());
        } catch (Exception ex) {
            throw new TransferException(TransferException.TransferExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

}
