/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package cn.ac.csns.ids.irods;

import java.io.FileNotFoundException;
import java.io.InputStream;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import org.irods.jargon.core.connection.AuthScheme;
import org.irods.jargon.core.connection.IRODSAccount;
import org.irods.jargon.core.exception.JargonException;
import org.irods.jargon.core.pub.IRODSAccessObjectFactory;
import org.irods.jargon.core.pub.IRODSFileSystem;
import org.irods.jargon.core.pub.IRODSFileSystemSingletonWrapper;
import org.irods.jargon.core.pub.domain.ObjStat;
import org.irods.jargon.core.pub.io.IRODSFile;
import org.irods.jargon.core.pub.io.IRODSFileInputStream;

/**
 *
 * @author tangm <a href="mailto:tangm@ihep.ac.cn">Tang Ming</a>
 */
public final class IrodsUtil {

    private static final Logger logger = Logger.getLogger(IrodsUtil.class.getName());   

    public static IRODSFileSystem getIRODSFileSystem() throws JargonException {
        return IRODSFileSystemSingletonWrapper.instance();
    }   

    public static AuthScheme getAuthScheme(String scheme) {
        if (null == scheme) {
            throw new IllegalArgumentException("Scheme must not null.");
        }
        switch (scheme.trim().toUpperCase()) {
            case "STANDARD":
                return AuthScheme.STANDARD;
            case "PAM":
                return AuthScheme.PAM;
            default:
                throw new IllegalArgumentException("Not supported scheme: " + scheme);
        }
    }

    public static IRODSAccount getIrodsAccount(final Properties properties) throws JargonException {
        if (null == properties) {
            Logger.getLogger(IrodsUtil.class.getName()).log(Level.SEVERE, "The properties is null");
            throw new IllegalArgumentException("The properties cannot be null");
        }
        try {
            String irodsHost = properties.getProperty(Constants.IRODS_HOST);
            checkNullOrEmptyValue(irodsHost, Constants.IRODS_HOST + " cannot be empty");

            String irodsPortStr = properties.getProperty(Constants.IRODS_PORT);
            checkNullOrEmptyValue(irodsPortStr, Constants.IRODS_PORT + " cannot be empty");
            int irodsPort = Integer.parseInt(irodsPortStr.trim());
            if (irodsPort <= 0) {
                throw new IllegalArgumentException(Constants.IRODS_PORT + " is less than or equals 0.");
            }

            String irodsZone = properties.getProperty(Constants.IRODS_ZONE);
            checkNullOrEmptyValue(irodsHost, Constants.IRODS_ZONE + " cannot be empty");

            String irodsResource = properties.getProperty(Constants.IRODS_RESOURCE);
            checkNullOrEmptyValue(irodsHost, Constants.IRODS_RESOURCE + " cannot be empty");

            String irodsAuthUser = properties.getProperty(Constants.IRODS_AUTH_USER);
            checkNullOrEmptyValue(irodsHost, Constants.IRODS_AUTH_USER + " cannot be empty");

            String irodsAuthPassword = properties.getProperty(Constants.IRODS_AUTH_PASSWORD);
            checkNullOrEmptyValue(irodsHost, Constants.IRODS_AUTH_PASSWORD + " cannot be empty");

            String irodsUserHome = properties.getProperty(Constants.IRODS_USER_HOME, "");

            String irodsAuthScheme = properties.getProperty(Constants.IRODS_AUTH_SCHEME, "STANDARD");
            AuthScheme authScheme = IrodsUtil.getAuthScheme(irodsAuthScheme);
            IRODSAccount irodsAccount = IRODSAccount.instance(irodsHost.trim(),
                    irodsPort,
                    irodsAuthUser.trim(),
                    irodsAuthPassword.trim(),
                    irodsUserHome.trim(),
                    irodsZone.trim(),
                    irodsResource.trim());
            irodsAccount.setAuthenticationScheme(authScheme);
            return irodsAccount;
        } catch (IllegalArgumentException ex) {
            Logger.getLogger(IrodsUtil.class.getName()).log(Level.SEVERE, ex.getMessage());
            throw new IllegalArgumentException(ex.getMessage());
        }
    }

    private static void checkNullValue(String value, String msg) {
        if (null == value) {
            throw new IllegalArgumentException(msg);
        }
    }

    private static void checkEmptyValue(String value, String msg) {
        if (value != null && value.isEmpty()) {
            throw new IllegalArgumentException(msg);
        }
    }

    private static void checkNullOrEmptyValue(String value, String msg) {
        checkNullValue(value, msg);
        checkEmptyValue(value.trim(), msg);
    }

    public static String translateToIrodsLocation(final Properties properties, String location) {
        if (null == location) {
            throw new IllegalArgumentException("Location cannot be null");
        }
        if (location.trim().isEmpty()) {
            throw new IllegalArgumentException("Location cannot be empty");
        }

        String irodsLocationPrefix = "";
        String irodsLocationSuffix = "";
        if (null != properties.getProperty(Constants.IRODS_LOCATION_PREFIX)) {
            irodsLocationPrefix = properties.getProperty(Constants.IRODS_LOCATION_PREFIX).trim();
        }
        if (null != properties.getProperty(Constants.IRODS_LOCATION_SUFFIX)) {
            irodsLocationSuffix = properties.getProperty(Constants.IRODS_LOCATION_SUFFIX);
        }
        return irodsLocationPrefix + location.trim() + irodsLocationSuffix;
    }

    public static InputStream getInputStreamForIrodsPath(IRODSAccount irodsAccount,
            String path) throws JargonException, FileNotFoundException {
        if (null == path || path.trim().isEmpty()) {
            throw new IllegalArgumentException("path is null or empty");
        }
        if (null == irodsAccount) {
            throw new IllegalArgumentException("irodsAccount is null");
        }

        logger.log(Level.FINE, "{0} request to get {1}",
                new Object[]{irodsAccount.toString(), path});
        IRODSAccessObjectFactory irodsAccessObjecFactory = getIRODSFileSystem()
                .getIRODSAccessObjectFactory();
        ObjStat objStat = irodsAccessObjecFactory
                .getCollectionAndDataObjectListAndSearchAO(irodsAccount)
                .retrieveObjectStatForPath(path);
        if (objStat.isSomeTypeOfCollection()) {
            throw new JargonException("It is not a file, it is a directory: " + path);
        }

        IRODSFile irodsFile = irodsAccessObjecFactory
                .getIRODSFileFactory(irodsAccount)
                .instanceIRODSFile(path);
        if (!irodsFile.exists()) {
            throw new FileNotFoundException("File not found: " + path);
        }
        if (!irodsFile.canRead()) {
            throw new FileNotFoundException("Cannot read file: " + path);
        }

        IRODSFileInputStream irodsFileInputStream = irodsAccessObjecFactory
                .getIRODSFileFactory(irodsAccount)
                .instanceIRODSFileInputStream(path);

        return irodsFileInputStream;
    }

}
