#!/usr/bin/python

import sys
import os
import epics
import redis
import NEON
import time
import numpy as np
import json
import threading
import subprocess
import shlex
import logging

logInfo=[]
class logHandler(logging.Handler):
    def __init__(self):
        super(logHandler, self).__init__()

    def emit(self, record):
        _dict={}
        _dict['time']=record.asctime
        _dict['level']=record.levelname
        _dict['message']=record.message
        if len(logInfo)>1000:
            logInfo.pop(0)
            logInfo.append(_dict)
        else:
            logInfo.append(_dict)

from logging.handlers import RotatingFileHandler
logger = logging.getLogger('GPPDcockpit')
hdlr = RotatingFileHandler(os.getcwd() + '/GPPDcockpit.log', maxBytes=1
                           * 1024, backupCount=0)
formatter = logging.Formatter('%(asctime)s %(levelname)s %(message)s')
hdlr.setFormatter(formatter)
logger.addHandler(hdlr)
webHandler=logHandler()
logger.addHandler(webHandler)
logger.setLevel(logging.DEBUG)

GPPDgroup1=['module521', 'module531', 'module532', 'module533', 'module541', 'module542', 'module543','module623', 'module631', 'module632', 'module633', 'module641', 'module642', 'module643']
GPPDgroup2=['module322', 'module331', 'module332', 'module333', 'module341', 'module342', 'module343', 'module422', 'module431', 'module432', 'module433', 'module441', 'module442','module443']
GPPDgroup3=['module123', 'module131', 'module132', 'module133', 'module141', 'module142', 'module143', 'module221', 'module231', 'module232', 'module233', 'module241', 'module242', 'module243']
GPPDgroup4=['monitor1', 'monitor2', 'monitor3']

pulseSource='/GPPD/earthworm/module123/pulse_counts'
monitorSource='/GPPD/heartbeat/detector/monitor01'

def run_this():
    mypid = str(os.getpid())
    myname = os.path.basename(__file__)

    # _cmd="ps -ef | grep "+myname+" | grep -v grep | awk '{print $2}'"

    pid1=[]
    pid2=[]

    _s=False
    try:
        _cmd = 'ps -C ' + myname + ' -o pid='
        pid1 = subprocess.check_output(_cmd, shell=True)
        pid1 = shlex.split(pid1)
        _s=True
    except:
        pid1=[]

    if not _s:
        try:
            _tmp="\"python "+myname+"\""
            _cmd = "ps -C " + _tmp + " -o pid="
            pid2 = subprocess.check_output(_cmd, shell=True)
            pid2 = shlex.split(pid2)
        except:
            pid2=[]
    
    pid=[]
    for p in pid1:pid.append(p)
    for p in pid2:pid.append(p)
  
    print "\n====================GPPD======================" 
    print "\n Hello ", mypid, " World ", pid
    print "\n==============================================" 
    for p in pid:
        if p != mypid:
            try:
                _cmd = '/usr/bin/kill -9 ' + str(p)

                # _cmd = shlex.split(_cmd)

                subprocess.check_call(_cmd, shell=True)
                logger.warning('GPPD Cockpit restarted')
            except:
                logger.warning('GPPD Cockpit not killed')
        else:
            pass


def kill_all():

    mypid = str(os.getpid())
    myname = os.path.basename(__file__)

    _cmd = 'ps -ef | grep ' + myname \
        + " | grep -v grep | awk '{print $2}'"

    # _cmd="ps -C "+myname+" -o pid="

    pid = subprocess.check_output(_cmd, shell=True)
    pid = shlex.split(pid)
    for p in pid:
        if p != mypid:
            try:
                _cmd = '/usr/bin/kill -9 ' + str(p)

                # _cmd = shlex.split(_cmd)

                subprocess.check_call(_cmd, shell=True)
                logger.warning('Last process killed')
            except:
                logger.warning('Last process not killed')
        else:
            pass

    _cmd = '/usr/bin/kill -9 ' + str(mypid)
    subprocess.call(_cmd, shell=True)


def getTime():
    return time.strftime('%Y-%m-%d %H:%M:%S')

class getRedisServer():

    def __init__(
        self,
        ip,
        passwd,
        timeout,
        ):

        pid = self.getPid('redis-server')
        if len(pid) != 0:
            pass
        else:
            _cmd = \
                '/opt/shared/library/redis-4.0.1/bin/redis-server /home/neon/workspace/redis/node0/redis_9000.conf'
            subprocess.call(_cmd, shell=True)

        pid = self.getPid('redis-sentinel')
        if len(pid) != 0:
            pass
        else:
            _cmd = \
                '/opt/shared/library/redis-4.0.1/bin/redis-sentinel /home/neon/workspace/redis/node0/sentinel_9000.conf'
            subprocess.call(_cmd, shell=True)

        from redis.sentinel import Sentinel

        self.status=False
        self.redisWrite=None
        self.redisRead=None
        begin=time.time()
        while True:
            if self.redisWrite is None:
                #self.server = redis.Redis(host=ip, port=port, password=passwd, db=0, socket_connect_timeout=1.0)
                sentinel = Sentinel(ip, socket_timeout=0.1)
                try:
                    self.redisWrite=sentinel.master_for('neonmaster', socket_timeout=30, password=passwd)
                    self.redisRead=sentinel.slave_for('neonmaster', socket_timeout=30, password=passwd)
                except (redis.exceptions.ConnectionError):
                    logger.error("Redis exception")
                if self.redisWrite is not None:
                    if self.redisRead is None: self.redisRead=self.redisWrite
                    self.status = True
                    break
                else:
                    logger.error("Redis falied")
                if time.time()-begin>timeout:
                    logger.warning("Reids timeout")
                    break

    def getStatus(self):
        return self.status

    def getRedisWrite(self):
        return self.redisWrite

    def getRedisRead(self):
        return self.redisRead

    def getPid(self, name):
        PID = []
        try:

            # PID=map(int,subprocess.check_output(["pidof", name]).split())

            PID = map(int, subprocess.check_output(['pgrep', '-f',
                      name]).split())
        except:
            pass

        return PID

class getEpicsServer:

    def __init__(self, retry):

        epics.ca.use_initial_context()

        self.status = False

        for i in range(int(retry)):
            try:
                _tmp = epics.caget('EXP_IB1_GP:soft:cmd')
                if _tmp is not None:
                    self.status = True
                    break
            except:
                pass
            time.sleep(0.1)

    def getStatus(self):
        return self.status

class setHeartbeat(threading.Thread):

    def __init__(
        self,
        refreshtime,
        ):
        threading.Thread.__init__(self)
        pvHeart = "EXP_IB1_GP:soft:analyse_hb"
        self.pvname=epics.PV(pvHeart)
        self.refreshtime=refreshtime


    def setStatus(self, _status):
        self.status=_status

    def getStatus(self):
        return self.status

    def process(self):
        self.pvname.put(getTime(), use_complete=True)

    def run(self):
        while True:
            try:
                self.process()
            except:
                pass
            time.sleep(self.refreshtime)

class setStatus(threading.Thread):

    def __init__(
        self,
        refreshtime,
        ):
        threading.Thread.__init__(self)

        self.status='WAITING'
        self.pvname = epics.PV('EXP_IB1_GP:soft:analyse_hb')
        self.refreshtime=refreshtime

    def setStatus(self, _status):
        self.status=_status

    def getStatus(self):
        return self.status

    def run(self):
        while True:
            try:
                self.pvname.put(self.getStatus(), use_complete=True)
            except:
                logger.warning('Status setting failed')
            time.sleep(self.refreshtime)

class main(threading.Thread):
    def __init__(self, redisServer):
        threading.Thread.__init__(self)
        self.stop = threading.Event()

        epics.ca.use_initial_context()
        self.redisServer = redisServer

        self.pvCommand=epics.PV('EXP_IB1_GP:soft:cmd')
        self.pvStatus = epics.PV('EXP_IB1_GP:soft:online_stat')

        self.pvRunNo = epics.PV('EXP_IB1_GP:soft:run_no')
        self.pvUserId = epics.PV('EXP_IB1_GP:soft:user_id')
        self.pvProposalId = epics.PV('EXP_IB1_GP:soft:proposal_id')
        self.pvM1Delay = epics.PV('EXP_IB1_GP:WR:T0:Ch1Val:ao')
        self.pvM2Delay = epics.PV('EXP_IB1_GP:WR:T0:Ch2Val:ao')
        self.pvM3Delay = epics.PV('EXP_IB1_GP:WR:T0:Ch3Val:ao')
        self.pvDetectorDelay = epics.PV('EXP_IB1_GP:WR:T0:Ch4Val:ao')
        self.pvEndtype = epics.PV('EXP_IB1_GP:soft:end_type')       
        self.pvNeutron = epics.PV('EXP_IB1_GP:soft:neutron_counts')

        self.pvModeSel = epics.PV('EXP_IB1_GP:CTRL:ExpEndChoise:stringin')

        self.pvCurTime = epics.PV('EXP_IB1_GP:soft:runtime')
        self.pvSetTime = epics.PV('EXP_IB1_GP:EXP:RuntimeSet:ai')
        
        self.pvCurProton = epics.PV('EXP_IB1_GP:soft:ProtonCountCalc')
        self.pvSetProton = epics.PV('EXP_IB1_GP:EXP:ProtonCalcSet:ai')

        #self.pvProgress = epics.PV('EXP_IB1_GP:soft:progress_bar')
        self.pvCurBeamTime = epics.PV('EXP_IB1_GP:soft:beamtime')
        self.pvSetBeamTime = epics.PV('EXP_IB1_GP:EXP:BeamtimeSet:ai')

        self.pvCurTemp = epics.PV('EXP_IB1_GP:SE:TEMP:RealtimeVal:ai')
        self.pvSetTemp = epics.PV('EXP_IB1_GP:SE:TEMP:Set:ao')
        self.pvTempStatus = epics.PV('EXP_IB1_GP:SE:TEMP:SEStatus:bi')

        self.GPPDstatusList=['INITIALIZED', 'READY', 'RUNNING']
        self.GPPDcommandList=["CONF", "START", "STOP", "EXIT", "CANCEL"]
        self.MYstatusList=["waiting", "unconfigured", "configuring", "ready", "running", "paused", "error"]
        self.MYcommandList=["configure", "unconfigure", "start", "pause", "resume", "stop", "abort"]

        self.mantidHeartBeat='/GPPD/heartbeat/mantid'
        self.pilotHeartBeat='/GPPD/heartbeat/pilot'

        self.iRetry = 30
        self.lastTimeStamp = '---'
        self.lastCmd = '---'
        self.endType="none"
        self.myState="unconfigured"
    
        self.tempStatus = -1
    
        self.setInitial()

    def setInitial(self):
        try:
            self.endType = self.pvEndtype.get()
        except:
            self.endType = 'none'

        self.resetRedis()
        self.resetStartTime()
        self.resetEndTime()
        self.resetProtonChargeList()
        self.resetMonitorCountList()
        self.resetDetectorCountList()
        self.resetPulseCountList()

        try:
            _pvState = str(self.pvStatus.get())
        except:
            _pvState = 'INITIALIZED'

        if _pvState == 'WAITING' or _pvState == 'INITIALIZED' \
            or _pvState == 'ERROR':
            self.seekCommand("mantid", "unconfigured")
            self.seekCommand("pilot", "unconfigured")
   
            self.verifyCommand(1, "INITIALIZED", "unconfigured")
            logger.info('Initializing: %s OK' % "INITIALIZED")

        elif _pvState == 'READY':
            self.seekCommand("mantid", 'ready')
            self.seekCommand("pilot", 'ready')

            self.verifyCommand(3, "READY", "ready")
            logger.info('Initializing: %s OK' % "READY")

        elif _pvState == 'RUNNING':
            self.seekCommand("mantid", "running")
            
            self.seekCommand("pilot", "running")

            self.verifyCommand(4, "RUNNING", "running")
            logger.info('Initializing: %s OK' % "RUNNING")
            
            try:
                _value = self.GPPDcommandList[int(self.pvCommand.get())]
                _time = str(self.pvCommand.timestamp)
                if _value == 'START':
                    self.lastTimeStamp=_time
                    self.lastCmd=_value
            except:
                pass
        else:

            if int(self.getOnlineStatus('mantid')[0]) != 1:
                _result = self.setCommand('mantid', 6, 1, self.iRetry)
            if int(self.getOnlineStatus('pilot')[0]) != 1:
                _result = self.setCommand('pilot', 6, 1, self.iRetry)

            self.setEpicsState('INITIALIZED')
            self.myState="unconfigured"
            logger.info('Initializing: %s OK' % "ABORT")

    def setEpicsState(self, _status):
        path = '/GPPD/cockpit/state'
        self.redisServer.set(path, json.dumps(_status))
        self.pvStatus.put(_status)

    def getProtonChargeList(self):
        try:
            _tmp = self.redisServer.get('/GPPD/drone/proton_charge')
            _tmp = json.loads(_tmp)
            self.protonChargeList = _tmp
        except:
            self.protonChargeList = [[], []]

    def resetProtonChargeList(self):
        self.protonChargeList = [[], []]

    def getDetectorCountList(self):
        try:
            _tmp = self.redisServer.get('/GPPD/drone/detector_counts')
            _tmp = json.loads(_tmp)
            self.detectorCountList = _tmp
        except:
            self.detectorCountList = [[], []]

    def resetDetectorCountList(self):
        self.detectorCountList = [[], []]

    def getMonitorCountList(self):
        try:
            _tmp = self.redisServer.get('/GPPD/drone/monitor_counts')
            _tmp = json.loads(_tmp)
            self.monitorCountList = _tmp
        except:
            self.monitorCountList = [[], []]

    def resetMonitorCountList(self):
        self.monitorCountList = [[], []]

    def getPulseCountList(self):
        try:
            _tmp = self.redisServer.get('/GPPD/drone/pulse_counts')
            _tmp = json.loads(_tmp)
            self.pulseCountList = _tmp
        except:
            self.pulseCountList = [[], []]
   
        try:
            _tmp = self.redisServer.get('/GPPD/drone/last_pulse')
            _tmp = json.loads(_tmp)
            self.lastPulse = _tmp
        except:
            self.lastPulse = 0
 
    def setLastPulse(self):
        self.redisServer.set('/GPPD/drone/last_pulse', json.dumps(self.lastPulse))

    def resetPulseCountList(self):
        self.pulseCountList = [[], []]

    def getPulseNumber(self):
        #print self.redisServer.get('/GPPD/heartbeat/detector/module521')
        #print self.redisServer.get('/GPPD/heartbeat/detector/module631')
            
        _tmp = \
            json.loads(self.redisServer.get(pulseSource))

        _tmp=int(_tmp)
        return _tmp

    def getMonitorNumber(self):

        _tmp = \
            json.loads(self.redisServer.get(monitorSource))
        _tmp=int(_tmp['event'])

        return _tmp

    def setProtonNeutron(self):
        #=======================================================
        try:
            _tmp = float(self.pvCurProton.get())
            _time = time.time()

            if len(self.protonChargeList[0])==0:
                self.protonChargeList[0].append(_time)
                self.protonChargeList[1].append(_tmp)
            elif len(self.protonChargeList[0])>0 and ((_time-self.protonChargeList[0][-1])>10):
                if len(self.protonChargeList) > 1000:
                    self.protonChargeList[0].pop(0)
                    self.protonChargeList[1].pop(0)

                self.protonChargeList[0].append(_time)
                self.protonChargeList[1].append(_tmp)
            else:
                pass


            self.redisServer.set('/GPPD/drone/proton_charge', json.dumps(self.protonChargeList))
        except:
            pass
            #logger.warning("Get Proton Charge failed.")


        #=======================================================
        try:
            _tmp=self.getPulseNumber()
            _time = time.time()

            if len(self.pulseCountList[0])==0:
                self.pulseCountList[0].append(_time)
                self.pulseCountList[1].append(_tmp)
            elif len(self.pulseCountList[0])>0 and ((_time-self.pulseCountList[0][-1])>10):
                if len(self.pulseCountList) > 1000:
                    self.pulseCountList[0].pop(0)
                    self.pulseCountList[1].pop(0)

                self.pulseCountList[0].append(_time)
                self.pulseCountList[1].append(_tmp)

            self.redisServer.set('/GPPD/drone/pulse_counts', json.dumps(self.pulseCountList))
        except:
            pass
            #logger.warning("Get Pulse Counts failed.")

        #=======================================================
        try:
            _tmp=self.redisServer.get('/GPPD/workspace/MantidData/neutroncounts')
            _tmp = json.loads(_tmp)
            _tmp = int(_tmp)

            _time = time.time()

            if len(self.detectorCountList[0])==0:
                self.detectorCountList[0].append(_time)
                self.detectorCountList[1].append(_tmp)
            elif len(self.detectorCountList[0])>0 and ((_time-self.detectorCountList[0][-1])>10):
                if len(self.detectorCountList) > 1000:
                    self.detectorCountList[0].pop(0)
                    self.detectorCountList[1].pop(0)

                self.detectorCountList[0].append(_time)
                self.detectorCountList[1].append(_tmp)
            else:
                pass


            _rates = float(_tmp) / float(_time-self.startTime)

            self.redisServer.set('/GPPD/drone/detector_rates',
                                json.dumps(format(_rates, '.2f')))
            self.redisServer.set('/GPPD/drone/detector_counts',
                                json.dumps(self.detectorCountList))

            #self.pvNeutron.put(_tmp)
        except:
            pass
        #    logger.warning("Get Detector Counts failed.")


        #=======================================================
        try:
            _tmp = int(json.loads(self.redisServer.get('/GPPD/workspace/MantidData/monitor1/neutroncounts')))
            #_tmp=self.getMonitorNumber()
            _time = time.time()

            if len(self.monitorCountList[0])==0:
                self.monitorCountList[0].append(_time)
                self.monitorCountList[1].append(_tmp)
            elif len(self.monitorCountList[0])>0 and ((_time-self.monitorCountList[0][-1])>10):
                if len(self.monitorCountList) > 1000:
                    self.monitorCountList[0].pop(0)
                    self.monitorCountList[1].pop(0)

                self.monitorCountList[0].append(_time)
                self.monitorCountList[1].append(_tmp)
            else:
                pass


            _rates = float(_tmp) / float(_time-self.startTime)
            self.redisServer.set('/GPPD/drone/monitor_rates',
                                json.dumps(format(_rates, '.2f')))
            self.redisServer.set('/GPPD/drone/monitor_counts',
                                json.dumps(self.monitorCountList))
        except:
            pass
        #    logger.warning("Get Monitor Counts failed.")

        #=======================================================

        progressPath = '/GPPD/cockpit/progress'
        if self.endType == 'protonCharge':
            self.redisServer.set(progressPath,
                                 json.dumps(self.protonChargeList[-1]))
        elif self.endType == 'detectorCount':
            self.redisServer.set(progressPath,
                                 json.dumps(self.detectorCountList[-1]))
        elif self.endType == 'monitorCount':
            self.redisServer.set(progressPath,
                                 json.dumps(self.monitorCountList[-1]))
        elif self.endType == 'time':
            self.redisServer.set(progressPath, json.dumps(_time))
        else:
            self.redisServer.set(progressPath, json.dumps(0))

    def resetRedis(self):
        # reset mantid workspace
        moduleList=['module123','module131', 'module132', 'module133','module141','module142','module143', 'module221','module231', 'module232', 'module233','module241','module242','module243','module322', 'module331', 'module332', 'module333', 'module341', 'module342', 'module343', 'module422', 'module431', 'module432', 'module433', 'module441', 'module442', 'module443','module521','module531', 'module532', 'module533', 'module541', 'module542', 'module543', 'module623', 'module631', 'module632', 'module633', 'module641', 'module642', 'module643']

        for module in moduleList:
            path= "/GPPD/workspace/detector/"+module+"/value"
            self.redisServer.delete(path)
            path = "/GPPD/workspace/MantidData/"+module+"/neutroncounts"
            self.redisServer.delete(path)

        # reset pilot data
        protonPath='/GPPD/drone/proton_charge'
        detectorPath='/GPPD/drone/detector_counts'
        detectorRatePath='/GPPD/drone/detector_rates'
        monitorPath='/GPPD/drone/monitor_counts'
        monitorRatePath='/GPPD/drone/monitor_rates'
        pulsePath='/GPPD/drone/pulse_counts'
        self.redisServer.set(protonPath, json.dumps([[],[]]))
        self.redisServer.set(detectorPath, json.dumps([],[]))
        self.redisServer.set(detectorRatePath, json.dumps(0))
        self.redisServer.set(monitorPath, json.dumps([],[]))
        self.redisServer.set(monitorRatePath, json.dumps(0))
        self.redisServer.set(pulsePath, json.dumps(0))

        path = '/GPPD/workspace/MantidData/neutroncounts'
        self.redisServer.delete(path)
    
        for i in range(3):
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/reduction/monitor/tof'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/reduction/monitor/counts'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/reduction/monitor/d'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/reduction/moniotr/intensity'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/reduction/PC/tof'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/reduction/PC/counts'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/reduction/PC/d'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/reduction/PC/intensity'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/raw/tof'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/raw/counts'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/raw/d'
            self.redisServer.delete(path)
            path = '/GPPD/workspace/MantidData/group'+str(i+1).zfill(2)+'/raw/intensity'
            self.redisServer.delete(path)

    def resetStartTime(self):
        self.redisServer.set('/GPPD/drone/startTime', json.dumps('---'))

    def setStartTime(self):
        _tmp = time.time()
        self.startTime = _tmp
        _str = time.strftime('%Y-%m-%d %H:%M:%S',time.localtime(_tmp))
        self.redisServer.set('/GPPD/drone/startTime',json.dumps(_str))

    def resetEndTime(self):
        self.redisServer.set('/GPPD/drone/endTime', json.dumps('---'))

    def setEndTime(self):
        _tmp = time.time()
        _str = time.strftime('%Y-%m-%d %H:%M:%S',time.localtime(_tmp))
        self.redisServer.set('/GPPD/drone/endTime', json.dumps(_str))

    def setDrone(self, cmd):
        cmd = int(cmd)
        self.redisServer.set('/GPPD/control/command/drone', json.dumps(cmd))
    
    def setEarthworm(self, cmd):
        cmd = int(cmd)
        self.redisServer.set('/GPPD/control/command/earthworm', json.dumps(cmd))
    
    def setCommand(
        self,
        com,
        cmd,
        stat,
        retry,
        ):
        com = str(com)
        cmd = int(cmd)
        pvState = int(stat)
        retry = int(retry)

        if com == 'pilot':
            self.redisServer.set('/GPPD/control/command/pilot',
                                 json.dumps(cmd))
        elif com == 'mantid':
            self.redisServer.set('/GPPD/control/command/mantid',
                                 json.dumps(cmd))
        else:
            return False

        _count = -1
        while True:
            if retry != -1:
                _count += 1
                if _count > retry:
                    success = False
                    break

            _r = self.getOnlineStatus(com)
            _state = int(_r[0])

            if _state == pvState:
                logger.debug('%s, %s' % (com,
                             self.MYstatusList[_state]))
                success = True
                break
            else:
                logger.debug('Waiting %s, from %s to %s' % (com,
                             self.MYstatusList[_state],
                             self.MYstatusList[pvState]))
            time.sleep(1.0)

        return success


    def getOnlineStatus(self,name):
        if name=="mantid": name="mantid/module521"
        if name=="drone": name="detector/module123"
        hbPath="/GPPD/heartbeat/"+name

        try:
            hb = self.redisServer.get(hbPath)
            if name=="drone":
                status = json.loads(hb)['state']
            else:
                status = json.loads(hb)['status']

            time = json.loads(hb)['timestamp']
            try:
                pid = int(json.loads(hb)['pid'])
            except:
                pid = '---'
        except:
            status = '0'
            time = '---'
            pid = '---'

        return [status, time, pid]
 
    def getEpicsConfigure(self):
        try:
            run_no = self.pvRunNo.get()
        except:
            run_no = '---'
        try:
            run_no=int(run_no)
        except:
            pass

        try:
            userID = self.pvUserId.get()
        except:
            userID = '---'
        try:
            proposalID = self.pvProposalId.get()
        except:
            proposalID = '---'

        try:
            m1Delay = self.pvM1Delay.get()
            if m1Delay is None:
                m1Delay = 0.0
        except:
            m1Delay = '---'
        try:
            m2Delay = self.pvM2Delay.get()
            if m2Delay is None:
                m2Delay = 0.0
        except:
            m2Delay = '---'
        try:
            m3Delay = self.pvM3Delay.get()
            if m3Delay is None:
                m3Delay = 0.0
        except:
            m3Delay = '---'
        try:
            detectorDelay = self.pvDetectorDelay.get()
            if detectorDelay is None:
                detectorDelay = 0.0
        except:
            detectorDelay = '---'

        self.endType="none"
        self.endValue=0.0

        _configure = {}
        _configure['runNo'] = str(run_no)
        _configure['userID'] = userID
        _configure['proposalID'] = proposalID
        
        _configure['m1Delay'] = m1Delay
        _configure['m2Delay'] = m2Delay
        _configure['m3Delay'] = m3Delay
        _configure['detectorDelay'] = detectorDelay

        _configure['endType'] = self.endType
        _configure['progressMin'] = 0.0
        _configure['progressMax'] = self.endValue

        _configure=json.dumps(_configure, ensure_ascii=False)
        self.redisServer.set("/GPPD/control/configure", _configure)
        logger.info("Set configure, %d" % run_no)

    

    def seekCommand(self, module, toState):

        try:
            fromState = self.MYstatusList[int(self.getOnlineStatus(module)[0])]
        except:
            return

        if fromState == toState:
            pass
        elif toState == "error":
            pass
        else:
            if fromState == "unconfigured":
                if toState == "ready":
                    _success = self.setCommand(module, 0, 3,
                            self.iRetry)
                elif toState == "running":
                    _success = self.setCommand(module, 0, 3,
                            self.iRetry)
                    _success = self.setCommand(module, 2, 4,
                            self.iRetry)
                else:
                    pass
            elif fromState == "ready":
                if toState == "unconfigured":
                    _success = self.setCommand(module, 1, 3,
                            self.iRetry)
                elif toState == "running":
                    _success = self.setCommand(module, 2, 4,
                            self.iRetry)
                else:
                    pass
            elif fromState == "running":
                if toState == "unconfigured":
                    _success = self.setCommand(module, 5, 3,
                            self.iRetry)
                    _success = self.setCommand(module, 1, 1,
                            self.iRetry)
                elif toState == "ready":
                    _success = self.setCommand(module, 5, 3,
                            self.iRetry)
                else:
                    pass
            else:
                pass

    def verifyCommand(self, toState, epicsState, myState):
        #if int(self.getOnlineStatus('mantid')[0]) == toState \
        #    and int(self.getOnlineStatus('pilot')[0]) == toState:
        #if int(self.getOnlineStatus('mantid')[0]) == toState:
        #if int(self.getOnlineStatus('mantid')[0]) == toState:
        self.setEpicsState(epicsState)
        self.myState=myState
        
        path = '/GPPD/cockpit/error'
        if int(self.getOnlineStatus('mantid')[0]) == toState \
            and int(self.getOnlineStatus('pilot')[0]) == toState:
            self.redisServer.set(path, json.dumps('done'))
        elif int(self.getOnlineStatus('mantid')[0]) != toState:
            logger.error('mantid fatal')
            self.redisServer.set(path, json.dumps('mantid fatal'))
        elif int(self.getOnlineStatus('pilot')[0]) != toState:
            logger.error('pilot fatal')
            self.redisServer.set(path, json.dumps('pilot fatal'))
        else:
            pass
        time.sleep(1.0)
            #self.setEpicsState('ERROR')

    def getProgress(self): 
        _modeSel = self.pvModeSel.get()
        if _modeSel == 'time':
            if float(self.pvSetTime.get()) != 0.0:
                _progress = float(self.pvCurTime.get())*100/float(self.pvSetTime.get())
            else:
                _progress = 0

        elif _modeSel == 'beam_time':
            if float(self.pvSetBeamTime.get()) != 0.0:
                _progress = float(self.pvCurBeamTime.get())*100/float(self.pvSetBeamTime.get())
            else:
                _progress = 0

        elif _modeSel == 'proton_charge':
            if float(self.pvSetProton.get()) != 0.0:
                _progress = float(self.pvCurProton.get())*100/float(self.pvSetProton.get())
            else:
                _progress = 0

        else:
            _progress = 0
        
        self.redisServer.set('/GPPD/drone/progress',json.dumps(_progress))

    def getTemp(self):
        _newStatus = self.pvTempStatus.timestamp
        if _newStatus != self.tempStatus:
            _temp = str(self.pvCurTemp.get()) + ' (' + str(self.pvSetTemp.get()) + ')'
            self.redisServer.set('/GPPD/drone/temperature',json.dumps(_temp))
            self.tempStatus = _newStatus

    def run(self):
        while True:
            try:
                _pvState = str(self.pvStatus.get())

                if _pvState == 'ERROR':
                    self.setDrone(5)
                    self.setEarthworm(5)
                    self.seekCommand("mantid", "unconfigured")
                    self.seekCommand("pilot", "unconfigured")
                    self.verifyCommand(1, "INITIALIZED", "unconfigured")
                    logger.info('Initializing: %s OK' % "INITIALIZED")
            except:
                pass

            # make sure mantid and pilot chase the state
            self.seekCommand("mantid", self.myState)
            self.seekCommand("pilot", self.myState)

            # get the epics command
            try:
                _value = self.GPPDcommandList[int(self.pvCommand.get())]
                _time = str(self.pvCommand.timestamp)
            except:
                #logger.debug('Receive epics command failed')
                continue

            # check if the command is updated by timestamp and name
            # keep running if the command is START
            if _time == self.lastTimeStamp and _value==self.lastCmd:
                if _value == 'START':
                    self.setProtonNeutron()
                    self.getProgress()
                    self.getTemp()

                continue

            else:
                newCmd = _value
                self.lastTimeStamp = _time
                self.lastCmd = _value
                logger.info('Receiving epics command %s' % newCmd)
          
            # respons the new command 
            if newCmd == 'CONF':
                self.getEpicsConfigure()

                self.seekCommand("mantid", "ready")
                self.seekCommand("pilot", "ready")

                self.verifyCommand(3, "READY", "ready")

                self.resetRedis()
                self.resetStartTime()
                self.resetEndTime()
                self.resetProtonChargeList()
                self.resetPulseCountList()
                self.resetDetectorCountList()
                self.resetMonitorCountList()

                self.tempStatus = self.pvTempStatus.timestamp

                logger.info('%s OK' % (newCmd))

            elif newCmd == 'START':
                self.lastPulse=self.getPulseNumber()
                self.setLastPulse()
                self.setDrone(2)
                self.setEarthworm(2)
                self.seekCommand("mantid", "running")

                self.setStartTime()

                self.seekCommand("pilot", "running")
                self.verifyCommand(4, "RUNNING", "running")
                logger.info('%s OK' % (newCmd))

            elif newCmd == 'STOP':
                self.setEndTime()
                time.sleep(1.0)
                
                self.setDrone(5)
                self.setEarthworm(5)
                self.seekCommand("mantid", "unconfigured")
                self.seekCommand("pilot", "unconfigured")
                self.verifyCommand(1, "INITIALIZED", "unconfigured")
               
                logger.info('%s OK' % (newCmd))  

            elif newCmd == 'EXIT' or newCmd == 'CANCEL':
                self.setEndTime()
                time.sleep(1.0)
                
                self.setDrone(5)
                self.setEarthworm(5)
                self.seekCommand("mantid", "unconfigured")
                self.seekCommand("pilot", "unconfigured")

                self.verifyCommand(1, "INITIALIZED", "unconfigured")

                logger.info('%s OK' % (newCmd))        
            
            else:
                logger.warning('Invalid command %s' % newCmd)             

            time.sleep(1.0)

        self.stop.set()

if __name__=="__main__":
    try:
        if sys.argv[1] == 'stop':
            kill_all()
    except:
        run_this()

    logger.info("===================")
    time.sleep(0.02)
    logger.info("Welcome to CockPit!")
    time.sleep(0.02)
    logger.info("===================")

    #"==================="
    # start Redis
    try:
        myredis=getRedisServer([('10.1.33.141', 9001)], "sanlie;123", 10)
        if myredis.getStatus():
            redisServer=myredis.getRedisWrite()
            logger.debug("Redis Connected")
    except:
        redisServer=None
        logger.error("Redis not available")
        sys.exit()
    #"==================="
    # connect Epics
    try:
        myepics = getEpicsServer(5)
        if myepics.getStatus():
            logger.debug('Epics Connected')
    except:
        logger.error("Epics Failed")
    
    #"==================="
    # connect Drone
    #try:
    #    droneServer=getDroneServer()
    #    logger.info("Drone Created")
    #except:
    #    logger.error("Drone Failed")

    #"==================="
    # receive epics command
    #try:
    if True:
        threadGetEpics = main(redisServer)
        threadGetEpics.setDaemon(True)
        threadGetEpics.start()
        logger.info("Mainthread created")
        time.sleep(0.1)
    #except:
    #    logger.error("Mainthread Failed")

    # "==================="
    logger.info('===================')
    time.sleep(0.02)
    logger.info('GPPD Cockpit Running')
    time.sleep(0.02)
    logger.info('===================')
    try:
        threadGetEpics.join()
    except:
        pass

    # "==================="

    try:
        logger.info('===================')
        time.sleep(0.02)
        logger.info(' GPPD Cockpit bye! ')
        time.sleep(0.02)
        logger.info('===================')
    except:
        pass
    sys.exit()
