#include "StateMachine.hh"
#include <stdexcept>      // std::invalid_argument
#include <algorithm>
#include <iostream>

StateMachine::StateMachine()
:m_status(StateMachine::Status::kUnconfigured)
,m_allowedStatusTransit(StateMachine::kNumStatus*StateMachine::kNumStatus, -1) //don't allow transit between 7 status
{

}

StateMachine::StateMachine(int instrument)
:m_status(StateMachine::Status::kUnconfigured)
,m_allowedStatusTransit(StateMachine::kNumStatus*StateMachine::kNumStatus, -1) //don't allow transit between 7 status
{
  if(instrument==1)
  {
    //in total 9 possible path of state transit
    grantStateTransit(Status::kUnconfigured, Status::kReady, Command::kConfigure);
    grantStateTransit(Status::kReady, Status::kUnconfigured, Command::kUnconfigure);

    grantStateTransit(Status::kReady, Status::kRunning, Command::kStart);
    grantStateTransit(Status::kRunning, Status::kReady, Command::kStop);

    grantStateTransit(Status::kRunning, Status::kPaused, Command::kPause);
    grantStateTransit(Status::kPaused, Status::kRunning, Command::kResume);

    grantStateTransit(Status::kReady, Status::kError, Command::kAbort);
    grantStateTransit(Status::kRunning, Status::kError, Command::kAbort);
    grantStateTransit(Status::kPaused, Status::kError, Command::kAbort);
  }
}

StateMachine::~StateMachine()
{

}


bool StateMachine::tableSanityCheck(Status from)
{
  std::vector<int> premitted;
  premitted.reserve(kNumStatus);
  for(unsigned i=0;i<kNumStatus;i++)
  {
    int possible_cmd = m_allowedStatusTransit[from*kNumStatus+i];
    if(possible_cmd>=0)
    {
      premitted.push_back(possible_cmd);
    }
  }

  std::sort(premitted.begin(), premitted.end());
  auto last = std::unique(premitted.begin(), premitted.end());
  if(last==premitted.end())
    return true;
  else
    return false;
}

std::string StateMachine::printStatus()
{
  return kStatusStrMap.find(m_status)->second;
}

std::string StateMachine::printStatus(Status s)
{
  if(s>=kNumStatus)
    throw std::invalid_argument("printStatus input must smaller than the number of status");

  return kStatusStrMap.find(s)->second;
}

StateMachine::Command StateMachine::int2cmd(int i) const
{
  if(i>=kNumStatus)
    throw std::invalid_argument("int2cmd input must smaller than the number of status");

  return static_cast<Command> (i);
}

StateMachine::Status StateMachine::int2status(int i) const
{
  if(i>=kNumStatus)
    throw std::invalid_argument("int2status input must smaller than the number of status");

  return static_cast<Status> (i);
}

void StateMachine::grantStateTransit(Status from, Status to, Command byCommand)
{
  m_allowedStatusTransit[from*kNumStatus + to] = byCommand;
  if(!tableSanityCheck(from))
  {
    std::cerr << "Granting transit from status " << from
          << " to status " << to << " by command " << byCommand << '\n';
    throw std::invalid_argument("The same command sends a status to different status");
  }
}

void StateMachine::denyStateTransit(Status from, Status to)
{
  m_allowedStatusTransit[from*kNumStatus + to] = -1;
}

StateMachine::Status StateMachine::transit(Status from, Command byCommand)
{
  StateMachine::Status s = from;
  for(unsigned i=0;i<kNumStatus;i++)
  {
    if( m_allowedStatusTransit[from*kNumStatus + i] == byCommand)
    {
      // i is gerentted to be smaller than the number of status
      s = static_cast<Status> (i);
    }
  }
  return s;
}

void StateMachine::transit(Command by)
{
  for(unsigned i=0;i<kNumStatus;i++)
  {
    if( m_allowedStatusTransit[m_status*kNumStatus + i] == by)
    {
      //in order to be valid,
      //element of the enum Status must start from 0 and incremental by 1
      // i is gerentted to be smaller than the number of status

      m_status = static_cast<Status> (i);
      break;
    }
  }
}


#include <random>
void StateMachine::selfDiagnosis()
{
  std::random_device rd;
  std::mt19937 gen(rd());
  std::uniform_int_distribution<> dis(0, 6);

  std::vector<std::string> history;

  std::cout << "Self-diagnosis:: generating 10k random status and commands \n";
  for (int n=0; n<10000; ++n)
  {
    int r1(dis(gen)), r2(dis(gen));
    Status oldstatus = int2status(r1);
    Command randcmd = int2cmd(r2);

    Status newstatus = transit(oldstatus, randcmd);

    if(oldstatus!=newstatus)
    {
      std::string record = kStatusStrMap.find(oldstatus)->second + "("
               + kCommandStrMap.find(randcmd)->second + ")"
               + kStatusStrMap.find(newstatus)->second ;

      if (std::find(history.begin(), history.end(), record) == history.end())
      {
        // Element not in vector.
        history.push_back(record);
      }
    }
  }

  std::cout << "Found " << history.size() << " valid transit" << '\n';
  for (auto const & i : history)
    std::cout << i << " \n";
  std::cout << "\n";
}
