#ifndef StateMachine_hh
#define StateMachine_hh

#include <string>
#include <vector>
#include <map>

//This class models a state machine of m_numStatus status.
//Transit is initiated by received commands.

//Internally,this class contains a m_numStatus by m_numStatus size matrix-like
//data, which records the premitted transit from state A to B triggered by a
//command X

class StateMachine {
public:
  const unsigned kNumStatus = 7;
  //Notice both Status and Command must start from zero
  //and each variable increases by 1. Limited by cmd2int and int2cmd
	enum Status {
    kWaiting,
    kUnconfigured,
    kConfiguring,
    kReady,
    kRunning,
    kPaused,
    kError
  };

  const std::map<Status, std::string> kStatusStrMap = {
  {Status::kWaiting, "waiting"},
  {Status::kUnconfigured, "unconfigured"},
  {Status::kConfiguring, "configuring"},
  {Status::kReady, "ready"},
  {Status::kRunning, "running"},
  {Status::kPaused, "paused"},
  {Status::kError, "error"}
  };


  enum Command {
    kConfigure, //0
    kUnconfigure,
    kStart, //2
    kPause,
    kResume,//4
    kStop,
    kAbort
  };

  const std::map<Command, std::string> kCommandStrMap = {
  {Command::kConfigure, "configure"},
  {Command::kUnconfigure, "unconfigure"},
  {Command::kStart, "start"},
  {Command::kPause, "pause"},
  {Command::kResume, "resume"},
  {Command::kStop, "stop"},
  {Command::kAbort, "abort"}
  };



public:
  StateMachine();
  //instrument 1 is MR, others to be implemented
  StateMachine(int instrument);
  virtual ~StateMachine();

  void grantStateTransit(Status from, Status to, Command by);
  void denyStateTransit(Status from, Status to);

  void transit(Command byCommand);

  std::string printStatus();
  std::string printStatus(Status s);

  void selfDiagnosis(); //slow, use with care

protected:
  Status m_status;
  Command int2cmd(int i) const;
  Status int2status(int i) const;

private:
  bool tableSanityCheck(Status from);
  Status transit(Status from, Command byCommand);
  std::vector<int> m_allowedStatusTransit;


};


#endif
