#include "RedisIO.hh"
#include <iostream>

#include "Earthworm.hh"

RedisIO::RedisIO(const std::string& ip_addr, unsigned port, const std::string& usrname, const std::string& password)
:m_ip_addr(ip_addr), m_port(port), m_usrname(usrname), m_password(password)
{
  cpp_redis::client m_client;
  //High availability requires at least 2 io service workers, but it is very stable so far
  cpp_redis::network::set_default_nb_workers(2);

}

RedisIO::~RedisIO()
{
  m_client.disconnect();
}

void RedisIO::connect()
{
  m_client.add_sentinel(m_ip_addr, m_port);

  //Call connect with optional timeout
  //Can put a loop around this until is_connected() returns true.
  m_client.connect(m_usrname, [](const std::string& host, std::size_t port, cpp_redis::connect_state status)
  {
    if (status == cpp_redis::connect_state::dropped)
    {
      EW_LOG_CRITICAL("Redis client disconnected to {}:{}", host, port);
    }
  }, 0, -1, 5000);

  const std::string &ip_addr = this->m_ip_addr;
  const unsigned port = this->m_port;
  m_client.auth(m_password, [&ip_addr, port](const cpp_redis::reply& reply) {
    if (reply.is_error()) { std::cerr << "Authentication failed: " << reply.as_string() << std::endl; }
    else
    {
      EW_LOG_INFO("Connection to {}:{} is authrised", ip_addr, port);
    }
  });
  m_client.sync_commit();
}

void RedisIO::writeString(const std::string& path,const std::string& str)
{
  if(!m_client.is_connected())
    connect();

  m_client.set(path, str, [](cpp_redis::reply& reply) {
    if (reply.is_bulk_string())
       if(reply.as_string()!="OK")
         EW_LOG_WARN("RedisIO::writeString failed");
    else
      EW_LOG_WARN("RedisIO::writeString failed");
  });
  // synchronous commit, no timeout
  m_client.sync_commit();
}

std::string RedisIO::readString(const std::string& path)
{
  if(!m_client.is_connected())
    connect();

  std::string str;
  m_client.get(path, [&str](cpp_redis::reply& reply) {
    if (reply.is_string())
      str = reply.as_string();
    else
    EW_LOG_WARN("RedisIO::read reply is not a string");
  });
  m_client.sync_commit();
  return str;

}

// int RedisIO::readInt(const std::string& path)
// {
//   if(!m_client.is_connected())
//     connect();
//
//   int num;
//   m_client.get(path, [&num](cpp_redis::reply& reply) {
//     if (reply.is_integer())
//       num = reply.as_integer ();
//     else
//     EW_LOG_WARN("RedisIO::read reply is not a integer");
//   });
//   m_client.sync_commit();
//   return num;
// }

void RedisIO::writeInt(const std::string& path, int i)
{
  if(!m_client.is_connected())
    connect();

  m_client.set(path, std::to_string(i), [](cpp_redis::reply& reply) {
    if (reply.is_bulk_string())
       if(reply.as_string()!="OK")
         EW_LOG_WARN("RedisIO::writeInt failed");
    else
      EW_LOG_WARN("RedisIO::writeInt failed");
  });
  // synchronous commit, no timeout
  m_client.sync_commit();
}
