#!/usr/bin/python

import ctypes
import numpy as np
import glob
import os


libfile = glob.glob(os.path.dirname(__file__)+'/../src/libearthworm.so')[0]
vacflib = ctypes.CDLL(libfile)

def vacf(vel, velSize, idx, nAtom, nStep):
    vacflib.vaf.restype = ctypes.c_double
    vacflib.vaf.argtypes = [np.ctypeslib.ndpointer(dtype=np.float64,ndim=3,flags='C_CONTIGUOUS'),
                      ctypes.c_uint, ctypes.c_uint,ctypes.c_uint,ctypes.c_uint  ]
    return vacflib.vaf(vel, idx, nAtom, nStep)

def vacf_vec(vel, velSize, returnPointer, retsize, nAtom, nStep):
    vacflib.vaf_vec.argtypes = [np.ctypeslib.ndpointer(dtype=np.float64,ndim=3,flags='C_CONTIGUOUS'),
                            ctypes.c_uint,
                            np.ctypeslib.ndpointer(dtype=np.float64,ndim=1,flags='C_CONTIGUOUS'),
                            ctypes.c_uint,ctypes.c_uint,ctypes.c_uint  ]
    vacflib.vaf_vec(vel, velSize, returnPointer, retsize, nAtom, nStep)

def grt_vec(pos,  posSize,
            box,  boxSize,
            returnPointer,  retsize,
            numSpatialBin,  wavelengthCut,
            dltT,  nAtom,  nStep):
    vacflib.grt_vec.argtypes = [np.ctypeslib.ndpointer(dtype=np.float64,ndim=3,flags='C_CONTIGUOUS'),
                            ctypes.c_uint,
                            np.ctypeslib.ndpointer(dtype=np.float64,ndim=2,flags='C_CONTIGUOUS'),
                            ctypes.c_uint,
                            np.ctypeslib.ndpointer(dtype=np.float64,ndim=2,flags='C_CONTIGUOUS'),
                            ctypes.c_uint,
                            ctypes.c_uint, ctypes.c_double,
                            ctypes.c_double,ctypes.c_uint,ctypes.c_uint  ]

    vacflib.grt_vec(pos,  posSize,
                box,  boxSize,
                returnPointer,  retsize,
                numSpatialBin,  wavelengthCut,
                dltT,  nAtom,  nStep)

    rSpacingInAa = wavelengthCut/numSpatialBin
    tSpacingInfs = dltT

    spaRange = np.arange(returnPointer.shape[1]) *  rSpacingInAa

    #normalise
    for gr in returnPointer:
        gr /= (np.trapz(gr,spaRange))

    #scale by 4pi*r*r
    rVector = np.arange(numSpatialBin)*rSpacingInAa
    scalefact = (rVector**2 * 4 * np.pi)
    returnPointer[:,1:] /= scalefact [1:] # the first element is zeros
    return rSpacingInAa, tSpacingInfs, scalefact

def test(vec, vecsize):
    vacflib.test.argtypes = [np.ctypeslib.ndpointer(dtype=np.float64,flags='C_CONTIGUOUS'),
                            ctypes.c_uint]

    vacflib.test(vec,vecsize)

def vaf_python(vel,idx, nAtom, nStep): #idx must <= nStep
    tot =0.
    for n in range(nAtom):
        for j in range(nStep-idx):
            tot += vel[n,j].dot(vel[n,j+idx])
    return tot/(nAtom*(nStep-idx))

def vaff_python(vel,idx, nAtom, nStep): #idx must <= nStep
    tot =0.
    for n in range(nAtom):
        tot += (vel[n,0:(nStep-idx)]*vel[n,idx:nStep]).sum()
    return tot/(nAtom*(nStep-idx))
