#include "EpicsClasses.hh"

using namespace epics::pvData;
using namespace epics::pvAccess;


/** Requester implementation,
 *  used as base for all the following *Requester
 */
void MyRequester::message(std::string const & message, MessageType messageType)
{
    std::cout << getMessageTypeName(messageType) << ": "
         << requester_name << " "
         << message << std::endl;
}

/** Requester for channel and status updates */


void MyChannelRequester::channelCreated(const Status& status, Channel::shared_pointer const & channel)
{
    std::cout << channel->getChannelName() << " created, " << status << std::endl;
}

void MyChannelRequester::channelStateChange(Channel::shared_pointer const & channel, Channel::ConnectionState connectionState)
{
    std::cout << channel->getChannelName() << " state: "
         << Channel::ConnectionStateNames[connectionState]
         << " (" << connectionState << ")" << std::endl;
    if (connectionState == Channel::CONNECTED)
        connect_event.signal();
}



/** Requester for 'monitoring' value changes of a channel */
void MyMonitorRequester::monitorConnect(Status const & status, MonitorPtr const & monitor, StructureConstPtr const & structure)
{
    std::cout << "Monitor connects, " << status << std::endl;
    if (status.isSuccess())
    {
        // Check the structure by using only the Structure API?
        // Need to navigate the hierarchy, won't get the overall PVStructure offset.
        // Easier: Create temporary PVStructure
        PVStructurePtr pvStructure = getPVDataCreate()->createPVStructure(structure);
        std::tr1::shared_ptr<PVInt> user_tag = pvStructure->getSubField<PVInt>("timeStamp.userTag");
        if (! user_tag)
        {
            std::cout << "No 'timeStamp.userTag'" << std::endl;
            return;
        }
        user_tag_offset = user_tag->getFieldOffset();

        std::tr1::shared_ptr<PVUIntArray> tof = pvStructure->getSubField<PVUIntArray>("time_of_flight.value");
        if (! tof)
        {
            std::cout << "No 'time_of_flight'" << std::endl;
            return;
        }
        tof_offset = tof->getFieldOffset();

        std::tr1::shared_ptr<PVUIntArray> pixel = pvStructure->getSubField<PVUIntArray>("pixel.value");
        if (! pixel)
        {
            std::cout << "No 'pixel'" << std::endl;
            return;
        }
        pixel_offset = pixel->getFieldOffset();

        // pvStructure is disposed; keep value_offset to read data from monitor's pvStructure

        monitor->start();
    }
}


void MyMonitorRequester::monitorEvent(MonitorPtr const & monitor)
{

	shared_ptr<MonitorElement> update;
  while ((update = monitor->poll() ))
  {
    ++updates;
    checkUpdate(update->pvStructurePtr);
    epics::pvData::PVUIntArrayPtr pixelsPtr_client = update->pvStructurePtr->getSubField<epics::pvData::PVUIntArray>("pixel.value");
    epics::pvData::PVUIntArrayPtr tofPtr_client = update->pvStructurePtr->getSubField<epics::pvData::PVUIntArray>("time_of_flight.value");
    if(pixelsPtr_client || tofPtr_client)
    {
  	 if(pixelsPtr_client)
  		 pixelsLength = pixelsPtr_client->getLength();
  	 else
  		 pixelsLength = tofPtr_client->getLength();
  	 if(pixelsPtr_client)
  		 pixelsData   = pixelsPtr_client->view();
  	 if(tofPtr_client)
  		 tofData      = tofPtr_client->view();

     for(int list_num_pixel = 0; list_num_pixel<pixelsLength; list_num_pixel++)
     {
       pSetFunc(tofPtr_client?tofData[list_num_pixel]:0, pixelsPtr_client?pixelsData[list_num_pixel]:0);
     }
   }

	 if (! update->overrunBitSet->isEmpty())
		 ++overruns;
	 if (quiet)
	 {
		 epicsTime now(epicsTime::getCurrent());
		 if (now >= next_run)
		 {
			 double received_perc = 100.0 * updates / (updates + missing_pulses);

			 overruns = 0;
			 missing_pulses = 0;
			 updates = 0;
			 array_size_differences = 0;

#               ifdef TIME_IT
			 std::cout << "Time for value lookup: " << value_timer << std::endl;
#               endif

			 next_run = now + 10.0;
		 }
	 }
	 monitor->release(update);
   }
    ++ monitors;
    //std::cout << " 222222...... "<<std::endl;
    if (limit > 0  &&  monitors >= limit)
    {
	    std::cout << "Received " << monitors << " monitors" << std::endl;
	    done_event.signal();
    }

    //std::cout << "pass pass go  go " <<std::endl;
}

// void MyMonitorRequester::getPulseData(NeutronPulseData *pNeutronPulseData)
// {

//    std::cout <<" print the structure::" <<*(pNeutronPulseData->pTimeOfFlight) <<std::endl;

// }


void MyMonitorRequester::checkUpdate(shared_ptr<PVStructure> const &pvStructure)
{
#   ifdef TIME_IT
	value_timer.start();
#   endif

	// Time for value lookup when re-using offset: 2us
	shared_ptr<PVInt> value = dynamic_pointer_cast<PVInt>(pvStructure->getSubField(user_tag_offset));

	// Compare: Time for value lookup when using name: 12us
	// shared_ptr<PVInt> value = pvStructure->getIntField("timeStamp.userTag");
	if (! value)
	{
		std::cout << "No 'timeStamp.userTag'" << std::endl;
		return;
	}

#   ifdef TIME_IT
	value_timer.stop();
#   endif

	// Check pulse ID for skipped updates
	uint64 pulse_id = static_cast<uint64>(value->get());
	if (last_pulse_id != 0)
	{
		int missing = pulse_id - 1 - last_pulse_id;
		if (missing > 0)
			missing_pulses += missing;
	}
	last_pulse_id = pulse_id;

	// Compare lengths of tof and pixel arrays
	shared_ptr<PVUIntArray> tof =
		dynamic_pointer_cast<PVUIntArray>(pvStructure->getSubField(tof_offset));
	epics::pvData::shared_vector<const epics::pvData::uint32> tofData = tof->view();
	if (!tof)
	{
		std::cout << "No 'time_of_flight' array" << std::endl;
		return;
	}

	shared_ptr<PVUIntArray> pixel = dynamic_pointer_cast<PVUIntArray>(pvStructure->getSubField(pixel_offset));
	if (!pixel)
	{
		std::cout << "No 'pixel' array" << std::endl;
		return;
	}

	if (tof->getLength() != pixel->getLength())
	{
		++array_size_differences;
		if (! quiet)
		{
			std::cout << "time_of_flight: " << tof->getLength() << " elements" << std::endl;
			shared_vector<const uint32> tof_data;
			tof->getAs(tof_data);
			std::cout << tof_data << std::endl;

			std::cout << "pixel: " << pixel->getLength() << " elements" << std::endl;
			shared_vector<const uint32> pixel_data;
			pixel->getAs(pixel_data);
			std::cout << pixel_data << std::endl;
		}
	}
}


void MyMonitorRequester::unlisten(MonitorPtr const & monitor)
{
	std::cout << "Monitor unlistens" << std::endl;
}
