#ifndef StateMachine_hh
#define StateMachine_hh

#include <string>
#include <vector>
#include <map>

//This class models a state machine of m_numState state.
//Transit is initiated by received commands.

//Internally,this class contains a m_numState by m_numState size matrix-like
//data, which records the premitted transit from state A to B triggered by a
//command X

class StateMachine {
public:
  const unsigned kNumState = 7;
  const unsigned kNumCommand = 7;
  //Notice both State and Command must start from zero
  //and each variable increases by 1. Limited by cmd2int and int2cmd
	enum State {
    kWaiting,
    kUnconfigured,
    kConfiguring,
    kReady,
    kRunning,
    kPaused,
    kError
  };

  const std::map<State, std::string> kStateStrMap = {
  {State::kWaiting, "waiting"},
  {State::kUnconfigured, "unconfigured"},
  {State::kConfiguring, "configuring"},
  {State::kReady, "ready"},
  {State::kRunning, "running"},
  {State::kPaused, "paused"},
  {State::kError, "error"}
  };


  enum Command {
    kUnknown = -1,
    kConfigure, //0
    kUnconfigure,
    kStart, //2
    kPause,
    kResume,//4
    kStop,
    kAbort
  };

  const std::map<Command, std::string> kCommandStrMap = {
  {Command::kConfigure, "configure"},
  {Command::kUnconfigure, "unconfigure"},
  {Command::kStart, "start"},
  {Command::kPause, "pause"},
  {Command::kResume, "resume"},
  {Command::kStop, "stop"},
  {Command::kAbort, "abort"}
  };



public:
  StateMachine();
  //instrument 1 is MR, others to be implemented
  StateMachine(int instrument);
  virtual ~StateMachine();

  void grantStateTransit(State from, State to, Command by);
  void denyStateTransit(State from, State to);

  //state machine accepts invalid commands only if m_state is unconfigured
  State transit(int byCommand); //perform on m_state
  State transit(State from, int byCommand);

  std::string printState() const;
  std::string printState(State s) const;

  void selfDiagnosis(); //slow, use with care

protected:
  State m_state;
  Command int2cmd(int i) const;
  State int2state(int i) const;

private:
  //check if a state accepts duplicated commmand
  bool tableSanityCheck(State from);
  std::vector<int> m_allowedStateTransit;
};

#endif
