#include <dic.hxx>
#include <iostream>
#include <vector>
#include <string>
#include <fstream>
#include <memory>
#include <chrono>

#include <nlohmann/json.hpp>

#include "DAQInterface.hh"
#include "Hist2DMap.hh"
#include "RedisIO.hh"
#include "Earthworm.hh"
#include "RedisNumpy.hh"
#include "RunStatus.hh"

int main()
{
	std::cout << "Starting client..." << std::endl;
  unsigned pid_num(1200), tof_num(2500);

	Hist2DMap::DetectorType detType = Hist2DMap::DetectorType::kUnknown;

	std::string prefix = "/sim";
	std::string cmdPath = prefix+"/MR/control/command/earthworm";
  std::string hbpath = prefix + "/MR/heartbeat/earthworm"; //heartbeat
	std::string mrconfPath = prefix+"/MR/control/configure";


  //dim and hist
	std::string dimAddr = prefix+"/dimserver/TEST_SWAP_1";
  std::string pidPath = prefix + "/MR/workspace/detector/module1/pid";
	std::string tofbinPath = prefix + "/MR/workspace/detector/module1/tof";
	std::string detHistPath = prefix + "/MR/workspace/detector/module1/value";


	std::string mwpcDetCntPath = prefix + "/MR/earthworm/detector_counts";
  std::string he3DetCntPath = prefix + "/MR/earthworm/detector_counts_tube";

  std::string protonChargePath = prefix + "/MR/earthworm/proton_charge";
  std::string moniPath = prefix +	"/MR/earthworm/monitor_counts";
	std::string pulsePath = prefix + "/MR/earthworm/pulse_counts";


	std::shared_ptr<RedisIO> redis(
		new RedisIO("10.1.31.116", 9001, "neonmaster", "sanlie;123"));
  //redis should retrive all parameters
  std::shared_ptr<Hist2DMap> histmap(new Hist2DMap(pid_num, 0, 300, tof_num, 0 , 40000 , Hist2DMap::DetectorType::kHe3) );
	std::shared_ptr<RunStatus> sMachine (new RunStatus(redis,cmdPath,hbpath));

  //RunNumber runNumber;
	DAQInterface daq(dimAddr,100);
  daq.setHist2DMap(histmap);

	std::vector<double> pid;
	for(unsigned i=0;i<pid_num;i++)
	  pid.push_back(i+1);
	RedisNumpy nvt;
  std::string pidstr;
	std::vector<uint64_t> shape;
	shape.push_back(pid_num);
	nvt.makeNumpyArr(pid, RedisNumpy::data_type::f8,
									 shape, pidstr);
	redis->writeString(pidPath, pidstr);

	std::vector<double> tof;
	for(unsigned i=0;i<tof_num+1;i++)
	{
		tof.push_back(i*16);
	}
	std::string tofstr;
	shape[0]=(tof_num+1);
	nvt.makeNumpyArr(tof, RedisNumpy::data_type::f8,
									 shape, tofstr);
	redis->writeString(tofbinPath, tofstr);


	StateMachine::Command lastCommand(StateMachine::Command::kIneffective);
	while(lastCommand != RunStatus::Command::kAbort)
	{
		sMachine->sendHeartBeat();

		std::this_thread::sleep_for(std::chrono::milliseconds(100));

		if(sMachine->getState()==StateMachine::State::kRunning)
		{
			// auto start = std::chrono::steady_clock::now();

			std::string str;
			histmap->makeNumpyRaw(str);

			// std::cout << "until makeNumpyRaw completed, elapsed time  : "
			// 	<< std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - start).count()
			// 	<< " ms" << std::endl;

			redis->writeString(detHistPath, str);
			//
			// std::cout << "until writeString completed, elapsed time  : "
			// 	<< std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - start).count()
			// 	<< " ms" << std::endl;

			if(detType==Hist2DMap::DetectorType::kHe3)
				  redis->writeInt(he3DetCntPath, histmap->getIntegral());
			else if(detType==Hist2DMap::DetectorType::kMWPC)
				  redis->writeInt(mwpcDetCntPath, histmap->getIntegral());
			else
			  EW_LOG_CRITICAL("Detector type is unconfigured!");

			// std::cout << "until writeInt completed, elapsed time  : "
			// 	<< std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - start).count()
			// 	<< " ms" << std::endl;

			EW_LOG_INFO("Data integral {}", histmap->getIntegral());



		}

		//detector command in redis
		StateMachine::Command cmd = sMachine->update();

		if(cmd==StateMachine::Command::kIneffective || cmd==lastCommand)
		  continue;

    //Configure detector type and reset buffer
  	if(cmd==StateMachine::Command::kConfigure)
		{
			daq.setActive(false);

			std::string conf = redis->readString(mrconfPath);
			auto j =	nlohmann::json::parse(conf);
			std::string dt= j["mode"];
			EW_LOG_INFO("Instrument active detector is {}", dt);

			if(dt=="He3")
				detType=Hist2DMap::DetectorType::kHe3;
			else if(dt=="MWPC")
				detType=Hist2DMap::DetectorType::kMWPC;
			histmap->reset();

			EW_LOG_INFO("Configuration completed.");

		}
		//Stop running and clear any runtime determined parameter
		else if(cmd==StateMachine::Command::kUnconfigure)
		{
			daq.setActive(false);
			detType = Hist2DMap::DetectorType::kUnknown;
			EW_LOG_INFO("kUnconfigured.");
		}
		//Start accumulate data, RESET AGAIN HERE
		else if(cmd==StateMachine::Command::kStart)
		{
			daq.setActive(true);
			histmap->reset();
			EW_LOG_INFO("Run started and reseted buffer.");
		}
		//Paused fill histogram, still receive but unprocess dim datastream
		else if(cmd==StateMachine::Command::kPause || cmd==StateMachine::Command::kStop)
	  {
			daq.setActive(false);
			EW_LOG_INFO("kPause or kStop.");
		}
		//Restart running
		else if(cmd==StateMachine::Command::kResume)
	  {
			daq.setActive(true);
			EW_LOG_INFO("kResume.");

		}
		//Aborting
		else if(cmd==StateMachine::Command::kAbort)
		{
			EW_LOG_INFO("aborting.");

		}
		else {
			//fixme: this will be fre
			EW_LOG_CRITICAL("earthworm_mr received unsupported command");
		}

		lastCommand = cmd;

	}

	EW_LOG_INFO("Earthworm_mr received abort command. Exiting...");
	sMachine->sendHeartBeat(RunStatus::State::kError);

	// std::this_thread::sleep_for(std::chrono::milliseconds(100));
  return 0;
}
