
#include <dic.hxx>
#include <iostream>
#include <vector>
#include <string>
#include <fstream>
#include <memory>
#include <chrono>
#include <future>
#include <nlohmann/json.hpp>

#include "Action_PMT.hh"
#include "Earthworm.hh"


Action_PMT::Action_PMT(RedisIO *redis, StateMachine *sMachine, std::string module)
: Action(redis),  m_detType(Hist2DMap::DetectorType::kUnknown),
  m_hist( nullptr),
	m_sMachine(sMachine), m_active_daq(nullptr), m_module(module)
{
  std::ifstream f("../allPathGPPD.json");
  m_allPath.clear();
  f >> m_allPath;
  f.close();

  m_active_daq = new DAQInterfaceGPPD(m_allPath[m_module]["dimserver"],1000);
}

Action_PMT::~Action_PMT()
{
	delete m_active_daq;

	EW_LOG_INFO("~Action_PMT ref count of hist is {}", m_hist.use_count());
	m_hist.reset();
}

bool Action_PMT::configure()
{
	//fixme: update m_allPath fisrt !!!!!

	std::ifstream f("../allPathGPPD.json");
	m_allPath.clear();
	f >> m_allPath;
  f.close();

  //construct histogram add couple with daq
  unsigned pid_num(1344), tof_num(2500);
	m_hist.reset(new NumpyHist2D ( tof_num, 0., tof_num*16, pid_num, 0.-0.5, pid_num-0.5));

  m_active_daq->setHist2D(m_hist);

  //send pid
	std::vector<unsigned> pid;
	for(unsigned i=0;i<pid_num;i++)
		pid.push_back(i+1230001);
	RedisNumpy nvt;
	std::string pidstr;
	std::vector<uint64_t> shape;
	shape.push_back(pid_num);
	nvt.makeNumpyArr(pid, RedisNumpy::data_type::i4,
									 shape, pidstr);
	m_redis->writeString(m_allPath[m_module]["rawDataPixel"], pidstr);

  //send tof id
	std::vector<unsigned> tof;
	for(unsigned i=0;i<tof_num+1;i++)
	{
		tof.push_back(i*16);
	}
	std::string tofstr;
	shape[0]=(tof_num+1);
	nvt.makeNumpyArr(tof, RedisNumpy::data_type::i4,
									 shape, tofstr);
	m_redis->writeString(m_allPath[m_module]["rawDataTof"], tofstr);

  //send empty hist
  std::string redisDataStr;
  m_hist->serialise(redisDataStr);
  m_redis->writeString(m_allPath[m_module]["rawDataHist"], redisDataStr);

	m_hist->reset();
  m_active_daq->reset();

	EW_LOG_INFO("configure:: Configuration completed. ");
	return true;
}

//Stop running and clear any runtime determined parameter
bool Action_PMT::unconfigure()
{
	m_active_daq->setActive(false);
	EW_LOG_INFO("unconfigure:: Earthworm unconfigured.");
	return true;
}

//Start accumulate data, RESET AGAIN HERE
bool Action_PMT::start()
{
  m_hist->reset();
  std::string redisDataStr;
  m_hist->serialise(redisDataStr);
  m_redis->writeString(m_allPath[m_module]["rawDataHist"], redisDataStr);

	m_active_daq->setActive(false);
	m_active_daq->setActive(true);
  m_active_daq->reset();

	m_startTime = std::chrono::steady_clock::now();
  EW_LOG_INFO("start:: Run started.");

	return true;
}

//Paused fill histogram, still receive but unprocess dim datastream
bool Action_PMT::pauseStop()
{
	m_active_daq->setActive(false);
	EW_LOG_INFO("pauseStop:: kPause or kStop.");
	return true;
}

bool Action_PMT::abort()
{
	m_active_daq->setActive(false);
	EW_LOG_INFO("abort:: aborting");
	return true;
}

//Restart running
bool Action_PMT::resume()
{
	m_active_daq->setActive(true);
	EW_LOG_INFO("resume:: kResume.");
	return true;
}

bool Action_PMT::kill()
{
	if(m_active_daq)
		m_active_daq->setActive(false);
	m_kill=true;
	return true;
}


void Action_PMT::sendHeartbeat(bool stateAsStr)
{
  nlohmann::json j;
  j["pid"] =std::to_string(m_proID);
	if(stateAsStr)
	  j["status"] = m_sMachine->printState();
  else
		j["status"] = std::to_string(m_sMachine->getState());

  std::time_t t = std::time(nullptr);
  char mbstr[100];
  if (std::strftime(mbstr, sizeof(mbstr), "%Y-%m-%dT%H:%M:%S.%03d+08:00", std::localtime(&t))) {
  }

  j["timestamp"] = std::string (mbstr);
  std::string hb= j.dump();
	// std::cout << "Heartbeart " << hb  << '\n';

  m_redis->writeString(m_allPath[m_module]["hbEarthworm"],hb);

  if(m_active_daq)
    m_redis->writeString(m_allPath[m_module]["pulstCnt"],std::to_string(m_active_daq->getDataFlow().num_dim_trigger));

}

bool Action_PMT::update()
{
	std::string str = m_redis->readString(m_allPath["ctrlEarthworm"]);
  return m_sMachine->transit(atoi(str.c_str()));
}

bool Action_PMT::keepRunning()
{
	std::string redisDataStr;
	double totcnt = m_hist->getIntegral();
	double cnt_rate=0.;
	if(totcnt)
  {
		cnt_rate=totcnt/std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - m_startTime).count()*1000;
		EW_LOG_INFO("keepRunning:: Data integral {}, rate is {} event/s", totcnt,
							cnt_rate);
  }
	else
		EW_LOG_INFO("keepRunning:: Data integral is 0");

	m_redis->writeString(m_allPath[m_module]["DetCnt"], std::to_string((int) m_hist->getIntegral()) );
	m_redis->writeString(m_allPath[m_module]["DetCntRate"], std::to_string(cnt_rate) );
  m_hist->serialise(redisDataStr);
  m_redis->writeString(m_allPath[m_module]["rawDataHist"], redisDataStr);
	return true;
}
