#include <dic.hxx>
#include <iostream>
#include <vector>
#include <string>
#include <fstream>
#include <memory>
#include <chrono>
#include <future>
#include <nlohmann/json.hpp>

#include "DAQInterface.hh"
#include "Hist2DMap.hh"
#include "RedisIO.hh"
#include "Earthworm.hh"
#include "RedisNumpy.hh"
#include "RunStatus.hh"

int main()
{
	std::cout << "Starting client..." << std::endl;
  unsigned pid_num(300), tof_num(2500);

	Hist2DMap::DetectorType detType = Hist2DMap::DetectorType::kUnknown;

	std::string prefix = "";
	std::string cmdPath = prefix+"/MR/control/command/earthworm";
  std::string hbpath = prefix + "/MR/heartbeat/earthworm"; //heartbeat
	std::string mrconfPath = prefix+"/MR/control/configure";


  //dim and hist
	std::string dimAddr = prefix+"dimserver/TEST_SWAP_1";
  std::string pidPath = prefix + "/MR/workspace/detector/module1/pid";
	std::string tofbinPath = prefix + "/MR/workspace/detector/module1/tof";
	std::string detHistPath = prefix + "/MR/workspace/detector/module1/value";


	std::string mwpcDetCntPath = prefix + "/MR/earthworm/detector_counts";
  std::string he3DetCntPath = prefix + "/MR/earthworm/detector_counts_tube";
  std::string he3DetRatePath = prefix + "/MR/earthworm/detector_counts_tube_rate";

  //to be impletment
  std::string protonChargePath = prefix + "/MR/earthworm/proton_charge";
  std::string moniPath = prefix +	"/MR/earthworm/monitor_counts"; //monitor
	std::string pulsePath = prefix + "/MR/earthworm/pulse_counts";


	std::shared_ptr<RedisIO> redis(new RedisIO(9001, "neonmaster", "sanlie;123"));
  //redis should retrive all parameters
  std::shared_ptr<Hist2DMap> histmap(new Hist2DMap(tof_num, 0 , 40000, pid_num, 0, 300 , Hist2DMap::DetectorType::kHe3) );
	std::shared_ptr<RunStatus> sMachine (new RunStatus(redis,cmdPath,hbpath));

  //RunNumber runNumber;
	DAQInterface daq(dimAddr,1000);
  daq.setHist2DMap(histmap);
        auto start = std::chrono::steady_clock::now();

	std::vector<double> pid;
	for(unsigned i=0;i<pid_num*4;i++)
	  pid.push_back(i+1);
	RedisNumpy nvt;
  std::string pidstr;
	std::vector<uint64_t> shape;
	shape.push_back(pid_num*4);
	nvt.makeNumpyArr(pid, RedisNumpy::data_type::f8,
									 shape, pidstr);
	redis->writeString(pidPath, pidstr);

	std::vector<double> tof;
	for(unsigned i=0;i<tof_num+1;i++)
	{
		tof.push_back(i*16);
	}
	std::string tofstr;
	shape[0]=(tof_num+1);
	nvt.makeNumpyArr(tof, RedisNumpy::data_type::f8,
									 shape, tofstr);
	redis->writeString(tofbinPath, tofstr);

	std::string redisDataStr;
	StateMachine::Command lastCommand(StateMachine::Command::kIneffective);
	while(lastCommand != RunStatus::Command::kAbort)
	{
		sMachine->sendHeartBeat();

		std::this_thread::sleep_for(std::chrono::milliseconds(100));
		if(sMachine->getState()==StateMachine::State::kRunning)
		{
			// auto start = std::chrono::steady_clock::now();
  		 //fixme:make this job detach if it is too heavy
      // redis_future = std::async([](int m) {return 2 * m;} , i);
			// auto status = m_future.wait_for(std::chrono::milliseconds(0));
		  // return status == std::future_status::ready

			histmap->makeNumpyRaw(redisDataStr);

			// std::cout << "until makeNumpyRaw completed, elapsed time  : "
			// 	<< std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - start).count()
			// 	<< " ms" << std::endl;

			redis->writeString(detHistPath, redisDataStr);

      double totcnt = histmap->getIntegral();
      double cnt_rate=0.;
      if(totcnt)
        cnt_rate=totcnt/std::chrono::duration_cast<std::chrono::seconds>(std::chrono::steady_clock::now() - start).count();
      if(totcnt)
      EW_LOG_INFO("Data integral {}, rate is {} event/s", totcnt,
                  cnt_rate);


			if(detType==Hist2DMap::DetectorType::kHe3)
      {
				  redis->writeString(he3DetCntPath, std::to_string((int) histmap->getIntegral()) );
          redis->writeString(he3DetRatePath,std::to_string(cnt_rate) );
      }
      else if(detType==Hist2DMap::DetectorType::kMWPC)
				  redis->writeString(mwpcDetCntPath,std::to_string((int) histmap->getIntegral())+"}");
			else
			  EW_LOG_CRITICAL("Detector type is unconfigured!");

			// std::cout << "until writeInt completed, elapsed time  : "
			// 	<< std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - start).count()
			// 	<< " ms" << std::endl;
		}

		//detector command in redis
		StateMachine::Command cmd = sMachine->update();

		if(cmd==StateMachine::Command::kIneffective || cmd==lastCommand)
		  continue;

    //Configure detector type and reset buffer
  	if(cmd==StateMachine::Command::kConfigure)
		{
			daq.setActive(false);

			//std::string conf = redis->readString(mrconfPath);
			//auto j =	nlohmann::json::parse(conf);
			//std::string dt= j["mode"];
			//EW_LOG_INFO("Instrument active detector is {}", dt);
      std::string dt="He3";
			if(dt=="He3")
				detType=Hist2DMap::DetectorType::kHe3;
			else if(dt=="MWPC")
				detType=Hist2DMap::DetectorType::kMWPC;
			histmap->reset();

			EW_LOG_INFO("Configuration completed.");

		}
		//Stop running and clear any runtime determined parameter
		else if(cmd==StateMachine::Command::kUnconfigure)
		{
			daq.setActive(false);
			detType = Hist2DMap::DetectorType::kUnknown;
			EW_LOG_INFO("kUnconfigured.");
		}
		//Start accumulate data, RESET AGAIN HERE
		else if(cmd==StateMachine::Command::kStart)
		{
                        start = std::chrono::steady_clock::now();
			daq.setActive(true);
			histmap->reset();
			EW_LOG_INFO("Run started and reseted buffer.");
		}
		//Paused fill histogram, still receive but unprocess dim datastream
		else if(cmd==StateMachine::Command::kPause || cmd==StateMachine::Command::kStop)
	  {
			daq.setActive(false);
			EW_LOG_INFO("kPause or kStop.");
		}
		//Restart running
		else if(cmd==StateMachine::Command::kResume)
	  {
			daq.setActive(true);
			EW_LOG_INFO("kResume.");

		}
		//Aborting
		else if(cmd==StateMachine::Command::kAbort)
		{
			EW_LOG_INFO("aborting.");

		}
		else {
			//fixme: this will be fre
			EW_LOG_CRITICAL("earthworm_mr received unsupported command");
		}

		lastCommand = cmd;

	}

	EW_LOG_INFO("Earthworm_mr received abort command. Exiting...");
	sMachine->sendHeartBeat(RunStatus::State::kError);

	// std::this_thread::sleep_for(std::chrono::milliseconds(100));
  return 0;
}
