#include "PackageHe3.hh"
#include "UtilsException.hh"
#include "Hist2DMap.hh"

Parser::PackageHe3::PackageHe3(double threshold, double hvlength,double tofscale)
:ParserHe3(), m_charge_threshold(threshold),
m_detector_half_length(hvlength),m_tof2us(tofscale)
{

}

Parser::PackageHe3::~PackageHe3()
{
}

double Parser::PackageHe3::getChargeThreshold() const
{
  return m_charge_threshold;
}
void Parser::PackageHe3::setChargeThreshold(double charge)
{
  std::lock_guard<std::mutex> guard(m_mutex);
  m_charge_threshold = charge;
}


bool Parser::PackageHe3::readNextPackage(const uint8_t*& pt, const uint8_t * const stream_end, Hist2DMap &hitmap) const
{
  const uint8_t *package_end;
  uint64_t offset = 0;
  if(!findNextPackage(pt, stream_end, package_end, offset))
    return false;

  const Parser::ParserHe3::Header *header = new Parser::ParserHe3::Header();
  readHeader(pt, header);
  delete header;

  Parser::ParserHe3::Info info;
  readInfo(pt, info);
  double baseline_factor = 1./info.sample_num_bl;

  Parser::ParserHe3::Hit hit ;

  //the next while loop breaks when pt==package_end-8
  while (pt != package_end-8) {
    error_assert(pt<package_end-8); //shouldn't really happen
    readHit(pt,hit);
    double upper = hit.upper_charge - hit.upper_baseline*baseline_factor;
    double lower = hit.lower_charge - hit.lower_baseline*baseline_factor;
    double tot = upper + lower;
    //fixme: why tot is sometimes zero?
    if(!tot || tot<m_charge_threshold)
      continue;
    else {
      hitmap.fill(hit.tube_id, hit.time*m_tof2us ,m_detector_half_length*( 1 + (upper-lower)/tot ) );

        // printf("pos %g, time stamp %d , tube_id %d\n", 150.*( 1 + (upper-lower)/tot ) ,hit.time, hit.tube_id );
      }
  }
  pt = package_end;

  //printf("left %ld, package_end %p, stream_end %p\n", stream_end-package_end, package_end, stream_end);
  return package_end==stream_end?false:true;
}


bool Parser::PackageHe3::readNextPackage(const uint8_t*& pt, const uint8_t * const stream_end, std::vector<HitInfo>& hitinfo) const
{
  const uint8_t *package_end;
  uint64_t offset=0;

  if(!findNextPackage(pt, stream_end, package_end, offset))
    return false;

  const Parser::ParserHe3::Header *header = new Parser::ParserHe3::Header();
  readHeader(pt, header);
  delete header;

  Parser::ParserHe3::Info info;
  readInfo(pt, info);
  double baseline_factor = 1./info.sample_num_bl;

  Parser::ParserHe3::Hit hit ;
  std::vector<HitInfo> temp;
  std::swap(temp,hitinfo);
  hitinfo.reserve(info.hit_cnt);

  //the next while loop breaks when pt==package_end-8
  while (pt != package_end-8) {
    error_assert(pt<package_end-8); //shouldn't really happen
    readHit(pt,hit);
    double upper = hit.upper_charge - hit.upper_baseline*baseline_factor;
    double lower = hit.lower_charge - hit.lower_baseline*baseline_factor;
    double tot = upper + lower;
    if(!tot || tot<m_charge_threshold)
      continue;
    else {
        hitinfo.push_back(HitInfo( m_detector_half_length*( 1 + (upper-lower)/tot ) ,
        hit.time*m_tof2us,tot, info.t0_count, hit.tube_id));
        // printf("pos %g, time stamp %d , tube_id %d\n", 150.*( 1 + (upper-lower)/tot ) ,hit.time, hit.tube_id );
      }
  }
  pt = package_end;

  //printf("left %ld, package_end %p, stream_end %p\n", stream_end-package_end, package_end, stream_end);
  return package_end==stream_end?false:true;
}
