#include "DAQInterface.hh"
#include "Earthworm.hh"
#include "Hist2DMap.hh"

#include <iostream>
#include <algorithm>  // remove_if
#include <time.h>
#include <csignal>

DAQInterface::DAQInterface(std::string daq_name,
                           unsigned houseKeeping_itervl_ms)
: DimClient(), m_active_dim(false), m_terminate(false), m_reconnecting(false), m_errhdl()
, m_daq_data(new DimStampedInfo(daq_name.c_str(),const_cast<char *>("dataAsCharArr"), this))
, m_flow(DataFlow{})
, m_houskpg_intvl_ms(houseKeeping_itervl_ms), m_start_time(std::chrono::system_clock::now())
, m_hist_map(nullptr), m_hist2d(nullptr), m_dim_name(daq_name)
{
  addErrorHandler(&m_errhdl);
  EW_LOG_INFO("Dim client to server {} is initialised, but server status is not check.",daq_name);

  // m_hk_thread = std::thread(&DAQInterface::houseKeeping, this);
  m_hk_future = std::async(std::launch::async, &DAQInterface::houseKeeping, this);

}

DAQInterface::~DAQInterface()
{
  m_active_dim=false;
  m_terminate=true;

  //fixme::put a loop to check
  auto status = m_hk_future.wait_for(m_houskpg_intvl_ms);

  EW_LOG_INFO("DAQInterface destructed ");

}

void DAQInterface::setHist2DMap(const std::shared_ptr<Hist2DMap>& map)
{
  std::lock_guard<std::mutex> guard(m_task_mutex);
  m_hist_map = std::shared_ptr<Hist2DMap>(map);
}

void DAQInterface::setHist2D(const std::shared_ptr<NumpyHist2D>& hist)
{
  std::lock_guard<std::mutex> guard(m_task_mutex);
  m_hist2d = std::shared_ptr<NumpyHist2D>(hist);
}

uint64_t DAQInterface::numTriggered()
{
  return m_flow.num_dim_trigger;
}

void DAQInterface::setActive(bool act)
{
  m_active_dim=act;
}

void DAQInterface::reconnect()
{
  m_daq_data.reset();
  std::this_thread::sleep_for(std::chrono::milliseconds(10000));
  m_daq_data = std::make_shared<DimStampedInfo>(m_dim_name.c_str(),const_cast<char *>("dataAsCharArr"), this);
}

void DAQInterface::infoHandler()
{
  if(!m_daq_data.get()) {
    return;
  }

  // // Get Current Server Identifier
  if(!(m_flow.num_dim_trigger%1000))
    EW_LOG_INFO("infoHandler getServerId {}", getServerId());

  m_server_pid = getServerId();
  if (!m_server_pid)
  {
    EW_LOG_WARN("Connect failed. Reconnecting to service {}", m_dim_name);
    reconnect();
    // m_hk_future = std::async(std::launch::async, &DAQInterface::reconnect, this);
    return;
  }


  // EW_LOG_INFO("my id {}", m_server_pid);
  m_flow.num_dim_trigger++;
  // auto start = std::chrono::system_clock::now();
  if (m_active_dim) {
    DimInfo *curr = getInfo(); // get current DimInfo address
    if(curr == m_daq_data.get())
    {
      //infohandler will be triggered by constructor in dim v20
      //when the buffer shouble empty
      if(m_flow.num_dim_trigger<2) return;
      unsigned dsize  = curr->getSize();
      m_flow.received_byte += dsize;

      uint8_t *data = static_cast<uint8_t*>(curr->getData());
      std::vector<uint8_t> buf(data, data+dsize);

      {
        std::lock_guard<std::mutex> guard(m_task_mutex);


        if(m_hist2d)
          m_tasks.emplace_back( DaqTask(data, data+dsize, m_hist2d));
        else if(m_hist_map)
          m_tasks.emplace_back( DaqTask(data, data+dsize, m_hist_map));
        else
          m_tasks.emplace_back( DaqTask(data, data+dsize));

        m_tasks.back().setTime(curr->getTimestamp(), curr->getTimestampMillisecs());
        m_tasks.back().start();
      }
    }
    else   {
      //fixme:log
    }
  }

}

void DAQInterface::report()
{
  if( !(m_flow.num_dim_trigger%1000))
  {
    std::cout << "processed " << m_flow.processed_byte/1024/1024  << "MB. Process rate " << m_flow.processed_byte*100./m_flow.received_byte <<"%.  ";

    // std::chrono::duration<double> elapsed_seconds = std::chrono::system_clock::now()-m_start_time;
    // double t = elapsed_seconds.count();
    //
     std::cout << "processed " << m_flow.processed_event << "events. Triggered " << m_flow.num_dim_trigger << " times."<< std::endl; //<< " events. Event rate "
    // <<  m_flow.processed_event/t <<" /s. \n";
  }

}

void DAQInterface::houseKeeping()
{
  EW_LOG_INFO("DAQInterface starting periodic housekeeping at {}ms interval",
  m_houskpg_intvl_ms.count());


  while (!m_terminate) //fixme: implement cv sync instead of this dead loop
  {
    std::this_thread::sleep_for(m_houskpg_intvl_ms);

    if(m_tasks.empty())
      continue;

    {
      std::lock_guard<std::mutex> guard(m_task_mutex);

      unsigned processed =0;

      auto newend = std::remove_if(m_tasks.begin(), m_tasks.end(),
      [&](DaqTask &atask ) -> bool
      {
        uint64_t processed = atask.readyAndValid();
        if(! processed )
          return false;
        else
        {
          //combine byte size and event count as a uint64_t
          int bytes = processed & 0xffffffff;
          int events = (processed >> 32) & 0xffffffff;
          m_flow.processed_byte += bytes;
          m_flow.processed_event += events;
          return true;
        }
      });

      if(newend == m_tasks.end() ) continue;

      unsigned deleted = m_tasks.end() - newend;
      m_flow.completed_task += deleted;
      m_tasks.erase(newend, m_tasks.end());
      }


    report();
  }
  EW_LOG_INFO("Exiting house keeping");
}
