#include "Hist2DMap.hh"
#include "UtilsException.hh"
#include "NumpyHist2D.hh"
#include "Earthworm.hh"
#include "RedisNumpy.hh"

Hist2DMap::Hist2DMap(unsigned nbinx, double xmin, double xmax,
       unsigned nbiny, double ymin, double ymax, DetectorType dt)
: m_xbin(nbinx), m_xl(xmin), m_xu(xmax), m_ybin(nbiny), m_yl(ymin), m_yu(ymax)
, m_dettype(dt)
{
}

Hist2DMap::~Hist2DMap()
{
  auto it = m_histmap.begin();
  for ( ; it != m_histmap.end(); ++it)
     delete it->second;
}

void Hist2DMap::reset()
{
  auto it = m_histmap.begin();
  for ( ; it != m_histmap.end(); ++it)
     it->second->reset();
}

void Hist2DMap::makeNumpyRaw(std::string &data, const std::vector<unsigned> &order)
{
  uint64_t dsize = m_xbin*m_ybin*order.size();
  std::vector<double> fulldata;
  fulldata.reserve(dsize);

  for(auto key:order)
  {
    auto it = m_histmap.find(key);
    if(it==m_histmap.end())
    {
      EW_LOG_WARN("key \"{}\" is not found in the map", key);
      fulldata.resize(fulldata.size()+m_xbin*m_ybin, 0.);
    }
    else
    {
      auto h = it->second->getRaw();
      fulldata.insert(fulldata.end(), h.begin(), h.end());
    }

  }

  RedisNumpy nvt;
  std::vector<uint64_t> shape;
  shape.push_back(dsize);
  nvt.makeNumpyArr(fulldata, RedisNumpy::data_type::f8,shape, data);

}

void Hist2DMap::makeNumpyXSum(std::string &data, const std::vector<unsigned> &order)
{
  uint64_t dsize = m_xbin*order.size();
  std::vector<double> fulldata;
  fulldata.reserve(dsize);

  for(auto key:order)
  {
    auto it = m_histmap.find(key);
    if(it==m_histmap.end())
    {
      EW_LOG_WARN("key \"{}\" is not found in the map", key);
      fulldata.resize(fulldata.size()+m_xbin, 0.);
    }
    else
    {
      std::vector<double> sumx;
      it->second->getXSum(sumx);
      fulldata.insert(fulldata.end(), sumx.begin(), sumx.end());
    }
  }

  RedisNumpy nvt;
  std::vector<uint64_t> shape;
  shape.push_back(dsize);

  nvt.makeNumpyArr(fulldata, RedisNumpy::data_type::f8,
                   shape, data);
}

void Hist2DMap::makeNumpyYSum(std::string &data, const std::vector<unsigned> &order)
{
  uint64_t dsize = m_ybin*order.size();
  std::vector<double> fulldata;
  fulldata.reserve(dsize);

  for(auto key:order)
  {
    auto it = m_histmap.find(key);
    if(it==m_histmap.end())
    {
      EW_LOG_WARN("key \"{}\" is not found in the map", key);
      fulldata.resize(fulldata.size()+m_ybin, 0.);
    }
    else
    {
      std::vector<double> sumy;
      it->second->getYSum(sumy);
      fulldata.insert(fulldata.end(), sumy.begin(), sumy.end());
    }
  }

  RedisNumpy nvt;
  std::vector<uint64_t> shape;
  shape.push_back(dsize);

  nvt.makeNumpyArr(fulldata, RedisNumpy::data_type::f8,
                   shape, data);


}

std::map<unsigned, NumpyHist2D* >::const_iterator Hist2DMap::getHistMapBegin() const
{
  return m_histmap.begin();
}

std::map<unsigned, NumpyHist2D* >::const_iterator Hist2DMap::getHistMapEnd() const
{
  return m_histmap.end();
}
