#include "DAQInterface.hh"
#include "Earthworm.hh"
#include "Hist2DMap.hh"

#include <iostream>
#include <algorithm>  // remove_if
#include <time.h>
#include <csignal>

DAQInterface::DAQInterface(std::string daq_name,
                           unsigned houseKeeping_itervl_ms)
: DimClient(), m_active(true), m_errhdl()
, m_daq_data(daq_name.c_str(),const_cast<char *>("dataAsCharArr"), this)
, m_flow(DataFlow{})
, m_houskpg_intvl_ms(houseKeeping_itervl_ms), m_start_time(std::chrono::system_clock::now())
, m_hist_map(nullptr)
{
  addErrorHandler(&m_errhdl);
  EW_LOG_INFO("Dim client connected to server {}",daq_name);
  // setExitHandler("RUN_INFO");
  m_hk_thread = std::thread(&DAQInterface::houseKeeping, this);
}

DAQInterface::~DAQInterface()
{
  if(m_hk_thread.joinable())
    m_hk_thread.detach();
}

void DAQInterface::setHist2DMap(Hist2DMap *map)
{
  std::lock_guard<std::mutex> guard(m_task_mutex);
  m_hist_map = std::shared_ptr<Hist2DMap>(map);
}

void DAQInterface::setActiveStatus(bool act)
{
  std::lock_guard<std::mutex> guard(m_task_mutex);
  m_active=act;
}

void DAQInterface::infoHandler()
{
  std::lock_guard<std::mutex> guard(m_task_mutex);
  // auto start = std::chrono::system_clock::now();
  if (m_active) { //fixme: see if this can stop receiving data?
    DimInfo *curr = getInfo(); // get current DimInfo address
    if(curr == &m_daq_data)
    {
      if(!m_flow.num_trigger) return; //infohandler will be triggered by constructor in dim v20
      unsigned dsize  = curr->getSize();
      m_flow.received_byte += dsize;

      uint8_t *data = static_cast<uint8_t*>(curr->getData());
      std::vector<uint8_t> buf(data, data+dsize);

      if(m_hist_map)
        m_tasks.emplace_back( DaqTask(data, data+dsize, m_hist_map));
      else
        m_tasks.emplace_back( DaqTask(data, data+dsize));
      m_tasks.back().setTime(curr->getTimestamp(), curr->getTimestampMillisecs());
      m_tasks.back().start();
    }
    else   {
      //fixme:log
    }
  }
  m_flow.num_trigger++;

}

void DAQInterface::report()
{
  std::cout <<  '\r' << "processed " << m_flow.processed_byte/1024/1024  << "MB. Received "
  << m_flow.processed_byte/1024/1024 << "MB. Processed " <<  m_flow.processed_byte*100./m_flow.received_byte <<"%. ";

  std::chrono::duration<double> elapsed_seconds = std::chrono::system_clock::now()-m_start_time;
  double t = elapsed_seconds.count();

  std::cout << "processed " << m_flow.processed_event  << " events. Event rate "
  <<  m_flow.processed_event/t <<" /s. ";

}

void DAQInterface::houseKeeping()
{
  EW_LOG_INFO("DAQInterface starting periodic housekeeping at {}ms interval",
  m_houskpg_intvl_ms.count());

  while (true) //fixme: implement timer on event instead of this dead loop
  {
    std::this_thread::sleep_for(m_houskpg_intvl_ms);
    if(m_tasks.empty())
      continue;

    std::lock_guard<std::mutex> guard(m_task_mutex);

    unsigned processed =0;

    auto newend = std::remove_if(m_tasks.begin(), m_tasks.end(),
    [&](DaqTask &atask ) -> bool
    {
      uint64_t processed = atask.readyAndValid();
      if(! processed )
        return false;
      else
      {
        //combine byte size and event count as a uint64_t
        int bytes = processed & 0xffffffff;
        int events = (processed >> 32) & 0xffffffff;
        m_flow.processed_byte += bytes;
        m_flow.processed_event += events;
        return true;
      }
    });

    if(newend == m_tasks.end() ) continue;

    unsigned deleted = m_tasks.end() - newend;
    m_flow.completed_task += deleted;
    m_tasks.erase(newend, m_tasks.end());


    report();
  }
}
