#include "PackageMWPC.hh"
#include <iostream>       // std::cout
#include <bitset>
#include "UtilsException.hh"
#include "Hist2DMap.hh"
#include "NumpyHist1D.hh"
#include <cmath>
#include "Earthworm.hh"

Parser::PackageMWPC::PackageMWPC(double threshold, double time_window)
:ParserMWPC(), m_charge_threshold(threshold), m_time_window(time_window), m_last_t0(0)
{
}

Parser::PackageMWPC::~PackageMWPC()
{
}

void Parser::PackageMWPC::process(std::vector<HitInfo>& hitinfo) const
{
    uint8_t chnl = 0;
    uint16_t Q = 0;
    unsigned chargeSum = 0;
    uint32_t hittof=m_tof_cq_map.begin()->first; //initialize the for loop
    auto last_item_in_map = --(m_tof_cq_map.end());
    uint32_t eventSz(0), accu_x(0), accu_y(0), accu_xw(0), accu_yw(0);

    for(auto it=m_tof_cq_map.begin();it!=m_tof_cq_map.end();++it)
    {
      getChannelCharge(it->second, chnl, Q);
      if(Q<m_charge_threshold) continue;

      // test if this hit is the new start of next event
      if(it->first>hittof + m_time_window || it==last_item_in_map)
      {
        //finish  accumulation and fill map
        if(it==last_item_in_map)
        {
          //this hit belongs to this event, this item must be accumulated in
          //this event
          eventSz++;
          chnl<51 ? accu_xw += chnl*Q : accu_yw += (chnl-50)*Q;
          chnl<51 ? accu_x += Q : accu_y += Q;
          chargeSum += Q;
        }


        if(accu_x*accu_y && eventSz>3)
        {

          // //fill charge hist
          // qhist.fill(chargeSum);
          // //fill pixel histogram
          float x = accu_xw/float(accu_x);
          float y = accu_yw/float(accu_y);
          // hitmap.fill( 0, hittof, std::round(x) + std::round(y)*50 );
          // printf("%g;%g;%g;%d;  %d\n", y*2.2, (50-x)*4., hittof/40.0, header.t0cnt-1, eventcnt-1 );
          // assert(chargeSum==accu_x+accu_y);
          hitinfo.push_back(HitInfo(   y*2.2, (50-x)*4. ,
          hittof/40. , chargeSum , m_header.t0cnt, 0));
          printf("%g;%g;%g;%d\n", y*2.2, (50-x)*4., hittof/40.0, m_header.t0cnt-1 );
          // abort();
        }

        if(it!=last_item_in_map)
        {
          //prepare next event
          hittof=it->first;
          chargeSum=Q;
          eventSz=1;
          accu_x=0;
          accu_y=0;
          accu_xw=0;
          accu_yw=0;

          chnl<51 ? accu_xw += chnl*Q : accu_yw += (chnl-50)*Q;
          chnl<51 ? accu_x += Q : accu_y += Q;
        }
      }
      else
      {
        //accumulate
        // printf("%d %d %d %d\n",eventcnt, it->first, chnl, Q);
        eventSz++;
        chnl<51 ? accu_xw += chnl*Q : accu_yw += (chnl-50)*Q;
        chnl<51 ? accu_x += Q : accu_y += Q;
        chargeSum += Q;
      }
    }

    std::multimap<uint32_t,uint32_t> temp; //release memory
    m_tof_cq_map.swap(temp);

}


bool Parser::PackageMWPC::readNextPackage(const uint8_t*& pt, const uint8_t * const stream_end,
  std::vector<HitInfo>& hitinfo) const
{
  hitinfo.clear();

  const uint8_t *package_end;
  uint64_t offset=0;
  if(!findNextPackage(pt, stream_end, package_end, offset))
    return false;

  //parsing raw data
  readHeader(pt, m_header);
  unsigned mapnum = m_header.map_num;
  //x chanel 50, spacing 4mm, y channel 94, spacing 2.2mm
  // std::bitset<50> x; //In standard C++, bits are initialized with zeros
  // std::bitset<94> y;

  if(!m_last_t0)
    m_last_t0 = m_header.t0cnt;

  if(blankData(pt, stream_end))
    printf("BLANK!\n");

  //if this is the start of a new t0 signal
  //process whatever in the map and get ready to restart the accumulation
  //Track reconstruction
  if((m_header.t0cnt!=m_last_t0 && !m_tof_cq_map.empty()) || blankData(pt, stream_end) )//if it is the beginning of the next frame, process data
  {

    process(hitinfo);
    m_last_t0=m_header.t0cnt;

  }

  //Sorting hit data
  // printf("*********%d maps************\n",header.map_num);
  uint32_t temp32=0;

  for(unsigned imap=0;imap<m_header.map_num;imap++)
  {
    error_assert(*pt == 0xAA);
    uint32_t t = 0;
    read_uint32(pt,t);
    t &= MASK_24;
    // printf("TdataID %d\n", t);
    uint32_t eventNum = 0;
    read_uint32(pt,eventNum);

    // printf("map %d, event number %d \n",imap, eventNum);
    for(unsigned ievt=0;ievt<eventNum;ievt++)
    {
      uint8_t ch = *(pt);
      read_uint32(pt, temp32);
      //error_assert(ch==*(pt));
      uint32_t t_hi_res = temp32&MASK_24;
      read_uint32(pt, temp32);
      //fixme: there is a bug in DAQ, the secend channel num sometimes are left zero
      temp32 = (temp32&MASK_24) + (ch<<24);
      uint16_t charge = (temp32&MASK_12)- ((temp32>>12)&MASK_12);
      m_tof_cq_map.emplace_hint(m_tof_cq_map.end(), t_hi_res, temp32);

      // printf("ch %d, charge %d, time stamp %d\n", ch, charge, t_hi_res);
    }
  }
  pt = package_end;

  //printf("left %ld, package_end %p, stream_end %p\n", stream_end-package_end, package_end, stream_end);
  return package_end==stream_end?false:true;

}

bool Parser::PackageMWPC::readNextPackage(const uint8_t*& pt, const uint8_t * const stream_end,
  Hist2DMap &hitmap, NumpyHist1D &qhist) const
{
  unsigned eventcnt = 0;

  const uint8_t *package_end;
  uint64_t offset=0;
  if(!findNextPackage(pt, stream_end, package_end, offset))
    return false;

  //parsing raw data
  ParserMWPC::MWPCHeader header;
  readHeader(pt, header);
  unsigned mapnum = header.map_num;
  //x chanel 50, spacing 4mm, y channel 94, spacing 2.2mm
  // std::bitset<50> x; //In standard C++, bits are initialized with zeros
  // std::bitset<94> y;

  if(!m_last_t0)
    m_last_t0 = header.t0cnt;

  //if this is the start of a new t0 signal
  //process whatever in the map and get ready to restart the accumulation
  if(header.t0cnt!=m_last_t0 && !m_tof_cq_map.empty())//if it is the beginning of the next frame, process data
  {
    // print();
    uint8_t chnl = 0;
    uint16_t Q = 0;
    unsigned chargeSum = 0;
    uint32_t hittof=m_tof_cq_map.begin()->first; //initialize the for loop
    auto last_item_in_map = --(m_tof_cq_map.end());
    uint32_t eventSz(0), accu_x(0), accu_y(0), accu_xw(0), accu_yw(0);

    for(auto it=m_tof_cq_map.begin();it!=m_tof_cq_map.end();++it)
    {
      getChannelCharge(it->second, chnl, Q);
      if(Q<m_charge_threshold) continue;

      // test if this hit is the new start of next event
      if(it->first>hittof + m_time_window || it==last_item_in_map)
      {
        //finish  accumulation and fill map
        if(it==last_item_in_map)
        {
          //this hit belongs to this event, this item must be accumulated in
          //this event
          eventSz++;
          chnl<51 ? accu_xw += chnl*Q : accu_yw += (chnl-50)*Q;
          chnl<51 ? accu_x += Q : accu_y += Q;
          chargeSum += Q;
        }

        // printf("%d %d %d %d\n",eventcnt, it->first, chnl, Q);
        eventcnt++;

        if(accu_x*accu_y && eventSz>3)
        {
          //fill charge hist
          qhist.fill(chargeSum);
          //fill pixel histogram
          float x = accu_xw/float(accu_x);
          float y = accu_yw/float(accu_y);
          hitmap.fill( 0, hittof, std::round(x) + std::round(y)*50 );
          printf("%g;%g;%g;%d;  %d\n", y*2.2, (50-x)*4., hittof/40.0, header.t0cnt-1, eventcnt-1 );
          assert(chargeSum==accu_x+accu_y);

        }

        if(it!=last_item_in_map)
        {
          //prepare next event
          hittof=it->first;
          chargeSum=Q;
          eventSz=1;
          accu_x=0;
          accu_y=0;
          accu_xw=0;
          accu_yw=0;

          chnl<51 ? accu_xw += chnl*Q : accu_yw += (chnl-50)*Q;
          chnl<51 ? accu_x += Q : accu_y += Q;
        }
      }
      else
      {
        //accumulate
        // printf("%d %d %d %d\n",eventcnt, it->first, chnl, Q);
        eventSz++;
        chnl<51 ? accu_xw += chnl*Q : accu_yw += (chnl-50)*Q;
        chnl<51 ? accu_x += Q : accu_y += Q;
        chargeSum += Q;
      }
    }

    std::multimap<uint32_t,uint32_t> temp; //release memory
    m_tof_cq_map.swap(temp);
    m_last_t0=header.t0cnt;
  }

  //Sorting hit data
  // printf("*********%d maps************\n",header.map_num);
  uint32_t temp32=0;

  for(unsigned imap=0;imap<header.map_num;imap++)
  {
    error_assert(*pt == 0xAA);
    uint32_t t = 0;
    read_uint32(pt,t);
    t &= MASK_24;
    // printf("TdataID %d\n", t);
    uint32_t eventNum = 0;
    read_uint32(pt,eventNum);

    // printf("map %d, event number %d \n",imap, eventNum);
    for(unsigned ievt=0;ievt<eventNum;ievt++)
    {
      uint8_t ch = *(pt);
      read_uint32(pt, temp32);
      //error_assert(ch==*(pt));
      uint32_t t_hi_res = temp32&MASK_24;
      read_uint32(pt, temp32);
      //fixme: there is a bug in DAQ, the secend channel num sometimes are left zero
      temp32 = (temp32&MASK_24) + (ch<<24);
      // uint16_t charge = (temp32&MASK_12)- ((temp32>>12)&MASK_12);
      m_tof_cq_map.emplace_hint(m_tof_cq_map.end(), t_hi_res, temp32);

      // printf("ch %d, charge %d, time stamp %d\n", ch, charge, t_hi_res);
    }
  }
  pt = package_end;

  //printf("left %ld, package_end %p, stream_end %p\n", stream_end-package_end, package_end, stream_end);
  return package_end==stream_end?false:true;
}

void Parser::PackageMWPC::write(std::ofstream &file, const char *filename)
{
  file.open(filename);
  auto it=m_tof_cq_map.begin();
  for(;it!=m_tof_cq_map.end();++it)
  {
    uint32_t temp32 = it->second;
    file << it->first << " " << ( temp32>>24) << " "  <<( (temp32&MASK_12) - ( (temp32>>12)&MASK_12) )<< "\n";
  }
  file.close();
}

void Parser::PackageMWPC::print() const
{
  uint8_t ch = 0;
  uint16_t Q = 0;
  for(auto it: m_tof_cq_map)
  {
    getChannelCharge(it.second, ch, Q);
    std::cout << it.first << " " << (uint16_t)ch << " " << Q << std::endl;
  }
  std::cout << std::endl;
}
