#include "StateMachine.hh"
#include <stdexcept>      // std::invalid_argument
#include <algorithm>
#include <iostream>
// #include "Earthworm.hh"

StateMachine::StateMachine()
:m_numState(7), m_numCommand(7), m_state(StateMachine::State::kUnconfigured)
,m_allowedStateTransit(m_numState*m_numState, -1) //don't allow transit between 7 state
// ,m_funcpt(m_numState*m_numState, nullptr)
{

}

StateMachine::StateMachine(int instrument)
:m_numState(7), m_numCommand(7), m_state(StateMachine::State::kUnconfigured)
,m_allowedStateTransit(StateMachine::m_numState*StateMachine::m_numState, -1) //don't allow transit between 7 state
// ,m_funcpt(m_numState*m_numState, nullptr)
{
  if(instrument==1)
  {
    //in total 9 possible path of state transit
    grantStateTransit(State::kUnconfigured, State::kReady, Command::kConfigure);
    grantStateTransit(State::kReady, State::kUnconfigured, Command::kUnconfigure);

    grantStateTransit(State::kReady, State::kRunning, Command::kStart);
    grantStateTransit(State::kRunning, State::kReady, Command::kStop);

    grantStateTransit(State::kRunning, State::kPaused, Command::kPause);
    grantStateTransit(State::kPaused, State::kRunning, Command::kResume);

    grantStateTransit(State::kReady, State::kError, Command::kAbort);
    grantStateTransit(State::kRunning, State::kError, Command::kAbort);
    grantStateTransit(State::kPaused, State::kError, Command::kAbort);
  }
}

StateMachine::~StateMachine()
{

}


bool StateMachine::tableSanityCheck(State from)
{
  std::vector<int> premitted;
  premitted.reserve(m_numState);
  for(unsigned i=0;i<m_numState;i++)
  {
    int possible_cmd = m_allowedStateTransit[from*m_numState+i];
    if(possible_cmd>=0)
    {
      premitted.push_back(possible_cmd);
    }
  }

  std::sort(premitted.begin(), premitted.end());
  auto last = std::unique(premitted.begin(), premitted.end());
  if(last==premitted.end())
    return true;
  else
    return false;
}

std::string StateMachine::printState() const
{
  return kStateStrMap.find(m_state)->second;
}

std::string StateMachine::printState(State s) const
{
  if(s>=m_numState)
    throw std::invalid_argument("printState input must be smaller than the number of state");

  return kStateStrMap.find(s)->second;
}

StateMachine::Command StateMachine::int2cmd(int i) const
{
  if(i>=m_numState)
    throw std::invalid_argument("int2cmd input must smaller be than the number of state");

  return static_cast<Command> (i);
}

StateMachine::State StateMachine::int2state(int i) const
{
  if(i>=m_numState)
    throw std::invalid_argument("int2state input must be smaller than the number of state");

  return static_cast<State> (i);
}

void StateMachine::grantStateTransit(State from, State to, Command byCommand)
{
  m_allowedStateTransit[from*m_numState + to] = byCommand;
  if(!tableSanityCheck(from))
  {
    std::cerr << "Granting transit from state " << from
          << " to state " << to << " by command " << byCommand << '\n';
    throw std::invalid_argument("The same command sends a state to different state");
  }
}

void StateMachine::denyStateTransit(State from, State to)
{
  m_allowedStateTransit[from*m_numState + to] = -1;
}

StateMachine::State StateMachine::transit(State from, int byCommand)
{
  StateMachine::State s = from;
  // if( byCommand<0 && byCommand>=m_numState) //invalid command
  // {
  //   if(m_state==State::kUnconfigured)
  //     return s;
  //   else
  //   {
  //     std::cerr << "Invalid command "<< byCommand
  //               << " received when m_state is"
  //               << printState() << '\n';
  //     throw std::invalid_argument("");
  //   }
  // }
  // else //valid command
  if( byCommand>=0 && byCommand<m_numState)
  {
    for(unsigned i=0;i<m_numState;i++)
    {

      if( m_allowedStateTransit[from*m_numState + i] == byCommand)
      {
        // i is gerentted to be smaller than the number of state
        s = static_cast<State> (i);
        break;
      }
    }
  }
  // EW_LOG_INFO("StateMachine state is now {} by command {}", kStateStrMap.find(s)->second, byCommand );
  return s;
}

StateMachine::State StateMachine::transit(int by)
{
    m_state = transit(m_state, by);
    return m_state;
}


#include <random>
void StateMachine::selfDiagnosis()
{
  std::random_device rd;
  std::mt19937 gen(rd());
  std::uniform_int_distribution<> dis(0, 7); //generate 7 as well!

  std::vector<std::string> history;

  std::cout << "Self-diagnosis:: generating 10k random state and commands \n";
  for (int n=0; n<10000; ++n)
  {
    int rState(dis(gen)), rCmd(dis(gen));
    if(rState>=m_numState)
      continue;
    State oldstate = int2state(rState);

    State newstate = transit(oldstate, rCmd);

    if(oldstate!=newstate)
    {
      std::string record = kStateStrMap.find(oldstate)->second + "("
               + (rState<7 ? kCommandStrMap.find(int2cmd(rCmd))->second : std::to_string(rCmd))
               + ")"
               + kStateStrMap.find(newstate)->second ;

      if (std::find(history.begin(), history.end(), record) == history.end())
      {
        // Element not in vector.
        history.push_back(record);
      }
    }
  }

  std::cout << "Found " << history.size() << " valid transit" << '\n';
  for (auto const & i : history)
    std::cout << i << " \n";
  std::cout << "\n";
}
