#include "MyStateMachine.hh"
#include <stdexcept>      // std::invalid_argument
#include <algorithm>
#include <iostream>
#include "Earthworm.hh"

const static unsigned kNumState = 7;


MyStateMachine::MyStateMachine()
:m_numState(kNumState), m_state(State::kUnconfigured)
,m_allowedStateTransit(m_numState*m_numState, -1) //don't allow transit between 7 state
,m_funcpt(m_numState*m_numState, nullptr)
,m_funcName(m_numState*m_numState, "")
,m_state2strMap ( {
{State::kWaiting, "waiting"},
{State::kUnconfigured, "unconfigured"},
{State::kConfiguring, "configuring"},
{State::kReady, "ready"},
{State::kRunning, "running"},
{State::kPaused, "paused"},
{State::kError, "error"}
})
{}


MyStateMachine::~MyStateMachine()
{}



bool MyStateMachine::tableSanityCheck(State from) const
{
  std::vector<int> premitted;
  premitted.reserve(m_numState);
  for(unsigned i=0;i<m_numState;i++)
  {
    int possible_cmd = m_allowedStateTransit[from*m_numState+i];
    if(possible_cmd>=0)
    {
      premitted.push_back(possible_cmd);
    }
  }

  std::sort(premitted.begin(), premitted.end());
  auto last = std::unique(premitted.begin(), premitted.end());
  if(last==premitted.end())
    return true;
  else
    return false;
}



bool MyStateMachine::validateCommand(int cmd)
{
  if((cmd>=0 && cmd<7) || cmd==Command::kKill)
    return true;
  else
    return false;
}

//return true if
bool MyStateMachine::transit(int intCommand)
{
  if( validateCommand(intCommand))
  {
    for(unsigned i=0;i<m_numState;i++)
    {
      unsigned index = m_state*m_numState + i;
      if( m_allowedStateTransit[index] == intCommand)
      {
        // i is gerentted to be smaller than the number of state
        m_state = static_cast<State>(i);
        if(m_funcpt[index])
          m_funcpt[index]();

        EW_LOG_INFO("received command {}", intCommand);
        return true;
      }
    }
  }
  return false;
}


void MyStateMachine::grantGlobalTransitTo(State to, Command by, const std::function<bool()> &executeFunc, const std::string &name)
{
  for(int from=kWaiting;from<=kError;from++)
  {
    grantStateTransit( static_cast<State>(from), to, by, executeFunc, name);
  }

}


void MyStateMachine::grantStateTransit(State from, State to, Command byCommand)
{
  grantStateTransit(from, to, byCommand, nullptr, "");
}


void MyStateMachine::grantStateTransit(State from, State to, Command byCommand, const std::function<bool()> &executeFunc, const std::string &name)
{
  unsigned index=getTableIndex(from, to);
  m_allowedStateTransit[index] = byCommand;
  if(!tableSanityCheck(from))
  {
    std::cerr << "Granting transit from state " << from
          << " to state " << to << " by command " << byCommand << '\n';
    throw std::invalid_argument("The same command sends a state to different states");
  }

  if(!m_funcpt[index])
  {
    if(executeFunc!=nullptr)
    {
      m_funcpt[index]=executeFunc;
      m_funcName[index]=name;
      EW_LOG_INFO("Granted transit from {} to {} by command {} and executing function {}", State2Str(from), State2Str(to), (byCommand), name );
    }
    else
    {
      EW_LOG_INFO("Granted transit from {} to {} by command {}", State2Str(from), State2Str(to), (byCommand) );
    }
  }
  else
  {
    EW_LOG_WARN("Request transit from {} to {} by command {} is denied. State table is untouched.", State2Str(from), State2Str(to), (byCommand) );
  }
}


std::string MyStateMachine::printState() const
{
  return State2Str(m_state);
}


void MyStateMachine::denyStateTransit(State from, State to)
{
  unsigned index=getTableIndex(from, to);
  m_allowedStateTransit[index] = -1;
  m_funcpt[index]=nullptr;
}


unsigned MyStateMachine::getTableIndex(State from, State to) const
{
  return from*m_numState + to;
}


std::string MyStateMachine::State2Str(State s) const
{
  auto pair = m_state2strMap.find(s);
  return pair==m_state2strMap.end()? "IMPOSSIBLE_STATE": pair->second;
}


void MyStateMachine::selfDiagnosis() const
{
  std::vector<std::string> history;

  std::cout << "Self-diagnosis:";

  std::cout << "Transit table:" << '\n';
  for(unsigned i=0;i<m_numState*m_numState;i++)
  {
    std::cout << m_allowedStateTransit[i] << ' ';
    if(i%7==0 && i>0)
      std::cout <<  '\n';
  }
  std::cout <<  '\n';

  for(int from=kWaiting;from<=kError;from++)
  {
    for(int to=kWaiting;to<=kError;to++)
    {
      unsigned index=getTableIndex(static_cast<State>(from), static_cast<State>(to));
      if(m_allowedStateTransit[index]>=0)
      {

        std::string record = State2Str(static_cast<State>(from))+" -> ("+
            std::to_string(m_allowedStateTransit[index])+") -> "+State2Str(static_cast<State>(to));
        if(m_funcpt[index])
          record += " executing function " + m_funcName[index];
        else
          record += " without attached function";
        history.push_back(record);
      }
    }
  }
  std::cout << "Found " << history.size() << " valid transit " << '\n';
  for (auto const & i : history)
    std::cout << i << " \n";
  std::cout << "\n";

}
