#include "DaqTask.hh"
#include <thread>
#include <chrono>
#include <iostream>
#include "PackageHe3.hh"
#include "PackageMWPC.hh"
#include "Earthworm.hh"
#include "PMTParser.hh"

DaqTask::DaqTask(uint8_t* begin, uint8_t* end, const std::shared_ptr<Hist2DMap>& map)
:m_data(begin, end), m_time_s(0), m_time_ms(0)
, m_hist_map(std::shared_ptr<Hist2DMap>(map)), m_hist2d(nullptr)
{

}

DaqTask::DaqTask(uint8_t* begin, uint8_t* end, const std::shared_ptr<NumpyHist2D>& hist)
:m_data(begin, end), m_time_s(0), m_time_ms(0)
, m_hist_map(nullptr), m_hist2d(std::shared_ptr<NumpyHist2D>(hist))
{
}

DaqTask::DaqTask(uint8_t* begin, uint8_t* end)
:m_data(begin, end), m_time_s(0), m_time_ms(0),m_hist_map(nullptr), m_hist2d(nullptr)
{

}

DaqTask::~DaqTask()
{

}


uint64_t DaqTask::readyAndValid() const
{
  if(!m_future.valid())
    return 0;
  auto status = m_future.wait_for(std::chrono::milliseconds(0));
  return status == std::future_status::ready ? m_future.get() : 0;
}

void DaqTask::start()
{
  std::lock_guard<std::mutex> guard(m_run_mutex);
  m_future = std::async(std::launch::async, &DaqTask::run, this).share();
}

uint64_t DaqTask::run()
{
  std::lock_guard<std::mutex> guard(m_run_mutex);

  const uint8_t * abuf = m_data.data();
  int event_cnt=0;
  std::vector<Parser::HitInfo> hitinfo;

  // if(!m_hist_map)
  // {
  //   EW_LOG_WARN("Histmap is not set in Daqtask");
  //   return 0;
  // }

  if(m_hist2d)
  {
    double digit2us = 1./40;
    auto *pmt = new Parser::PMTParser();

    while(pmt->extractNextPackage(abuf, m_data.data()+m_data.size(), hitinfo))
    {
      event_cnt+=hitinfo.size();
      for(auto v: hitinfo)
      {
        // unsigned pid = pmt->getPixelID((int)v.pos, , false);
        // m_hist2d->fill( (int)(v.t/40), pid);

        unsigned pid = 7*(int)v.posy + (int)(v.pos/8);
        m_hist2d->fill( (int)(v.t/40), pid );
        // printf("%d;%d;%d\n", (int)pid, (int)(v.t)/40, (int)v.hit_t0 );
      }
    }
  }
  else if(m_hist_map->getDetType()==Hist2DMap::DetectorType::kHe3)
  {
    Parser::PackageHe3 he3(0.);

    while(he3.extractNextPackage(abuf, m_data.data()+m_data.size(), hitinfo))
    {
      unsigned pkgevt = hitinfo.size();
      event_cnt += pkgevt;

      if(pkgevt){
        if(m_hist_map)
        {
          for(auto const& hit:hitinfo)
          {
            m_hist_map->fill(hit.id, hit.t, hit.pos);
          }
        }
      }
    }
  }
  else if (m_hist_map->getDetType()==Hist2DMap::DetectorType::kMWPC)
  {
    Parser::PackageMWPC mwpc(0., 10.);//chargeThre, timewindow

    double invx=1/2.2;
    double invy=1/4.0;

    while(mwpc.extractNextPackage(abuf, m_data.data()+m_data.size(), hitinfo))
    {
      unsigned pkgevt = hitinfo.size();
      event_cnt += pkgevt;

      if(pkgevt){
        if(m_hist_map)
        {
          for(auto const& hit:hitinfo)
          {
            int x = hit.pos*invx;
            int y = hit.posy*invy;
            // std::cout << x+y*92 << '\n';
            m_hist_map->fill(1 , hit.t, x+y*92 );
          }
        }
      }
    }
    // //FIXME: work around for single event package
    // hitinfo.clear();
    // mwpc.process(hitinfo);
    // if(m_hist_map)
    // {
    //   for(auto const& hit:hitinfo)
    //   {
    //     int x = hit.pos*invx;
    //     int y = hit.posy*invy;
    //     // std::cout << x+y*92 << '\n';
    //     m_hist_map->fill(1 , hit.t, x+y*92 );
    //   }
    // }
  }
  else
    EW_LOG_CRITICAL("DaqTask::run Unsupported detector type");



  // if(pkgevt)
  // {
  //   char fn[200];
  //   sprintf(fn,"data/%d_%d.dat",m_time_s, m_time_ms);
  //   FILE *datafile = fopen(fn, "wb");
  //   fwrite (m_data.data() , sizeof(char), m_data.size(), datafile);
  //   fclose(datafile);
  //   // printf("%s, %d events\n", fn, event_cnt);
  // }
  uint64_t shiftedEvt = event_cnt*4294967296;  //shifting 32 bits to the higher order
  return (uint64_t)m_data.size() +  shiftedEvt;
}

void DaqTask::setTime(int second, int ms)
{
  m_time_s = second;
  m_time_ms = ms;
}
void DaqTask::getTime(int &second, int &ms) const
{
  second = m_time_s;
  ms = m_time_ms;
}

// Move initialization
DaqTask::DaqTask(DaqTask&& other) {
  std::lock_guard<std::mutex> lock(other.m_run_mutex);
  m_data = std::move(other.m_data);
  m_future = std::move(other.m_future);
  m_hist_map = std::move(other.m_hist_map);
  m_hist2d = std::move(other.m_hist2d);
  m_time_s = other.m_time_s;
  m_time_ms = other.m_time_ms;
  other.m_time_s = 0;
  other.m_time_ms = 0;
};

// // Copy initialization
// DaqTask::DaqTask(const DaqTask& other) {
//   std::lock_guard<std::mutex> lock(other.m_run_mutex);
//   m_data = other.m_data;
//   m_future = other.m_future;
//   m_hist_map = other.m_hist_map;
//   m_time_s = other.m_time_s;
//   m_time_ms = other.m_time_ms;
// };

// Move assignment
DaqTask& DaqTask::operator = (DaqTask&& other) {
  std::lock(m_run_mutex, other.m_run_mutex);
  std::lock_guard<std::mutex> self_lock(m_run_mutex, std::adopt_lock);
  std::lock_guard<std::mutex> other_lock(other.m_run_mutex, std::adopt_lock);
  m_data = std::move(other.m_data);
  m_future = std::move(other.m_future);
  m_hist_map = std::move(other.m_hist_map);
  m_hist2d = std::move(other.m_hist2d);
  m_time_s = other.m_time_s;
  m_time_ms = other.m_time_ms;
  other.m_time_s = 0;
  other.m_time_ms = 0;
  return *this;
};
//
// // Copy assignment
// DaqTask& DaqTask::operator = (const DaqTask& other) {
//   std::lock(m_run_mutex, other.m_run_mutex);
//   std::lock_guard<std::mutex> self_lock(m_run_mutex, std::adopt_lock);
//   std::lock_guard<std::mutex> other_lock(other.m_run_mutex, std::adopt_lock);
//   m_data = other.m_data;
//   m_future = other.m_future;
//   m_hist_map = other.m_hist_map;
//   m_time_s = other.m_time_s;
//   m_time_ms = other.m_time_ms;
//   return *this;
// };
