
#include <dic.hxx>
#include <iostream>
#include <vector>
#include <string>
#include <fstream>
#include <memory>
#include <chrono>
#include <future>
#include <nlohmann/json.hpp>

#include "Action_MR.hh"
#include "Earthworm.hh"


Action_MR::Action_MR(RedisIO *redis, MyStateMachine *sMachine)
: Action(redis),  m_detType(Hist2DMap::DetectorType::kUnknown),
  m_histmap( nullptr),
	m_sMachine(sMachine),
	m_daq_he3(nullptr), m_daq_mwpc(nullptr) , m_active_daq(nullptr)
{
  std::ifstream f("../allPath.json");
  m_allPath.clear();
  f >> m_allPath;
  f.close();

  m_daq_he3 =	new DAQInterface(m_allPath["dimserverHe3"],1000);
	m_daq_mwpc =	new DAQInterface(m_allPath["dimserverMWPC"],1000);
}

Action_MR::~Action_MR()
{
	delete m_daq_he3;
  delete m_daq_mwpc;

	EW_LOG_INFO("~Action_MR ref count of histmap is {}", m_histmap.use_count());
	m_histmap.reset();
}

bool Action_MR::configure()
{
	//fixme: update m_allPath fisrt !!!!!

	std::ifstream f("../allPath.json");
	m_allPath.clear();
	f >> m_allPath;
  f.close();

	std::string conf = m_redis->readString(m_allPath["configure"]);//
	auto confjson =	nlohmann::json::parse(conf);
	std::string dt= confjson["mode"];

	if(dt=="He3")
	{
		m_active_daq = m_daq_he3;
		m_detType=Hist2DMap::DetectorType::kHe3;
		unsigned pid_num(300), tof_num(2500);
		m_histmap.reset(new Hist2DMap(tof_num, 0 , 40000, pid_num, 0, pid_num ,
      Hist2DMap::DetectorType::kHe3,4));
		m_active_daq->setHist2DMap(m_histmap);


		std::vector<double> pid;
		for(unsigned i=0;i<pid_num*4;i++)
			pid.push_back(i+1);
		RedisNumpy nvt;
		std::string pidstr;
		std::vector<uint64_t> shape;
		shape.push_back(pid_num*4);
		nvt.makeNumpyArr(pid, RedisNumpy::data_type::f8,
										 shape, pidstr);
		m_redis->writeString(m_allPath["rawDataPixel"], pidstr);

		std::vector<double> tof;
		for(unsigned i=0;i<tof_num+1;i++)
		{
			tof.push_back(i*16);
		}
		std::string tofstr;
		shape[0]=(tof_num+1);
		nvt.makeNumpyArr(tof, RedisNumpy::data_type::f8,
										 shape, tofstr);
		m_redis->writeString(m_allPath["rawDataTof"], tofstr);

    //hist
    std::vector<double> h(pid_num*tof_num*4, 0.);
    std::string histstr;
    shape[0] = pid_num*tof_num*4;
    nvt.makeNumpyArr(h, RedisNumpy::data_type::f8,
                     shape, histstr);
    m_redis->writeString(m_allPath["rawDataHist"], histstr);
	 }
	else if(dt=="MWPC")
	{
		m_active_daq = m_daq_mwpc;
		unsigned pid_num(4600), tof_num(2500);
		m_detType=Hist2DMap::DetectorType::kMWPC;
		m_histmap.reset(new Hist2DMap(tof_num, 0 , 40000, pid_num, -.5, pid_num-.5 ,
      Hist2DMap::DetectorType::kMWPC,1));
		m_active_daq->setHist2DMap(m_histmap);

    //pid
		std::vector<double> pid;
		for(unsigned i=0;i<pid_num;i++)
			pid.push_back(i+1);
		RedisNumpy nvt;
		std::string pidstr;
		std::vector<uint64_t> shape;
		shape.push_back(pid_num);
		nvt.makeNumpyArr(pid, RedisNumpy::data_type::f8,
										 shape, pidstr);
		m_redis->writeString(m_allPath["rawDataPixel"], pidstr);

    //hist
    std::vector<double> h(pid_num*tof_num, 0.);
    std::string histstr;
    shape[0] = pid_num*tof_num;
    nvt.makeNumpyArr(h, RedisNumpy::data_type::f8,
										 shape, histstr);
    m_redis->writeString(m_allPath["rawDataHist"], histstr);

    //tof
		std::vector<double> tof;
		for(unsigned i=0;i<tof_num+1;i++)
		{
			tof.push_back(i*16);
		}
		std::string tofstr;
		shape[0]=(tof_num+1);
		nvt.makeNumpyArr(tof, RedisNumpy::data_type::f8,
										 shape, tofstr);
		m_redis->writeString(m_allPath["rawDataTof"], tofstr);
	}
  else
    return false;

	m_histmap->reset();
  m_active_daq->reset();

	EW_LOG_INFO("configure:: Configuration completed. Instrument active detector is {}.", dt);
	return true;
}

//Stop running and clear any runtime determined parameter
bool Action_MR::unconfigure()
{
	m_active_daq->setActive(false);
	m_active_daq = nullptr;
	m_detType = Hist2DMap::DetectorType::kUnknown;
	EW_LOG_INFO("unconfigure:: Earthworm unconfigured.");
	return true;
}

//Start accumulate data, RESET AGAIN HERE
bool Action_MR::start()
{
  unsigned dsize = 4600*2500;
  std::vector<double> h(dsize, 0.);
  std::string histstr;
  std::vector<uint64_t> shape;
  shape.push_back(dsize);
  RedisNumpy nvt;
  nvt.makeNumpyArr(h, RedisNumpy::data_type::f8,
                   shape, histstr);
  m_redis->writeString(m_allPath["rawDataHist"], histstr);


	m_active_daq->setActive(false);
	m_histmap->reset();
	m_active_daq->setActive(true);
  m_active_daq->reset();

	m_startTime = std::chrono::steady_clock::now();
  EW_LOG_INFO("start:: Run started.");

	return true;
}

//Paused fill histogram, still receive but unprocess dim datastream
bool Action_MR::pauseStop()
{
	m_active_daq->setActive(false);
	EW_LOG_INFO("pauseStop:: kPause or kStop.");
	return true;
}

bool Action_MR::abort()
{
	m_active_daq->setActive(false);
	EW_LOG_INFO("abort:: aborting");
	return true;
}

//Restart running
bool Action_MR::resume()
{
	m_active_daq->setActive(true);
	EW_LOG_INFO("resume:: kResume.");
	return true;
}

bool Action_MR::kill()
{
	if(m_active_daq)
		m_active_daq->setActive(false);
	m_kill=true;
	return true;
}


void Action_MR::sendHeartbeat(bool stateAsStr)
{
  nlohmann::json j;
  j["pid"] =std::to_string(m_proID);
	if(stateAsStr)
	  j["status"] = m_sMachine->printState();
  else
		j["status"] = std::to_string(m_sMachine->getState());

  std::time_t t = std::time(nullptr);
  char mbstr[100];
  if (std::strftime(mbstr, sizeof(mbstr), "%Y-%m-%dT%H:%M:%S.%03d+08:00", std::localtime(&t))) {
  }

  j["timestamp"] = std::string (mbstr);
  std::string hb= j.dump();
	std::cout << "Heartbeart " << hb  << '\n';

  m_redis->writeString(m_allPath["hbEarthworm"],hb);

  if(m_active_daq)
    m_redis->writeString(m_allPath["pulstCnt"],std::to_string(m_active_daq->getDataFlow().num_dim_trigger));

}

bool Action_MR::update()
{
	std::string str = m_redis->readString(m_allPath["ctrlEarthworm"]);
  return m_sMachine->transit(atoi(str.c_str()));
}

bool Action_MR::keepRunning()
{
	EW_LOG_INFO("keepRunning:: Keep running");

	std::string redisDataStr;
	double totcnt = m_histmap->getIntegral();
	double cnt_rate=0.;
	if(totcnt)
  {
		cnt_rate=totcnt/std::chrono::duration_cast<std::chrono::milliseconds>(std::chrono::steady_clock::now() - m_startTime).count()*1000;
		EW_LOG_INFO("keepRunning:: Data integral {}, rate is {} event/s", totcnt,
							cnt_rate);
  }
	else
		EW_LOG_INFO("keepRunning:: Data integral is 0");

	if( m_detType==Hist2DMap::DetectorType::kHe3)
	{
		m_histmap->makeNumpyRaw(redisDataStr);
		m_redis->writeString(m_allPath["he3DetCnt"], std::to_string((int) m_histmap->getIntegral()) );
		m_redis->writeString(m_allPath["he3DetCntRate"], std::to_string(cnt_rate) );
	}
	else if( m_detType==Hist2DMap::DetectorType::kMWPC)
	{
		m_histmap->makeNumpyRaw(redisDataStr,std::vector<unsigned>{1});
		m_redis->writeString(m_allPath["mwpcDetCnt"],std::to_string((int) m_histmap->getIntegral()));
		m_redis->writeString(m_allPath["mwpcDetCntRate"],std::to_string((int) m_histmap->getIntegral()));
  }
	else
		EW_LOG_CRITICAL("keepRunning:: Detector type is unconfigured!");

  m_redis->writeString(m_allPath["rawDataHist"], redisDataStr);
	return true;
}
