#ifndef StateMachine_hh
#define StateMachine_hh

#include <string>
#include <vector>
#include <map>
#include <atomic>
#include <functional>

//This class models a state machine of m_numState state.
//Transit is initiated by received commands.

//Internally,this class contains a m_numState by m_numState sized matrix-like
//data, which records the premitted transit from state A to B triggered by a
//command X

enum State {
	kWaiting,
	kUnconfigured,
	kConfiguring,
	kReady,
	kRunning,
	kPaused,
	kError
};


enum Command {
	kIneffective = -1,
	kConfigure, //0
	kUnconfigure,
	kStart, //2
	kPause,
	kResume,//4
	kStop,
	kAbort,
	kKill=6402
};

const std::map<Command, std::string> kCommandStrMap = {
{Command::kConfigure, "configure"},
{Command::kUnconfigure, "unconfigure"},
{Command::kStart, "start"},
{Command::kPause, "pause"},
{Command::kResume, "resume"},
{Command::kStop, "stop"},
{Command::kAbort, "abort"}
};

class StateMachine {
public:
  StateMachine();
  virtual ~StateMachine();

	void selfDiagnosis() const;
	std::string printState() const;
	State getState() const {return m_state;}

	void grantStateTransit(State from, State to, Command by);
	void grantStateTransit(State from, State to, Command by, const std::function<bool()> &executeFunc, const std::string &name);
	void denyStateTransit(State from, State to);

  void grantGlobalTransitTo(State to, Command by, const std::function<bool()> &executeFunc, const std::string &name);
	bool transit(int intCommand);

private:
	const unsigned m_numState;
	std::atomic<State> m_state;
	std::vector<std::vector<int>> m_allowedStateTransit;
	std::vector<std::function<bool()>> m_funcpt;
	std::vector<std::string> m_funcName;
	const std::map<State, std::string> m_state2strMap;

	//make sure a state can't accept duplicated commmand
	bool tableSanityCheck(State from) const;
	unsigned getTableIndex(State from, State to) const;
	std::string State2Str(State s) const;
	bool validateCommand(int cmd);
};

#endif
