#!/usr/bin/env python3

import struct
import redis
import numpy as np

def toRedis(r,n):
   """Send Numpy array 'n' to Redis and return unique key for retrieval"""
   h, w = n.shape
   shape = struct.pack('>II',h,w)
   encoded = shape + n.tobytes()

   key = r.incr('unique')
   r.set(key,encoded)
   return key

def fromRedis(r,unique):
   """Given a unique key, retrieve Numpy array from Redis"""
   str = r.get(unique)
   h, w = struct.unpack('>II',str[:8])
   n = np.frombuffer(str, dtype=np.uint16, offset=8).reshape(h,w)
   return n

# Create 80x80 numpy array to store
a0 = np.arange(6400,dtype=np.uint16).reshape(80,80)

# Redis connection
r = redis.Redis(host='localhost', port=6379, db=0)

# Store in Redis
unique = toRedis(r,a0)

# Retrieve from Redis
a1 = fromRedis(r,unique)

# Check we got back what we wrote
np.testing.assert_array_equal(a0,a1)
