#ifndef Utils_Exception_hh
#define Utils_Exception_hh


#include <stdexcept>
#include <sstream>
#include <string>

namespace Error {

  class Exception : public std::runtime_error {
  public:
    explicit Exception(const std::string& msg, const char * f, unsigned l) throw();
    explicit Exception(const char * msg,  const char * f, unsigned l) throw();
    virtual const char * getTypeName() const throw() = 0;
    const char * getFile() const throw() { return m_file; }
    unsigned getLineNo() const throw() { return m_lineno; }
    Exception( const Exception & o ) throw();
    Exception & operator= ( const Exception & o ) throw();
    virtual ~Exception() throw();
  private:
    const char * m_file;
    unsigned m_lineno;
  };

#define ADD_ERROR_TYPE(ErrType)                                                               \
  struct ErrType : public Exception {                                                                \
    explicit ErrType(const std::string& m, const char * f, unsigned l) throw() : Exception(m,f,l) {} \
    explicit ErrType(const char * m,  const char * f, unsigned l) throw() : Exception(m,f,l) {}      \
    virtual const char * getTypeName() const throw() { return #ErrType; }                            \
    virtual ~ErrType() throw();                                                                      \
  }
  //List of error types (destructors implemented in .cc):
  ADD_ERROR_TYPE(FileNotFound);
  ADD_ERROR_TYPE(DataLoadError);
  ADD_ERROR_TYPE(MissingInfo);
  ADD_ERROR_TYPE(CalcError);
  ADD_ERROR_TYPE(LogicError);
  ADD_ERROR_TYPE(BadInput);
#undef ADD_ERROR_TYPE
}

#define ERROR_THROW(ErrType, msg)                     \
  {                                                   \
    throw ::Error::ErrType(msg,__FILE__,__LINE__);    \
  }

#define ERROR_THROW2(ErrType, msg)        \
  {                                       \
    std::ostringstream err_oss;           \
    err_oss << msg;                       \
    ERROR_THROW(ErrType,err_oss.str())    \
  }


#define error_str(s) #s
#define error_xstr(s) error_str(s)
#define error_assert_always(x) do { if (!(x)) { ERROR_THROW(LogicError,\
                              "Assertion failure: " error_xstr(x)); } } while(0)

#ifndef NDEBUG
#  define error_assert(x) error_assert_always(x)
#  define error_assert2(x) error_assert_always(x)
#else
#  define error_assert(x) do {} while(0)
#  define error_assert2(x) do { (void)sizeof(x); } while(0)//use but dont evaluate x
#endif

#endif
