#include "DAQInterface.hh"
#include <iostream>
#include <algorithm>  // remove_if
#include <time.h>

DAQInterface::DAQInterface(std::string daq_name, std::string ctrl_name)
: DimClient(), m_errhdl()
, m_daq_data(daq_name.c_str(),const_cast<char *>("daq"), this)
, m_control(daq_name.c_str(),const_cast<char *>("control"), this)
, m_num_trigger(0), m_received_byte(0), m_processed_byte(0), m_processed_event(0)
, m_houskpg_intvl_ms(1000), m_start_time(std::chrono::system_clock::now())
{
  std::cout << "service name is " << daq_name << std::endl;
  // setExitHandler("RUN_INFO");
  m_hk_thread = std::thread(&DAQInterface::houseKeeping, this);
}

DAQInterface::~DAQInterface()
{
  if(m_hk_thread.joinable())
    m_hk_thread.detach();
}


void DAQInterface::infoHandler()
{
  std::lock_guard<std::mutex> guard(m_task_mutex);
  // auto start = std::chrono::system_clock::now();

  DimInfo *curr = getInfo(); // get current DimInfo address
  if(curr == &m_daq_data)
  {
    //if(!m_num_trigger) return; //info handler will be triggered by constructor in dim v20
    unsigned dsize  = curr->getSize();
    m_received_byte += dsize;

    uint8_t *data = static_cast<uint8_t*>(curr->getData());
    std::vector<uint8_t> buf(data, data+dsize);

    m_tasks.emplace_back( DaqTask(data, data+dsize));
    m_tasks.back().setTime(curr->getTimestamp(), curr->getTimestampMillisecs());
    m_tasks.back().start();
  }
  else if(curr == &m_daq_data)
  {
    std::cout << "command info handler" << std::endl;
    char *type = curr->getString( );
  }

  m_num_trigger++;

  // auto end = std::chrono::system_clock::now();
  // std::chrono::duration<double> elapsed_seconds = end-start;
  // std::time_t end_time = std::chrono::system_clock::to_time_t(end);
  // std::cout << "elapsed time: " << elapsed_seconds.count()*1000 << "ms\n";

}


void DAQInterface::houseKeeping()
{
  std::cout <<"Starting house kepping ..." << std::endl;

  while (1) {
    std::this_thread::sleep_for(m_houskpg_intvl_ms);
    if(m_tasks.empty()) continue;

    std::lock_guard<std::mutex> guard(m_task_mutex);

    unsigned processed =0;

    auto newend = std::remove_if(m_tasks.begin(), m_tasks.end(),
    [&](DaqTask &atask ) -> bool
    {
      uint64_t processed = atask.readyAndValid();
      if(! processed )
        return false;
      else
      {
        int bytes = processed & 0xffffffff;
        int events = (processed >> 32) & 0xffffffff;
        m_processed_byte += bytes;
        m_processed_event += events;
        return true;
      }
    });

    if(newend == m_tasks.end() ) continue;

    unsigned deleted = m_tasks.end() - newend;
    m_tasks.erase(newend, m_tasks.end());

    std::cout <<  '\r' << "processed " << m_processed_byte/1024/1024  << "MB. Received "
    << m_processed_byte/1024/1024 << "MB. Processed " <<  m_processed_byte*100./m_received_byte <<"%. ";

    std::chrono::duration<double> elapsed_seconds = std::chrono::system_clock::now()-m_start_time;
		double t = elapsed_seconds.count();

    std::cout << "processed " << m_processed_event  << " events. Event rate "
    <<  m_processed_event/t <<" /s. ";

    std::cout  << m_tasks.size() << " tasks in queue, " << deleted << " finished tasks"
    << std::flush << std::flush << std::flush;
  }
}
