#include "MantidDataHandling/LoadCSNSNexus.h"
#include "MantidDataHandling/EventWorkspaceCollection.h"

#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_real.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/shared_array.hpp>
#include <boost/scoped_array.hpp>
#include <boost/function.hpp>
#include <functional>

#include "MantidKernel/ArrayProperty.h"
#include "MantidKernel/ThreadPool.h"
#include "MantidKernel/UnitFactory.h"
#include "MantidKernel/ThreadSchedulerMutexes.h"
#include "MantidKernel/BoundedValidator.h"
#include "MantidKernel/VisibleWhenProperty.h"
#include "MantidKernel/TimeSeriesProperty.h"
#include "MantidGeometry/Instrument/RectangularDetector.h"
#include "MantidAPI/FileProperty.h"
#include "MantidAPI/MemoryManager.h"
#include "MantidAPI/RegisterFileLoader.h"
#include "MantidAPI/SpectrumDetectorMapping.h"
#include "MantidKernel/Timer.h"
//#include <iostream>
//#include <fstream>
//#include <cstdlib>
#include <algorithm>
#include <math.h>

using std::cout;
using std::endl;
using std::map;
using std::string;
using std::vector;
using std::ofstream;

namespace Mantid
{
namespace DataHandling
{

DECLARE_NEXUS_FILELOADER_ALGORITHM(LoadCSNSNexus)

using namespace Kernel;
using namespace Geometry;
using namespace API;
using namespace DataObjects;

LoadCSNSNexus::LoadCSNSNexus()
    : m_bTotalPixels(0),m_mTotalPixels(0), pulseTimes(0), m_numBins(0), m_spec_min(0),
      m_spec_max(0), m_fileMutex() {}

//-------------------------------------------------------------------------------------------------
/// Initialisation method.
void LoadCSNSNexus::init()
{
    declareProperty(
        new FileProperty("Filename", "", FileProperty::Load, {".nxs"}),
        "The name of the NeXus file to load");
    declareProperty(new WorkspaceProperty<API::Workspace>("OutputWorkspace", "",
                    Direction::Output),
                    "The name of the Workspace2D to create.");
    declareProperty(
        new PropertyWithValue<string>("Entryname", "csns",Direction::Input), "Optional: Name of entry (default csns)");

    declareProperty(
        new ArrayProperty<string>("Bankname", Direction::Input),
        "Optional: A comma-separated list of bank to read. (default all banks)");

    declareProperty(
        new PropertyWithValue<bool>("Loadmonitor", true, Direction::Input),
        "Default true: load monitor data, false: no monitor data.");

    declareProperty(
        new ArrayProperty<string>("Monitorname", Direction::Input),
        "Optional: A comma-separated list of monitor to read. (default all monitors)");
    declareProperty(
        new PropertyWithValue<bool>("Loadevent", true, Direction::Input),
        "Default true: load event data mode, false: Load histogram data.");

    auto mustBePositive = boost::make_shared<BoundedValidator<int>>();
    mustBePositive->setLower(1);
    declareProperty(
        new PropertyWithValue<specid_t>("SpectrumMin", 1, mustBePositive),
        "The index number of the first spectrum to read.  Only used if spectrum_max is set.");
    declareProperty(
        new PropertyWithValue<specid_t>("SpectrumMax", Mantid::EMPTY_INT(),
                                        mustBePositive),
        "The number of the last spectrum to read. Only used if explicitly set.");
}

//-------------------------------------------------------------------------------------------------
/**
 * Return the confidence with with this algorithm can load the file
 * @param descriptor A descriptor for the file
 * @returns An integer specifying the confidence level. 0 indicates it will not
 * be used
 */
int LoadCSNSNexus::confidence(Kernel::NexusDescriptor &descriptor) const
{
    int confidence(0);
    if (descriptor.pathOfTypeExists("/csns", "NXentry"))
    {
        const bool hasEventData = descriptor.classTypeExists("NXevent_data");
        const bool hasHistogramData = descriptor.classTypeExists("NXdata");
        if (hasHistogramData && hasEventData)
            // Event data = this is event NXS
            confidence = 20;
        else if (hasHistogramData && !hasEventData)
            // No event data = this is the one
            confidence = 80;
        else if (!hasHistogramData && hasEventData)
            // Histogram Data only
            confidence = 40;
        else
            // No data ?
            confidence = 10;
    }
    return confidence;
}
//-------------------------------------------------------------------------------------------------
/**
 * Return the entry name
 * @param[in] file :: nxs file
 * @returns newEntry :: entry name
 */
std::string LoadCSNSNexus::getEntryName(::NeXus::File *file)
{
    std::string newEntry="";
    std::map<std::string, std::string> entries = file->getEntries();//(name,NXclass)
    if (entries.empty())
    {
        throw std::runtime_error("No entries in the NXS file!");
    }
    else if (entries.find("csns") != entries.end())
    {
        newEntry = "csns"; //if find csns,enwEntry is csns.
    }
    else
    {
        newEntry = entries.begin()->first; //get entry name not csns
    }
    return newEntry;
}
//-------------------------------------------------------------------------------------------------
/**
* Return Bank ot Monitor list NX/entry/instrument/bank* or monitor*
* @param[in] file :: nxs file
* @param[in] bm :: true means bank and false means monitor.
* @return detectorNames :: bank or monitor list
*/
std::vector<std::string> LoadCSNSNexus::getDetectorNames(::NeXus::File *file, bool bm)
{
    size_t numBank = m_bankNames.size();
    size_t numMonitor = m_monitorNames.size();
    std::vector<string> detectorNames;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    std::map<std::string, std::string> entries;
    std::map<std::string, std::string>::iterator it;
    if(bm)
    {
	if (numBank==0)
    	{
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                //if (name.substr(0, 4) == "bank")
                if (name.substr(0, 6) == "module")
	            detectorNames.push_back(name);
            }
        }else{
	     for (auto i = m_bankNames.begin(); i != m_bankNames.end(); ++i)
             {
                std::string name = *i;
		detectorNames.push_back(name);
	     }
        }
    }else{
	if (numMonitor==0)
        {
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                if (name.substr(0, 7) == "monitor")
                    detectorNames.push_back(name);
            }
        }else{
             for (auto i = m_monitorNames.begin(); i != m_monitorNames.end(); ++i)
             {
                std::string name = *i;
                detectorNames.push_back(name);
             }
        }
    }
    file->closeGroup();
    file->closeGroup();
    return detectorNames;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the CSNS NeXus definiton version
* @param[in] filename :: nxs file name
* @return version :: CSNS NeXus definition version
*/
std::string LoadCSNSNexus::getVersion(::NeXus::File *file)
{
    std::string version="";
    file->openGroup(m_entryName, "NXentry");
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("version") != entries.end())
    {
        file->readData("version", version);
    }
    else
    {
        throw std::runtime_error("No version in the NXS file!");
    }
    file->closeGroup();
    return version;
}
//-------------------------------------------------------------------------------------------------
/**
* Return sample information
* @param[in] filename :: nxs file name
* @return sampleInfo :: sample information
*/
CSNSsample LoadCSNSNexus::getSampleInfo(::NeXus::File *file)
{
//must consider if the data not exist!!!
    CSNSsample sampleInfo;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument","NXinstrument");
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("sample") != entries.end())
    {
        file->openGroup("sample", "NXsample");
    }
    else
    {
        throw std::runtime_error("No sample in the NXS file!");
    }
    std::map<std::string, std::string> entries2 = file->getEntries();
    if (entries2.find("name") != entries2.end())
    {
        file->readData("name",sampleInfo.name);
    }else
    {
        throw std::runtime_error("No sample name in the NXS file!");
    }
    if (entries2.find("type") != entries2.end())
    {
        file->readData("type",sampleInfo.type);
    }else
    {
        throw std::runtime_error("No sample type in the NXS file!");
    }
    if (entries2.find("situation") != entries2.end())
    {
        file->readData("situation",sampleInfo.situation);
    }else
    {
        throw std::runtime_error("No sample situation in the NXS file!");
    }
    if (entries2.find("chemical_formula") != entries2.end())
    {
        file->readData("chemical_formula",sampleInfo.chemical_formula);
    }else
    {
        throw std::runtime_error("No sample chemical formula in the NXS file!");
    }
    if (entries2.find("density") != entries2.end())
    {
        file->readData("density",sampleInfo.density);
    }else
    {
        throw std::runtime_error("No sample density in the NXS file!");
    }
    if (entries2.find("mass") != entries2.end())
    {
        file->readData("mass",sampleInfo.mass);
    }else
    {
        throw std::runtime_error("No sample mass in the NXS file!");
    }
    if (entries2.find("distance") != entries2.end())
    {
        file->readData("distance",sampleInfo.distance);
    }else
    {
        throw std::runtime_error("No sample distance in the NXS file!");
    }
    if (entries2.find("geometry") != entries2.end())
    {
        file->openGroup("geometry","NXgeometry");
        std::map<std::string, std::string> entries3 = file->getEntries();
        if(entries3.find("shape") != entries3.end())
        {
            file->openGroup("shape", "NXshape");
            std::map<std::string, std::string> entries4 = file->getEntries();
            if(entries4.find("shape") != entries4.end())
            {
                file->readData("shape",sampleInfo.shape);
            }else
            {
                throw std::runtime_error("No sample shape in the NXS file!");
            }
            if(entries4.find("size") != entries4.end())
            {
                file->readData("size",sampleInfo.size);
            }else
            {
                throw std::runtime_error("No sample size in the NXS file!");
            }
            
        }else
        {
            throw std::runtime_error("No sample shape entry in the NXS file!");
        }
    }else
    {
        throw std::runtime_error("No sample geometry in the NXS file!");
    }
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    return sampleInfo;   
}
//-------------------------------------------------------------------------------------------------
/**
* Return the start time of experiment
* @param[in] filename :: nxs file name
* @return startTime :: start time
*/
Kernel::DateAndTime LoadCSNSNexus::getStartTime(::NeXus::File *file)
{
    std::string startTime;
    file->openGroup(m_entryName, "NXentry");
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("start_time_utc") != entries.end())
    {
        file->readData("start_time_utc", startTime);
    }
    else
    {
        throw std::runtime_error("No start time in the NXS file!");
    }
    file->closeGroup();
    return Kernel::DateAndTime(startTime);;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the end time of experiment
* @param[in] filename :: nxs file name
* @return endTime :: end time
*/
Kernel::DateAndTime LoadCSNSNexus::getEndTime(::NeXus::File *file)
{
    std::string endTime;
    file->openGroup(m_entryName, "NXentry");
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("end_time_utc") != entries.end())
    {
        file->readData("end_time_utc", endTime);
    }
    else
    {
        throw std::runtime_error("No end time in the NXS file!");
    }
    file->closeGroup();
    return Kernel::DateAndTime(endTime);
}
//-------------------------------------------------------------------------------------------------
/**
* Return the pixel_id of all (or the specified) banks or monitors.
* @param[in] file :: nxs file
* @return pixelId :: pixel_id from all (or choose) banks or monitors.
*/
std::vector<int64_t> LoadCSNSNexus::getPixelId(::NeXus::File *file, bool bm)
{
    std::vector<string> detGroup = getDetectorNames(file, bm);
    std::vector<int64_t> _tmp;
    std::vector<int64_t> pixelId;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    for (auto j = detGroup.begin(); j != detGroup.end(); j++)
    {
	    string name = *j;
	    file->openGroup(name, "NXdetector");
        file->readData("pixel_id",_tmp);
	    for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
            pixelId.push_back(*i);
        file->closeGroup();
	    _tmp=std::vector<int64_t>();
    }                
    sort(pixelId.begin(),pixelId.end());
    file->closeGroup();
    file->closeGroup();
    return pixelId;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the time bins, banks and monitors have the same time bins 
* @param[in] filen :: nxs file
* @return TOF :: time of flight
*/

std::vector<uint32_t> LoadCSNSNexus::getTimeBin(::NeXus::File *file, bool bm)
{
    std::vector<int> tmp;
    std::map<std::string, std::string> entries;
    std::map<std::string, std::string> entries2;
    std::map<std::string, std::string>::iterator it;
    string detName="";
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    entries = file->getEntries();
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        std::string name = it->first;
        if (bm)
        {
            detName="module";
        }else{
            detName="monito";
        }
        cout<<name<<" ; "<<detName<<endl;
        
        if (name.substr(0, 6) == detName)
        {
            cout<<it->second<<endl;
            file->openGroup(name, it->second);
                cout<<222<<endl;
            entries2 = file->getEntries();
            if (entries2.find("time_of_flight") != entries2.end())
            {
                cout<<1111<<endl;
                file->readData("time_of_flight",tmp);
                cout<<"successful read timebin"<<endl;
            }else
            {
            throw std::runtime_error("No time_of_flight in the NXS file!");
            }
            file->closeGroup();
            break;
        }else{
            continue;
            //throw std::runtime_error("No module or monitor in the NXS file!");
        }
    }
    file->closeGroup();
    file->closeGroup();
    std::vector<uint32_t> timeBin(tmp.begin(), tmp.end());
    return timeBin;
}

//-------------------------------------------------------------------------------------------------
/**
* Return true if has event data
* @param[in] file :: nxs file
* @return hasEventData :: true|false
*/
bool LoadCSNSNexus::hasEventData(::NeXus::File *file)
{
    bool hasEventData=false;
    file->openGroup(m_entryName, "NXentry");
    try
    {
        file->openGroup("event_data", "NXcollection");
        std::map<std::string, std::string> entries = file->getEntries();
        std::map<std::string, std::string>::iterator it;
        for (it = entries.begin(); it != entries.end(); ++it)
        {
            if (it->second=="NXevent_data")
            {
                hasEventData = true;
                break;
            }
        }
        file->closeGroup();
    }
    catch(::NeXus::Exception)
    {
        hasEventData=false;
    }
    file->closeGroup();
    return hasEventData;
}
//-------------------------------------------------------------------------------------------------
/**
* Return true if has histogram data
* @param[in] file :: nxs file
* @return hasEventData :: true|false
*/
bool LoadCSNSNexus::hasHistogramData(::NeXus::File *file)
{
    bool hasHistogramData=false;
    file->openGroup(m_entryName, "NXentry");
    try
    {
        file->openGroup("histogram_data", "NXcollection");
        std::map<std::string, std::string> entries = file->getEntries();
        std::map<std::string, std::string>::iterator it;
        for (it = entries.begin(); it != entries.end(); ++it)
        {
            if (it->second=="NXdata")
            {
                hasHistogramData = true;
                break;
            }
        }
        file->closeGroup();
    }
    catch(::NeXus::Exception)
    {
        hasHistogramData=false;
    }
    file->closeGroup();
    return hasHistogramData;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the spectraMapping for bank
* @param[in] file :: nxs file
* @return spectraMapping :: mapping(pixel_id, spectrumNo from 0)
*/
/*
std::map<int64_t, uint32_t> LoadCSNSNexus::getSpectraMapping_bank(::NeXus::File *file)
{
    std::map<int64_t, uint32_t> spectraMapping;
    std::vector<int64_t> PId = getPixelId_bank(file);
    uint32_t p = 0;
//    spectraMapping.insert(std::map<int64_t, uint32_t>::value_type(-1,0));
    for (auto i = PId.begin(); i != PId.end(); ++i)
    {
        spectraMapping.insert(std::map<int64_t, uint32_t>::value_type(*i,p));
        p++;
    }
    return spectraMapping;
}
*/
//-------------------------------------------------------------------------------------------------
/**
* Return the spectraMapping for monitor
* @param[in] file :: nxs file
* @return spectraMapping :: mapping(pixel_id, spectrumNo from 0)
*/
/*
std::map<int64_t, uint32_t> LoadCSNSNexus::getSpectraMapping_mon(::NeXus::File *file)
{
    std::map<int64_t, uint32_t> spectraMapping;
    std::vector<int64_t> PId = getPixelId_mon(file);
	cout<<"the size of pid in monitor is: "<<PId.size()<<endl;
//    uint32_t p = 0;
    spectraMapping.insert(std::map<int64_t, uint32_t>::value_type(-1,0));
//    for (auto i = PId.begin(); i != PId.end(); ++i)
//    {
//        spectraMapping.insert(std::map<int64_t, uint32_t>::value_type(*i,p));
//        p++;
//    }
    return spectraMapping;
}
*/
//-------------------------------------------------------------------------------------------------
/**
* Return the pixel positions
* @param[in] file :: nxs file
* @param[in] bankName :: which bank you want to obtain the pixel positions
* @return pixelOffset :: [x,y]
*/
std::vector<std::vector<float> > LoadCSNSNexus::getPixelOffset(::NeXus::File *file, const std::string &bankName)
{
    std::vector<std::vector<float> > pixelOffset;
//    size_t numBank = m_bankNames.size();
//    size_t numMonitor = m_monitorNames.size();
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup(bankName, "NXdetector");

    std::vector<float> _tmp;
    file->readData("x_pixel_offset",_tmp);
    pixelOffset.push_back(_tmp);
    file->readData("y_pixel_offset",_tmp);
    pixelOffset.push_back(_tmp);

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();

    return pixelOffset;
}
//-------------------------------------------------------------------------------------------------
/**
 * Creates a time series property from the currently opened log entry. It is
 * assumed to
 * have been checked to have a time field and the value entry's name is given as
 * an argument
 * @param file :: A reference to the file handle
 * @param prop_name :: The name of the property
 * @returns A pointer to a new property containing the time series
 */
Kernel::Property *LoadCSNSNexus::createTimeSeries(::NeXus::File *file,
		const std::string &prop_name) const
{
	file->openData("time");
	//----- Start time is an ISO8601 string date and time. ------
	std::string start;
	try{
		file->getAttr("start", start);}
	catch (::NeXus::Exception &){
		start="2017-12-31T00:00:00+08:00";}
	// Convert to date and time
	Kernel::DateAndTime start_time = Kernel::DateAndTime(start);
	std::string time_units;
	file->getAttr("units", time_units);
	if (time_units.compare("second") < 0 && time_units != "s" &&
			time_units != "minutes") // Can be s/second/seconds/minutes
	{
		file->closeData();
		throw ::NeXus::Exception("Unsupported time unit '" + time_units + "'");
	}
	//--- Load the seconds into a double array ---
	std::vector<double> time_double;
	try
	{
		file->getDataCoerce(time_double);
	}
	catch (::NeXus::Exception &e)
	{
		g_log.warning() << "Log entry's time field could not be loaded: '"
			<< e.what() << "'.\n";
		file->closeData();
		throw;
	}
	file->closeData(); // Close time data
	g_log.debug() << "   done reading \"time\" array\n";
	// Convert to seconds if needed
	if (time_units == "minutes")
	{
		std::transform(time_double.begin(), time_double.end(), time_double.begin(),
				std::bind2nd(std::multiplies<double>(), 60.0));
	}
	// Now the values: Could be a string, int or double
	file->openData("value");
	// Get the units of the property
	std::string value_units("");
	try
	{
		file->getAttr("units", value_units);
	}
	catch (::NeXus::Exception &)
	{
		// Ignore missing units field.
		value_units = "";
	}
	// Now the actual data
	::NeXus::Info info = file->getInfo();
	// Check the size
	if (size_t(info.dims[0]) != time_double.size())
	{
		file->closeData();
		throw ::NeXus::Exception("Invalid value entry for time series");
	}
	if (file->isDataInt()) // Int type
	{
		std::vector<int> values;
		try
		{
			file->getDataCoerce(values);
			file->closeData();
		}
		catch (::NeXus::Exception &)
		{
			file->closeData();
			throw;
		}
		// Make an int TSP
		auto tsp = new TimeSeriesProperty<int>(prop_name);
		tsp->create(start_time, time_double, values);
		tsp->setUnits(value_units);
		g_log.debug() << "   done reading \"value\" array\n";
		return tsp;
	}
	else if (info.type == ::NeXus::CHAR)
	{
		std::string values;
		const int64_t item_length = info.dims[1];
		try
		{
			const int64_t nitems = info.dims[0];
			const int64_t total_length = nitems * item_length;
			boost::scoped_array<char> val_array(new char[total_length]);
			file->getData(val_array.get());
			file->closeData();
			values = std::string(val_array.get(), total_length);
		}
		catch (::NeXus::Exception &)
		{
			file->closeData();
			throw;
		}
		// The string may contain non-printable (i.e. control) characters, replace
		// these
		std::replace_if(values.begin(), values.end(), iscntrl, ' ');
		auto tsp = new TimeSeriesProperty<std::string>(prop_name);
		std::vector<DateAndTime> times;
		DateAndTime::createVector(start_time, time_double, times);
		const size_t ntimes = times.size();
		for (size_t i = 0; i < ntimes; ++i)
		{
			std::string value_i =
				std::string(values.data() + i * item_length, item_length);
			tsp->addValue(times[i], value_i);
		}
		tsp->setUnits(value_units);
		g_log.debug() << "   done reading \"value\" array\n";
		return tsp;
	}
	else if (info.type == ::NeXus::FLOAT32 || info.type == ::NeXus::FLOAT64)
	{
		std::vector<double> values;
		try
		{
			file->getDataCoerce(values);
			file->closeData();
		}
		catch (::NeXus::Exception &)
		{
			file->closeData();
			throw;
		}
		auto tsp = new TimeSeriesProperty<double>(prop_name);
		tsp->create(start_time, time_double, values);
		tsp->setUnits(value_units);
		g_log.debug() << "   done reading \"value\" array\n";
		return tsp;
	}
	else
	{
		throw ::NeXus::Exception(
				"Invalid value type for time series. Only int, double or strings are "
				"supported");
	}
}
/*-----------------------------------------------------------------------------------------------------------
 * Load an log entry
 * @param file:: NeXus file
 * @param workspace :: A pointer to the workspace to store the logs
 */
void LoadCSNSNexus::loadLog( ::NeXus::File *file, boost::shared_ptr<API::MatrixWorkspace> workspace) const
//void LoadCSNSNexus::loadLog( ::NeXus::File *file, EventWorkspace_sptr workspace) const
{
	file->openGroup(m_entryName, "NXentry");
	try{file->openGroup("logs", "NXcollection");}
	catch (::NeXus::Exception &e){
		g_log.warning() <<"Read logs: " << e.what()<<"\n";
		file->closeGroup();
		return;
	}
	// Validate the NX log class.
	std::map<std::string, std::string> entries = file->getEntries();
	std::map<std::string, std::string>::iterator it;
	for (it = entries.begin(); it != entries.end(); ++it)
	{
		std::string logName=it->first;
		if (it->second == "NXlog")
		{
			file->openGroup(it->first, "NXlog");
			std::map<std::string, std::string> subEntries = file->getEntries();
			if ((subEntries.find("value") == subEntries.end()) || (subEntries.find("time") == subEntries.end()))
			{   g_log.warning() << "Invalid NXlog entry " << logName
				<< " found. Did not contain 'value' and 'time'.\n";}
			else{
				try{
					if (!(workspace->run().hasProperty(logName)))
					{
						g_log.debug()<<logName<<"\n";
						Kernel::Property *logValue = createTimeSeries(file, logName);
						workspace->mutableRun().addProperty(logValue, true);
					}
				}
				catch (::NeXus::Exception &e)
				{
					g_log.warning() << "NXlog entry " << logName
						<< " gave an error when loading:'" << e.what() << "'.\n";
				}
			}
			file->closeGroup();
		}//if
		if (it->second == "NXcollection")
		{
			std::string groupName=it->first;
			file->openGroup(it->first, "NXcollection");

			std::map<std::string, std::string> subEntries = file->getEntries();
			std::map<std::string, std::string>::iterator subit;
			for (subit = subEntries.begin(); subit != subEntries.end(); ++subit)
			{
				if (subit->second == "NXlog")
				{
					file->openGroup(subit->first, "NXlog");
					std::map<std::string, std::string> subEntries = file->getEntries();
					if ((subEntries.find("value") == subEntries.end()) || (subEntries.find("time") == subEntries.end()))
					{
						g_log.warning() << "Invalid NXlog entry " << subit->first
							<< " found. Did not contain 'value' and 'time'.\n";
					}
					else
					{
						try
						{
							if (!(workspace->run().hasProperty(subit->first)))
							{
								g_log.debug()<<subit->first<<"\n";
								Kernel::Property *logValue = createTimeSeries(file, groupName+"_"+subit->first);
								workspace->mutableRun().addProperty(logValue, true);
							}
						}
						catch (::NeXus::Exception &e)
						{
							g_log.warning() << "NXlog entry " << subit->first
								<< " gave an error when loading:'" << e.what() << "'.\n";
						}
					}
					file->closeGroup();
				}
			}
			file->closeGroup();
		}//if
	}//for
	file->closeGroup();
	file->closeGroup();
}
//-------------------------------------------------------------------------------------------------
/**
* Return the value in histogram data
* @param[in] file :: nxs file
* @param[in] bm :: true for banks false for monitors
* @return value :: value in nexus file is 3-dimension data.
*/
std::vector<uint32_t> LoadCSNSNexus::getHistData(::NeXus::File *file, bool bm)
{
    //size_t numBank = m_bankNames.size();
    //size_t numMonitor = m_monitorNames.size();
    std::vector<int> _tmp;
    std::vector<int> tmp;

    std::vector<std::string> detGroup = getDetectorNames(file, bm);
    std::map<std::string, std::string> entries;
    std::map<std::string, std::string>::iterator it;
    cout<<m_entryName<<endl;
    file->openGroup("csns", "NXentry"); 
    file->openGroup("histogram_data", "NXcollection");

    for (auto j = detGroup.begin(); j != detGroup.end(); ++j)
    {
        std::string name = *j;
        file->openGroup(name, "NXdata");
        file->readData("histogram_data", _tmp);
        for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
            tmp.push_back((*i));
        file->closeGroup();
    }
    std::vector<uint32_t>value(tmp.begin(), tmp.end());
    file->closeGroup();
    file->closeGroup();
    return value;
}

//------------------------------------------------------------------------------------------------
/**
 * * Return the event pid for bank
 * * @param[in] file :: nxs file
 * * @return eventPixelId 
 * */
/*
std::vector<int64_t> LoadCSNSNexus::getEventPixelId_bank(::NeXus::File *file)
{
    std::vector<string> bankGroup = getDetectorNames(file, true);
    std::vector<int64_t> _tmp;
    std::vector<int64_t> eventPixelId;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("event_data", "NXcollection");
    for (auto j = bankGroup.begin(); j != bankGroup.end(); j++)
    {
        string name = *j;
        file->openGroup(name, "NXevent_data");
        file->readData("event_pixel_id",_tmp);
        cout<<"the size of event pid in bank is: "<<_tmp.size()<<endl;
        for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
            eventPixelId.push_back(*i);
        file->closeGroup();
        _tmp.clear();
    }
    cout<<"the total event pid in bank is: "<<eventPixelId.size()<<endl;
    file->closeGroup();
    file->closeGroup();
    return eventPixelId; 
}
*/
//------------------------------------------------------------------------------------------------
/**
 * * Return the event pid for monitor
 * * @param[in] file :: nxs file
 * * @return eventPixelId 
 * */
/*
std::vector<int64_t> LoadCSNSNexus::getEventPixelId_mon(::NeXus::File *file)
{
    std::vector<string> monitorGroup = getDetectorNames(file, false);
    std::vector<int64_t> _tmp;
    std::vector<int64_t> eventPixelId;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("event_data", "NXcollection");
    for (auto j = monitorGroup.begin(); j != monitorGroup.end(); j++)
    {
        string name = *j;
        file->openGroup(name, "NXevent_data");
        file->readData("event_pixel_id",_tmp);
        cout<<"the size of event pid in monitor is: "<<_tmp.size()<<endl;
        for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
            eventPixelId.push_back(*i);
        file->closeGroup();
        _tmp.clear();
    }
    cout<<"the total event pid in monitor is: "<<eventPixelId.size()<<endl;
    file->closeGroup();
    file->closeGroup();
    return eventPixelId; 
}
*/
//------------------------------------------------------------------------------------------------
/**
 * * Return the event tof for bank
 * * @param[in] file :: nxs file
 * * @return eventTOF 
 * */
/*
std::vector<float> LoadCSNSNexus::getEventTOF_bank(::NeXus::File *file)
{
    std::vector<string> bankGroup = getDetectorNames(file, true);
    std::vector<float> _tmp;
    std::vector<float> eventTOF;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("event_data", "NXcollection");
    cout<<"In getEventTOF: the size of bankGroup is "<<bankGroup.size()<<endl;
    for (auto j = bankGroup.begin(); j != bankGroup.end(); j++)
    {
        string name = *j;
        file->openGroup(name, "NXevent_data");
        file->readData("event_time_of_flight",_tmp);
        cout<<"the size of event tof in bank is: "<<_tmp.size()<<endl;
        for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
            eventTOF.push_back(*i);
        file->closeGroup();
        _tmp.clear();
    }
    cout<<"the total event TOF in bank is: "<<eventTOF.size()<<endl;
    file->closeGroup();
    file->closeGroup();
    return eventTOF; 
}
*/
//------------------------------------------------------------------------------------------------
/**
 * * Return the event tof for monitor
 * * @param[in] file :: nxs file
 * * @return eventTOF 
 * */
/*
std::vector<float> LoadCSNSNexus::getEventTOF_mon(::NeXus::File *file)
{
    std::vector<string> monitorGroup = getDetectorNames(file, false);
    std::vector<float> _tmp;
    std::vector<float> eventTOF;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("event_data", "NXcollection");
    for (auto j = monitorGroup.begin(); j != monitorGroup.end(); j++)
    {
        string name = *j;
        file->openGroup(name, "NXevent_data");
        file->readData("event_time_of_flight",_tmp);
        cout<<"the size of event tof in monitor is: "<<_tmp.size()<<endl;
        for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
            eventTOF.push_back(*i);
        file->closeGroup();
        _tmp.clear();
    }
    cout<<"the total event TOF in mon is: "<<eventTOF.size()<<endl;
    file->closeGroup();
    file->closeGroup();
    return eventTOF; 
}
*/
//------------------------------------------------------------------------------------------------
/**
 * * Return the event pulse time for bank
 * * @param[in] file :: nxs file
 * * @return eventPulseTime 
 * */
/*
std::vector<uint64_t> LoadCSNSNexus::getEventPulseTime_bank(::NeXus::File *file)
{
    std::vector<string> bankGroup = getDetectorNames(file, true);
    std::vector<float> _tmptof;
    std::vector<int64_t> _tmpindex;
    std::vector<uint64_t> _tmp;
    std::vector<uint64_t> _tmpeventPulseTime;
    std::vector<uint64_t> eventPulseTime;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("event_data", "NXcollection");
    std::vector<int64_t>::iterator it_index;
    std::vector<int64_t>::iterator it_tmp;
    std::vector<uint64_t>::iterator it_ptime;
    cout<<"In getEventPulseTime: the size of bankGroup is "<<bankGroup.size()<<endl;
    for (auto j = bankGroup.begin(); j != bankGroup.end(); j++)
    {
        string name = *j;
        file->openGroup(name, "NXevent_data");
        file->readData("event_time_of_flight",_tmptof);
        size_t num = _tmptof.size();
        int evtnum = int(num);
        //	cout<<"the event number is "<<evtnum<<endl;
        file->readData("event_index",_tmpindex);
        file->readData("event_pulse_time",_tmp);
        uint32_t n = 0;
        it_ptime = _tmp.begin();
        it_tmp = _tmpindex.end()-1;
        for(it_index = _tmpindex.begin(); it_index != _tmpindex.end()-1; it_index ++)
        {
            if(*it_index >= 0){
                auto delta = *(it_index+1)-*it_index;
                if (delta != 0){
                    auto value = *(it_ptime+n);
                    for(int64_t i = 0; i != delta ; i++)
                        _tmpeventPulseTime.push_back(value);
                    n++;
                }else{
                    n++;
                }
            }else{
                continue;
                n++;
            } 
        }
        auto delta2 = evtnum-*it_index;
        if (delta2 != 0){
            auto value2 = *(it_ptime+n);
            for(int64_t i = 0; i != delta2 ; i++)
                _tmpeventPulseTime.push_back(value2);
            n++;
        }else{
            n++;
        }
        for (auto i = _tmpeventPulseTime.begin(); i != _tmpeventPulseTime.end(); ++i)
            eventPulseTime.push_back(*i);
        file->closeGroup();
        _tmpeventPulseTime.clear();
        _tmp.clear();
        _tmpindex.clear();
        _tmptof.clear();
    }
    cout<<"the total number of ptime in bank is: "<<eventPulseTime.size()<<endl;

    file->closeGroup();
    file->closeGroup();
    return eventPulseTime; 
}
*/
//------------------------------------------------------------------------------------------------
/**
 * * Return the event pulse time for monitor
 * * @param[in] file :: nxs file
 * * @return eventPulseTime 
 * */

/*
std::vector<uint64_t> LoadCSNSNexus::getEventPulseTime_mon(::NeXus::File *file)
{
    std::vector<string> monitorGroup = getDetectorNames(file, false);
    std::vector<float> _tmptof;
    std::vector<int64_t> _tmpindex;
    std::vector<uint64_t> _tmp;
    std::vector<uint64_t> _tmpeventPulseTime;
    std::vector<uint64_t> eventPulseTime;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("event_data", "NXcollection");
    std::vector<int64_t>::iterator it_index;
    std::vector<int64_t>::iterator it_tmp;
    std::vector<uint64_t>::iterator it_ptime;
    cout<<"In getEventPulseTime: the size of monitorGroup is "<<monitorGroup.size()<<endl;
    for (auto j = monitorGroup.begin(); j != monitorGroup.end(); j++)
    {
        string name = *j;
        file->openGroup(name, "NXevent_data");
        file->readData("event_time_of_flight",_tmptof);
        size_t num = _tmptof.size();
        int evtnum = int(num);
        //	cout<<"the event number is "<<evtnum<<endl;
        file->readData("event_index",_tmpindex);
        file->readData("event_pulse_time",_tmp);
        uint32_t n = 0;
        it_ptime = _tmp.begin();
        it_tmp = _tmpindex.end()-1;
        for(it_index = _tmpindex.begin(); it_index != _tmpindex.end()-1; it_index ++)
        {
            if(*(it_index)<0)
            {
                //	it_index++;
                n++;
            }else{	
                auto delta = *(it_index+1)-*it_index;
                if (delta != 0){
                    auto value = *(it_ptime+n);
                    for(int64_t i = 0; i != delta ; i++)
                        _tmpeventPulseTime.push_back(value);
                    n++;
                }else{
                    n++;
                }
            }
        }
        auto delta2 = evtnum-*it_index;
        if (delta2 != 0){
            auto value2 = *(it_ptime+n);
            for(int64_t i = 0; i != delta2 ; i++)
                _tmpeventPulseTime.push_back(value2);
            //n++;
        }
        for (auto i = _tmpeventPulseTime.begin(); i != _tmpeventPulseTime.end(); ++i)
            eventPulseTime.push_back(*i);
        file->closeGroup();
        _tmpeventPulseTime.clear();
        _tmp.clear();
        _tmpindex.clear();
        _tmptof.clear();
    }
    cout<<"the total number of ptime in mon is: "<<eventPulseTime.size()<<endl;
    file->closeGroup();
    file->closeGroup();
    return eventPulseTime; 
}
*/
//-------------------------------------------------------------------------------------------------
/**
 * Return the event data for bank
 * @param[in] file :: nxs file
 * @return eventData :: event data include (specNo, tof, pulse_time). As a multimap, one pixel_id has several pairs of (tof, pulsetime).
 * pulse time unit is nanoseconds
 */
/*
std::multimap<uint32_t, std::pair<float, uint64_t>> LoadCSNSNexus::getEventData_bank(::NeXus::File *file)
{
    uint32_t specNo;
    std::multimap<uint32_t, std::pair<float, uint64_t>> eventData;
    std::vector<int64_t> _pixelId = getEventPixelId_bank(file);
    std::vector<float> _timeOfFlight = getEventTOF_bank(file);
    std::vector<uint64_t> _ptime = getEventPulseTime_bank(file);
    std::map<int64_t, uint32_t> spectraMapping = getSpectraMapping_bank(file);
    std::pair<float, uint64_t> _tofPulse;

    for (auto i=0; i<int(_pixelId.size()); ++i)
    {
        specNo = spectraMapping[_pixelId[i]];
        _tofPulse = std::make_pair(_timeOfFlight[i], _ptime[i]);
        eventData.insert(std::multimap<uint32_t, std::pair<float, uint64_t>>::value_type(specNo, _tofPulse));
    }
    cout<<"the size of event data in banks is: "<<eventData.size()<<endl;
    //	cout<<"successful in get event data in bank"<<endl;
    return eventData;
}
*/
//-------------------------------------------------------------------------------------------------
/**
 * Return the event data for monitor
 * @param[in] file :: nxs file
 * @return eventData :: event data include (specNo, tof, pulse_time). As a multimap, one pixel_id has several pairs of (tof, pulsetime).
 * pulse time unit is nanoseconds
 */
/*
std::multimap<uint32_t, std::pair<float, uint64_t>> LoadCSNSNexus::getEventData_mon(::NeXus::File *file)
{
    std::multimap<uint32_t, std::pair<float, uint64_t>> eventData;
    //std::pair<float, uint64_t>> eventData;
    std::vector<int64_t> _pixelId = getEventPixelId_mon(file);
    std::vector<float> _timeOfFlight = getEventTOF_mon(file);
    std::vector<uint64_t> _ptime = getEventPulseTime_mon(file);
    std::map<int64_t, uint32_t> spectraMapping;
    //std::map<int64_t, uint32_t> spectraMapping = getSpectraMapping_mon(file);
    uint32_t specNo;
    spectraMapping.insert(map<int64_t, uint32_t>::value_type(-1,0));
    std::pair<float, uint64_t> _tofPulse;

    for (auto i=0; i< int(_pixelId.size()); ++i)
        //for (auto i=0; i< int(_timeOfFlight.size()); ++i)
    {
        specNo = spectraMapping[_pixelId[i]];
        _tofPulse = std::make_pair(_timeOfFlight[i], _ptime[i]);
        eventData.insert(std::multimap<uint32_t, std::pair<float, uint64_t>>::value_type(specNo, _tofPulse));
    }
    //cout<<"the size of event data in monitors is: "<<eventData.size()<<endl;
    cout<<"successful in get event data in monitor"<<endl;
    return eventData;
}
*/
//-------------------------------------------------------------------------------------------------
/**
 * Load event data into workspace
 * @param[in] filename :: nxs file name
 * @param[in] workspace :: workspace name
 * @param[in] totalPixels :: total pixel numbers
 * @param[in] timeBin :: time of flight
 */
/*
void LoadCSNSNexus::loadEventData(::NeXus::File *file, boost::shared_ptr<DataObjects::EventWorkspace> workspace, size_t totalPixels, std::vector<uint32_t> timeBin)
{
    cout<<"start load event data into ws_bank"<<endl;
    workspace->initialize(totalPixels,1,1);
    std::multimap<uint32_t,std::pair<float, uint64_t>> eventData_bank = getEventData_bank(file);
    float m_tof;
    uint64_t m_pulseTime;
    std::multimap<uint32_t,std::pair<float, uint64_t>>::iterator it;
    int numPixels = int(totalPixels);
    int n2=0;
    cout<<"In the load event data into ws, the numPixels is "<<numPixels<<endl;
    cout<<"In the load event data into ws, the eventData_bank is "<<eventData_bank.size()<<endl;
    //	it = eventData_mon.find();
    int p1;
    for (int i = 0; i != numPixels; i++)
    {
        it = eventData_bank.find(i);
        if(it == eventData_bank.end()){
            continue;
        }else{
            //		cout<<"specNo || events is "<<i<<"||"<<eventData_bank.count(i)<<endl;
            p1=0;
            for (size_t j = 0; j != eventData_bank.count(i); j++, it++)
            {
                m_pulseTime = it->second.second;
                m_tof = it->second.first;
                //			cout<<"number is: "<<n2<<"||"<<m_tof<<"||"<<m_pulseTime<<endl;
                workspace->getEventList(i) += TofEvent(m_tof, m_pulseTime);//Y values are 0 
                n2++;
                p1++;
            }
            cout<<i<<" || "<<p1<<endl;
            workspace->getEventList(i).setSpectrumNo(i);
        }
    }
    cout<<"the total number in WS_bank is "<<n2<<endl;
    Kernel::cow_ptr<MantidVec> axis; // MantidVec = std::vector<double>. axis point x axis.
    MantidVec &xRef = axis.access(); //if data is shared, creates a copy of data,return new copy of *this.
    xRef.resize(2); //set new column size of 2
    xRef[0] = *(timeBin.begin());
    xRef[1] = *(timeBin.end()-1); //set tof-max
    workspace->setAllX(axis); //set X size. setAllX(const HistogramData::BinEdges &x)
}
*/
//----------------------------------------------------------------------------------
/**
 * Load histgram data into workspace
 * @param[in] workspace :: workspace name
 * @param[in] totalPixels :: total pixel numbers
 * @param[in] timeBin :: time of flight
 */
void LoadCSNSNexus::loadHistData(::NeXus::File *file, boost::shared_ptr<API::MatrixWorkspace> workspace, std::vector<uint32_t> timeOfFlight, size_t totalPixels, std::vector<uint32_t> histData)
{
    std::vector<double>::iterator it9;
    std::vector<uint32_t>::iterator it2;
    size_t numBins = timeOfFlight.size();
    MantidVecPtr x, e;
    MantidVec &xRef = x.access();
    xRef.resize(numBins);
    int st1=0;
    for (it2 = timeOfFlight.begin(); it2 != timeOfFlight.end(); it2++)
    {
        xRef[st1] = (*it2);
        st1++;
    }
    size_t hist=0;
    std::vector<uint32_t>::iterator it_start = histData.begin();
    std::vector<uint32_t>::iterator it_end = it_start + numBins-1;
    std::vector<uint32_t>::iterator it_tof = timeOfFlight.begin();

    std::vector<uint32_t>::iterator it_tmp;
    std::vector<double> err;
    for(it_tmp=histData.begin();it_tmp!=histData.end();it_tmp++)
    {
        err.push_back(sqrt(*it_tmp));
    }
    std::vector<double>::iterator it_err_start = err.begin();
    std::vector<double>::iterator it_err_end = it_err_start + numBins-1 ;
    while(hist < totalPixels){
        MantidVec &Y = workspace->dataY(hist);
        MantidVec &E = workspace->dataE(hist);
        Y.assign(it_start, it_end);
        E.assign(it_err_start, it_err_end);
        it_start += (numBins-1);
        it_end += (numBins-1);
        it_err_start += (numBins-1);
        it_err_end += (numBins-1);
        workspace->setX(hist, x);
        hist++;
    }
}







//-------------------------------------------------------------------------------------------------
/**
 * Load monitor data into workspace as event
 * @param[:in] filename :: nxs file name
 * @param[in] workspace :: workspace name
 * @param[in] timeBin :: time of flight
 */
/*
   void LoadCSNSNexus::loadMonitorData(::NeXus::File *file, boost::shared_ptr<DataObjects::EventWorkspace> workspace, std::vector<uint32_t> timeBin)
   {
   cout<<"start load monitor data into ws_mon"<<endl;
   workspace->initialize(1,1,1);
//cout<<"the total pixel is "<<totalPixels<<endl;
std::multimap<uint32_t,std::pair<float, uint64_t>> eventData_mon = getEventData_mon(file);
float m_tof;
uint64_t m_pulseTime;

std::vector<float> _timeOfFlight = getEventTOF_mon(file);
std::vector<uint64_t> _ptime = getEventPulseTime_mon(file);

std::multimap<uint32_t,std::pair<float, uint64_t>>::iterator it_mon;
uint32_t n2=0;
//	it_mon = eventData_mon.begin();
//	for (uint32_t a = 0; a<100001; a++){
//	it_mon++;
//	}
//	cout<<"the  specNo= 100000 is: "<<it_mon->first<<"||"<<it_mon->second.first<<"||"<<it_mon->second.second<<endl;
//	cout<<"the size of eventdata in monitor is: "<<eventData_mon.count(0)<<endl;
//	it_mon++;
//	cout<<"the specNo 2 is: "<<it_mon->first<<"||"<<it_mon->second.first<<"||"<<it_mon->second.second<<endl;

int p =0;
//	for (int i = 0; i<10; i++)
//	{
//cout<<"i is: "<<i<<endl;
for (it_mon = eventData_mon.begin();it_mon != eventData_mon.end(); it_mon++)
{
m_pulseTime = it_mon->second.second;
m_tof = it_mon->second.first;
//cout<<"p || j : "<<p<<" || "<<j<<endl;
//	std::vector<Mantid::DataObjects::TofEvent> *eventVector;
//	eventVector->push_back(TofEvent(m_tof, m_pulseTime));
workspace->getEventList(p) += TofEvent(m_tof, m_pulseTime);//Y values are 0 
//	getEventsFrom(workspace->getEventList(p),eventVector);

n2++;
}
workspace->getEventList(p).setSpectrumNo(p+1);
//	cout<<"p is: "<<p<<endl;
p++;
//	}

it_mon = eventData_mon.begin();
m_pulseTime = it_mon->second.second;
m_tof = it_mon->second.first;
// cout<<"number is: 0 "<<"||"<<m_tof<<"||"<<m_pulseTime<<endl;
//	cout<<"the total number in WS_mon is "<<n2<<endl;
//	cout<<"the total number in WS is "<<n2<<endl;
Kernel::cow_ptr<MantidVec> axis; // MantidVec = std::vector<double>. axis point x axis.
MantidVec &xRef = axis.access(); //if data is shared, creates a copy of data,return new copy of *this.
xRef.resize(2); //set new column size of 2
xRef[0] = *(timeBin.begin());
xRef[1] = *(timeBin.end()-1); //set tof-max
workspace->setAllX(axis); //set X size. setAllX(const HistogramData::BinEdges &x)
}
*/
//=============================================================================================================
//-------------------------------------------------------------------------------------------------
void LoadCSNSNexus::exec()
{
    // The input properties
    std::string filename = getPropertyValue("Filename");
    m_entryName = getPropertyValue("Entryname");
    m_bankNames = getProperty("Bankname");
    m_monitorNames = getProperty("Monitorname");
    //	cout<<"the number of bank is: "<<m_bankNames.size()<<endl;
    //cout<<"the number of monitors is: "<<m_monitorNames.size()<<endl;
    bool m_loadMonitor = getProperty("Loadmonitor");
    bool m_loadEvent = getProperty("Loadevent");
    m_spec_min = getProperty("SpectrumMin");
    m_spec_max = getProperty("SpectrumMax");
    auto prog = new Progress(this, 0.0, 1.0, 10);
    prog->doReport("Start to load Nexus");

    // Open nxs file
    auto file = new ::NeXus::File(filename);
    // make sure that the entry name in NXS file is csns.
    std::string newEntryName = getEntryName(file);
    cout<<"Get Entry Name: "<< newEntryName<<endl;
    if (newEntryName != m_entryName)
    {
        throw std::runtime_error(m_entryName+" not found in the NXS file!");
    }

    //Get version
    std::string m_version = getVersion(file);
    cout<<"CSNS Nexus version is: "<< m_version<<endl;
    g_log.debug()<<"CSNS Nexus version is: " <<m_version<<"\n";
    //Get sample information
    CSNSsample m_sampleInfo = getSampleInfo(file);
    cout<<"Have get sample Info: "<<endl;
    // Get start time and end time
    Kernel::DateAndTime start_time = getStartTime(file);
    Kernel::DateAndTime end_time = getEndTime(file);
    cout<<"the start_time is "<<start_time<<endl;

    // process histogram data
    g_log.debug() <<"Histogram data will be loaded.\n";
    prog->doReport("load time of flight");
    std::vector<uint32_t> m_timeBin = getTimeBin(file, true);
    m_bNumBins = m_timeBin.size();
    cout<<"the bins is: "<<m_bNumBins<<endl;

    prog->doReport("Counting pixels");
    std::vector<int64_t> pixelId_bank = getPixelId(file, true);
    m_bTotalPixels = pixelId_bank.size();
    g_log.debug() <<m_bTotalPixels <<" pixels found."<< std::endl;
    cout<<"the pixel number in the all banks: "<<m_bTotalPixels<<endl;

    prog->doReport("histogram data");
    std::vector<uint32_t> histData=getHistData(file, true);
    cout<<"size of hist data: "<<histData.size()<<endl;

    prog->doReport("Creating workspace");
    MatrixWorkspace_sptr WS1 = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D", m_bTotalPixels, m_bNumBins, m_bNumBins-1));
    if(m_loadMonitor)
    {
        std::vector<uint32_t> m_timeBin_mon = getTimeBin(file, false);
        m_mNumBins = m_timeBin_mon.size();
        std::vector<int64_t> pixelId_mon = getPixelId(file, false);
        m_mTotalPixels = pixelId_mon.size();
        std::vector<uint32_t> histData_mon=getHistData(file, false);

        MatrixWorkspace_sptr WS2 = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D", m_mTotalPixels, m_mNumBins, m_mNumBins-1));
        WorkspaceGroup_sptr wksp_group(new WorkspaceGroup);
        wksp_group->addWorkspace(WS1);
        wksp_group->addWorkspace(WS2);
        setProperty("OutputWorkspace",boost::dynamic_pointer_cast<Workspace>(wksp_group));
        loadHistData(file, WS2,m_timeBin_mon, m_mTotalPixels, histData_mon);
        WS2->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");
        WS2->setYUnit("Counts");
    }else{
    setProperty("OutputWorkspace",boost::dynamic_pointer_cast<Workspace>(WS1));
    } 
    loadHistData(file, WS1,m_timeBin, m_bTotalPixels, histData);
    WS1->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");
    WS1->setYUnit("Counts");
        
    /*
       prog->doReport("Counting pixels");
       std::vector<int64_t> pixelId_bank = getPixelId_bank(file);
       m_bTotalPixels = pixelId_bank.size();
       g_log.debug() <<m_bTotalPixels <<" pixels found."<< std::endl;
       cout<<"the pixel number in the all banks: "<<m_bTotalPixels<<endl;
       prog->doReport("Counting TOF bins");
       std::vector<uint32_t> m_timeBin = getTimeBin(file);
       m_numBins = m_timeBin.size();
       cout<<"the bins is: "<<m_numBins<<endl;
    // g_log.debug() <<m_bNumBins <<" TOF bins found."<< std::endl;



    //    std::vector<uint32_t> pixelId_monitor = getPixelId(file, false);
    std::vector<int64_t> pixelId_mon = getPixelId_mon(file);
    m_mTotalPixels = pixelId_mon.size();
    */
    //	m_mTotalPixels = pixelId_mon.size();
    //test pixelID
    //cout<<"the pixel number in the all monitor: "<<m_mTotalPixels<<endl;
    //	cout<<"the pixel number in the monitors: "<<m_mTotalPixels<<endl;

    //	bPixelNum = int(m_bTotalPixels/bankNum);
    //    mPixelNum = int(m_mTotalPixels/monitorNum);


    //std::vector<float> m_eventTOF = getEventTOF_bank(file);
    //cout<<"the total size of tof is "<<m_eventTOF.size()<<endl;

    //std::vector<uint64_t> m_eventPulseTime = getEventPulseTime_bank(file);
    //cout<<"the size of pulse time in banks is "<<m_eventPulseTime.size()<<endl;
    //std::vector<uint64_t> m_eventPulseTime_mon = getEventPulseTime_mon(file);
    //cout<<"the size of pulse time in mon is "<<m_eventPulseTime_mon.size()<<endl;

    //std::multimap<uint32_t, std::pair<float, uint64_t>> eventData_bank = getEventData_bank(file);
    //std::multimap<uint32_t, std::pair<float, uint64_t>> eventData_mon = getEventData_mon(file);
    //cout<<"the size of m_eventData_bank is: "<<eventData_bank.size()<<endl;
    //	cout<<"the event data in monitor pid = -1 is: "<<eventData_mon.count(0)<<endl;
    /*
    if(m_loadEvent)
    {
        if (hasEventData(file))
        {
            cout<<"has Event data\n";
        }else
        {
            throw std::runtime_error(" No event data in the NXS file!");
        }
    }
    */	
    // create workspace
    //	MatrixWorkspace_sptr ws_mon = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D", 1, m_numBins + 1, m_numBins));
    /*
    EventWorkspace_sptr ws_bank(new EventWorkspace);
    EventWorkspace_sptr ws_mon(new EventWorkspace);
    WorkspaceGroup_sptr wksp_group(new WorkspaceGroup);
    //     setProperty("OutputWorkspace", WS);
    wksp_group->addWorkspace(ws_bank);
    wksp_group->addWorkspace(ws_mon);
    // set outputworkspace
    setProperty("OutputWorkspace",boost::dynamic_pointer_cast<Workspace>(wksp_group));
    //setProperty("OutputWorkspace", ws);

    cout<<"successful create event WS"<<endl;
    loadEventData(file, ws_bank, m_bTotalPixels, m_timeBin);
    ws_bank->mutableRun().setStartAndEndTime(start_time, end_time);
    ws_bank->setTitle(m_sampleInfo.name);
    ws_bank->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");
    ws_bank->setYUnit("Counts");

    loadMonitorData(file, ws_mon, m_timeBin);
    //	loadMonitorData(file, ws_mon, m_mTotalPixels, m_timeBin);
    ws_mon->mutableRun().setStartAndEndTime(start_time, end_time);
    ws_mon->setTitle(m_sampleInfo.name);
    ws_mon->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");
    ws_mon->setYUnit("Counts");
    */

    // Load log of source, chopper and SE    
    //prog->doReport("Loading time log");
    //loadLog(file, ws_bank);

    //    }else
    //    {
    //         if (hasHistogramData(file))
    //        {
    //            cout<<"has Histogram data\n";
    //        }else
    //        {
    //            throw std::runtime_error(" No histgram data in the NXS file!");
    //        }
    //    }

    // load monitor
    //   MatrixWorkspace_sptr ws_mon = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D",1, m_numBins + 1, m_numBins));
    //  setProperty("OutputWorkspace",boost::dynamic_pointer_cast<Workspace>(ws_mon));
    /*
       loadMonitorData(file, ws_mon);
       ws_mon->mutableRun().setStartAndEndTime(start_time, end_time);
       ws_mon->setTitle(m_sampleInfo.name);
       ws_mon->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");
       ws_mon->setYUnit("Counts");
       */
    /*
       prog->doReport("Creating workspace");
    // Bank workspace
    MatrixWorkspace_sptr WS1 = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D", m_bTotalPixels, m_bNumBins + 1, m_bNumBins));
    // Monitor workspace
    MatrixWorkspace_sptr WS2 = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D", m_mTotalPixels, m_bNumBins + 1, m_bNumBins));
    // make group
    WorkspaceGroup_sptr wksp_group(new WorkspaceGroup);
    wksp_group->addWorkspace(WS1);
    wksp_group->addWorkspace(WS2);
    // set outputworkspace
    setProperty("OutputWorkspace",boost::dynamic_pointer_cast<Workspace>(wksp_group));
    loadHistData(file, WS1, m_bTotalPixels, m_timeBin);
    WS1->mutableRun().setStartAndEndTime(start_time, end_time);
    WS1->setTitle(m_sampleInfo.name);
    WS1->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");
    WS1->setYUnit("Counts");
    */

    // Load log of source, chopper and SE    
    //        prog->doReport("Loading time log");
    //        loadLog(file, WS1);
    //    }

    /*
    // test monitor data
    std::vector<std::vector<uint32_t> > m_monitorData = getMonitorData(file);
    cout<<"the size of mon data is: "<<m_monitorData.size()<<endl;
    std::vector<uint32_t> mon_tof = m_monitorData[0];
    std::vector<uint32_t> mon_count = m_monitorData[1];
    cout<<"the size of tof in mon data is: "<<mon_tof.size()<<endl;
    uint32_t dtof, dcount;
    dtof = mon_tof[1];
    dcount = mon_count[1];
    cout<<"the second tof||counts:  "<<dtof<<"||"<<dcount<<endl;
    */  




    //----------------------------------------- test---------------------------------------------------------
    /*
       EventWorkspace_sptr WS(new EventWorkspace);
       setProperty("OutputWorkspace", WS);
       WS->initialize(bPixelNum,1,1);
       WS->mutableRun().setStartAndEndTime(start_time, end_time);
    // load info into workspace 
    // CSNSsample m_sampleInfo = getSampleInfo(file);
    WS->setTitle(m_sampleInfo.name);
    // set units in X and Y axis 
    WS->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");    
    WS->setYUnit("Counts");
    */
    // test getSpectraMapping
    //   std::map<uint32_t, uint32_t> spectraMapping_bank = getSpectraMapping(file, true);
    //   std::cout<<"total number of spectrum in banks is: "<<spectraMapping_bank.size()<<std::endl;
    //    std::cout<<"total number of spectrum in monitor is: "<<spectraMapping_monitor.size()<<std::endl;
    //    if (spectraMapping_bank.count(510))
    //    std::cout<<"Mapping of detector ID 510  to spectrum no in bank is: "<<spectraMapping_bank[510]<<std::endl;
    //    if (spectraMapping_bank.count(515))
    //    std::cout<<"Mapping of detector ID 515  to spectrum no in bank is: "<<spectraMapping_bank[515]<<std::endl;
    //    if (spectraMapping_bank.count(1025))
    //    std::cout<<"Mapping of detector ID 1025  to spectrum no in bank is: "<<spectraMapping_bank[1025]<<std::endl;
    //    if (spectraMapping_monitor.count(255))
    //    std::cout<<"Mapping of detector ID 255  to spectrum no in monitor is: "<<spectraMapping_monitor[255]<<std::endl;
    //    if (spectraMapping_monitor.count(260))
    //    std::cout<<"Mapping of detector ID 260  to spectrum no in monitor  is: "<<spectraMapping_monitor[260]<<std::endl;
    /*	
    // test getEventData
    //	uint64_t sTime = start_time.totalNanoseconds();
    std::multimap<int64_t,std::pair<float, uint64_t>> m_eventData = getEventData(file, true);
    cout<<"the size of m_eventData_bank is: "<<m_eventData.size()<<endl;
    */
    //typedef std::multimap<uint32_t,std::pair<uint64_t, float>>::iterator it_multimap;
    //	std::multimap<uint32_t,std::pair<uint64_t, float>>::iterator specNo;
    //	std::multimap<uint32_t,std::pair<uint64_t, float>>::iterator end0;
    // event data in banks
    //	float tof1;
    //	uint64_t ptime1;
    //	specNo = m_eventData.find(12387);
    //	for (size_t j = 0; j != m_eventData.count(12387); j++, specNo++)
    //	{
    //	ptime1 = specNo->second.first;
    //	tof1 = specNo->second.second;
    //	cout<<"detector ID || m_pulsetime || m_tof "<<specNo->first<<" || "<<ptime1<<" || "<<tof1<<endl;
    //	}



    /*

       for(specNo = m_eventData.begin(); specNo != m_eventData.end(); specNo++)
       {
       if(specNo->first == 12387){
       tof1=specNo->second.second;
       ptime1=specNo->second.first;
       std::cout<<"spectra No "<<specNo->first<<" in bank has "<<m_eventData.count(12387)<<" event datas!"<<std::endl;
       cout<<"They are: pulse time || tof: "<<ptime1<<" || "<<tof1<<endl;
       }else
       continue;
       }
       */
    //it_multimap _specNo_bank1 = m_eventData_bank.find(12387);
    //  if(_specNo_bank1 != m_eventData_bank.end()){
    //  while((_specNo_bank0 = m_eventData_bank.find(12387, _specNo_bank0))!= m_eventData_bank.end()){
    //    tof1=_specNo_bank0->second.second;
    //  ptime1=_specNo_bank0->second.first;
    //  std::cout<<"spectra No "<<_specNo_bank0->first<<" in bank has "<<m_eventData_bank.count(12387)<<" event datas!"<<std::endl;
    //  cout<<"They are: pulse time || tof: "<<ptime1<<" || "<<tof1<<endl;
    //  _specNo_bank0++;}
    //    it_multimap _specNo_bank2 = m_eventData_bank.find(515);
    //    if(_specNo_bank2 != m_eventData_bank.end())
    //        std::cout<<"spectra No "<<_specNo_bank2->first<<" in bank has "<<m_eventData_bank.count(515)<<" event datas! "<<std::endl;

    //    it_multimap _specNo_bank3 = m_eventData_bank.find(1025);
    //    if(_specNo_bank3 != m_eventData_bank.end())
    //        std::cout<<"spectra No "<<_specNo_bank3->first<<" in bank has "<<m_eventData_bank.count(1025)<<" event datas! "<<std::endl;




    //load instrument
    //    std::vector<std::vector<float>>pixelOffset = getPixelOffset(file,bankNames[0]);
    //    m_xOffset = pixelOffset[0];
    //    m_yOffset = pixelOffset[1];
    //    std::cout<<"xoffsets size is "<<m_xOffset.size()<<std::endl;

    // close nxs file
    file->close();
    delete file;
    }

} // namespace DataHandling
} // namespace Mantid
