//----------------------------------------------------------------------
// Includes
//----------------------------------------------------------------------
#include "MantidDataHandling/LoadCSNSNexus.h"
#include "MantidDataHandling/EventWorkspaceCollection.h"

#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_real.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/shared_array.hpp>
#include <boost/scoped_array.hpp>
#include <boost/function.hpp>
#include <functional>

#include "MantidKernel/ArrayProperty.h"
#include "MantidKernel/ThreadPool.h"
#include "MantidKernel/UnitFactory.h"
#include "MantidKernel/ThreadSchedulerMutexes.h"
#include "MantidKernel/BoundedValidator.h"
#include "MantidKernel/VisibleWhenProperty.h"
#include "MantidKernel/TimeSeriesProperty.h"
#include "MantidGeometry/Instrument/RectangularDetector.h"
#include "MantidAPI/FileProperty.h"
#include "MantidAPI/MemoryManager.h"
#include "MantidAPI/RegisterFileLoader.h"
#include "MantidAPI/SpectrumDetectorMapping.h"
#include "MantidKernel/Timer.h"

using std::cout;
using std::endl;
using std::map;
using std::string;
using std::vector;

namespace Mantid
{
namespace DataHandling
{

DECLARE_NEXUS_FILELOADER_ALGORITHM(LoadCSNSNexus)

using namespace Kernel;
using namespace Geometry;
using namespace API;
using namespace DataObjects;

LoadCSNSNexus::LoadCSNSNexus()
    : m_bNumPixels(0),m_mNumPixels(0), pulseTimes(0), m_numBins(0), m_spec_min(0),
      m_spec_max(0), m_fileMutex() {}

//-------------------------------------------------------------------------------------------------
/// Initialisation method.
void LoadCSNSNexus::init()
{
    declareProperty(
        new FileProperty("Filename", "", FileProperty::Load, {".nxs"}),
        "The name of the NeXus file to load");
    declareProperty(new WorkspaceProperty<API::Workspace>("OutputWorkspace", "",
                    Direction::Output),
                    "The name of the Workspace2D to create.");
    declareProperty(
        new PropertyWithValue<string>("Entryname", "csns",Direction::Input), "Optional: Name of entry (default csns)");

    declareProperty(
        new ArrayProperty<string>("Bankname", Direction::Input),
        "Optional: A comma-separated list of bank to read. (default all banks)");
    declareProperty(
        new ArrayProperty<string>("Monitorname", Direction::Input),
        "Optional: A comma-separated list of monitor to read. (default all monitors)");
    declareProperty(
        new PropertyWithValue<bool>("Loadevent", true, Direction::Input),
        "Default true: load event data mode, false: Load histogram data.");

    auto mustBePositive = boost::make_shared<BoundedValidator<int>>();
    mustBePositive->setLower(1);
    declareProperty(
        new PropertyWithValue<specid_t>("SpectrumMin", 1, mustBePositive),
        "The index number of the first spectrum to read.  Only used if spectrum_max is set.");
    declareProperty(
        new PropertyWithValue<specid_t>("SpectrumMax", Mantid::EMPTY_INT(),
                                        mustBePositive),
        "The number of the last spectrum to read. Only used if explicitly set.");
}

//-------------------------------------------------------------------------------------------------
/**
 * Return the confidence with with this algorithm can load the file
 * @param descriptor A descriptor for the file
 * @returns An integer specifying the confidence level. 0 indicates it will not
 * be used
 */
int LoadCSNSNexus::confidence(Kernel::NexusDescriptor &descriptor) const
{
    int confidence(0);
    if (descriptor.pathOfTypeExists("/csns", "NXentry"))
    {
        const bool hasEventData = descriptor.classTypeExists("NXevent_data");
        const bool hasHistogramData = descriptor.classTypeExists("NXdata");
        if (hasHistogramData && hasEventData)
            // Event data = this is event NXS
            confidence = 20;
        else if (hasHistogramData && !hasEventData)
            // No event data = this is the one
            confidence = 80;
        else if (!hasHistogramData && hasEventData)
            // Histogram Data only
            confidence = 40;
        else
            // No data ?
            confidence = 10;
    }
    return confidence;
}

//-------------------------------------------------------------------------------------------------
/**
 * Return entry name
 * @param[in] file :: nxs file
 * @returns newEntry :: entry name
 */
std::string LoadCSNSNexus::getEntryName(::NeXus::File *file)
{
    std::string newEntry="";

    std::map<std::string, std::string> entries = file->getEntries();

    if (entries.empty())
    {
        throw std::runtime_error("No entries in the NXS file!");
    }
    else if (entries.find("csns") != entries.end())
    {
        newEntry = "csns";
    }
    else
    {
        newEntry = entries.begin()->first;
    }

    return newEntry;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the pixel number
* @param[in] file :: nxs file
* @return numPixels
*/
size_t LoadCSNSNexus::getNumPixels(::NeXus::File *file, bool bm)
{
    size_t numPixels=0;
    size_t newPixels=0;
    size_t numBank = m_bankNames.size();
    size_t numMonitor = m_monitorNames.size();

    std::map<std::string, std::string> entries;
    std::map<std::string, std::string> subentries;
    std::map<std::string, std::string>::iterator it;
    
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    if (bm)
    {
        if (numBank==0)
        {
            // Look for all the banks
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                if (name.substr(0, 4) == "bank")
                {
                    file->openGroup(name, it->second);
                    subentries = file->getEntries();

                    if (subentries.find("pixel_id") != subentries.end())
                    {
                        file->openData("pixel_id");
                        std::vector<int64_t> dims = file->getInfo().dims;
                        file->closeData();

                        if (!dims.empty())
                        {
                            newPixels = 1;
                            for (size_t i = 0; i < dims.size(); i++)
                                newPixels *= dims[i];
                            // cout<<"newPixels is: "<<newPixels<<endl;
                            numPixels += newPixels;
                        }
                        // cout<<"numPixels is:"<<numPixels<<endl;
                    }
                    else
                    {
                        throw std::runtime_error("No pixels in the NXS file!");
                    }
                    file->closeGroup();
                }
            }
        }else{
            // Look for all the banks
            for (auto j = m_bankNames.begin(); j != m_bankNames.end(); ++j)
            {
                std::string name = *j;
                file->openGroup(name, "NXdetector");
                
                entries = file->getEntries();
                if (entries.find("pixel_id") != entries.end())
                {
                    file->openData("pixel_id");
                    std::vector<int64_t> dims = file->getInfo().dims;
                    file->closeData();
                    if (!dims.empty())
                    {
                        newPixels = 1;
                        for (size_t i = 0; i < dims.size(); i++)
                            newPixels *= dims[i];
                        numPixels += newPixels;
                    }
                }
                else
                {
                    throw std::runtime_error("No pixels in the banks file!");
                }
                file->closeGroup();
            }
        }
    }
    else
    {
        if (numMonitor==0)
        {
            // Look for all the banks
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;

                if (name.substr(0, 7) == "monitor")
                {
                    // cout<<"name is:"<<name<<endl;
                    // file->openGroup(name, "NXdetector");
                    file->openGroup(name, it->second);

                    subentries = file->getEntries();

                    if (subentries.find("pixel_id") != subentries.end())
                    {
                        file->openData("pixel_id");
                        std::vector<int64_t> dims = file->getInfo().dims;
                        file->closeData();

                        if (!dims.empty())
                        {
                            newPixels = 1;
                            for (size_t i = 0; i < dims.size(); i++)
                                newPixels *= dims[i];
                            // cout<<"newPixels is: "<<newPixels<<endl;
                            numPixels += newPixels;
                        }
                        // cout<<"numPixels is:"<<numPixels<<endl;
                    }
                    else
                    {
                        throw std::runtime_error("No pixels in the NXS file!");
                    }
                    file->closeGroup();
                }
            }
        }else{
            // Look for all the monitors
            for (auto j = m_monitorNames.begin(); j != m_monitorNames.end(); ++j)
            {
                std::string name = *j;
                file->openGroup(name, "NXdetector");
                entries = file->getEntries();

                if (entries.find("pixel_id") != entries.end())
                {
                    file->openData("pixel_id");
                    std::vector<int64_t> dims = file->getInfo().dims;
                    file->closeData();
                    if (!dims.empty())
                    {
                        newPixels = 1;
                        for (size_t i = 0; i < dims.size(); i++)
                            newPixels *= dims[i];
                        numPixels += newPixels;
                    }
                }
                else
                {
                    throw std::runtime_error("No pixels in the monitors file!");
                }
                file->closeGroup();
            }
        }
    }

    file->closeGroup();
    file->closeGroup();
    return numPixels;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the bin number
* @param[in] filen :: nxs file
* @return numBins :: bins of TOF
*/
size_t LoadCSNSNexus::getNumBins(::NeXus::File *file, bool bm)
{
    size_t numBins=0;
    size_t numBank = m_bankNames.size();
    size_t numMonitor = m_monitorNames.size();

    std::map<std::string, std::string> entries;
    std::map<std::string, std::string> subentries;
    std::map<std::string, std::string>::iterator it;

    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    if (bm)
    {
        if (numBank==0)
        {
            // Look for all the banks
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;

                if (name.substr(0, 4) == "bank")
                {
                    file->openGroup(name, it->second);
                    subentries = file->getEntries();

                    if (subentries.find("time_of_flight") != subentries.end())
                    {
                        file->openData("time_of_flight");
                        std::vector<int64_t> dims = file->getInfo().dims;
                        file->closeData();
                        numBins = dims[0];
                        break;
                    }else{
                        throw std::runtime_error("No pixels in the NXS file!");
                    }
                    file->closeGroup();
                    break;
                }
             }
        }else{
             for (auto j = m_bankNames.begin(); j != m_bankNames.end(); ++j)
             {
                std::string name = *j;
                file->openGroup(name, "NXdetector");

                entries = file->getEntries();
                if (entries.find("time_of_flight") != entries.end())
                {
                    file->openData("time_of_flight");
                    std::vector<int64_t> dims = file->getInfo().dims;
                    file->closeData();
                    numBins = dims[0];
                }else{
                    throw std::runtime_error("No pixels in the NXS file!");
                }
                file->closeGroup();
                break;
              }
            }
    }else{
         if (numMonitor==0)
        {
            // Look for all the banks
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;

                if (name.substr(0, 7) == "monitor")
                {
                    file->openGroup(name, it->second);
                    subentries = file->getEntries();

                    if (subentries.find("time_of_flight") != subentries.end())
                    {
                        file->openData("time_of_flight");
                        std::vector<int64_t> dims = file->getInfo().dims;
                        file->closeData();
                        numBins = dims[0];
                    }else{
                        throw std::runtime_error("No pixels in the NXS file!");
                    }
                file->closeGroup();
                break;
                }
             }
        }else{
             for (auto j = m_monitorNames.begin(); j != m_monitorNames.end(); ++j)
             {
                std::string name = *j;
                file->openGroup(name, "NXdetector");

                entries = file->getEntries();
                if (entries.find("time_of_flight") != entries.end())
                {
                    file->openData("time_of_flight");
                    std::vector<int64_t> dims = file->getInfo().dims;
                    file->closeData();
                    numBins = dims[0];
                }else{
                    throw std::runtime_error("No pixels in the NXS file!");
                }
                file->closeGroup();
                break;
                }
            }
}
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    return numBins;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the event data
* @param[in] file :: nxs file
* @param[in] startTime :: start time in nxs file
* @param[in] bm :: true is bank and false is monitor
* @return eventData :: event data include event_pixel_id, pulse_time,tof.
* pulse time unit is 
*/
//std::vector<std::vector< std::pair<float,float> > > LoadCSNSNexus::getEventData(::NeXus::File *file, const std::string &bankName)
std::multimap<int, std::pair<double, int64_t>> LoadCSNSNexus::getEventData(::NeXus::File *file, int64_t startTime, bool bm)
{
    size_t numBank = m_bankNames.size();
    size_t numMonitor = m_monitorNames.size();
    std::multimap<int, std::pair<double, int64_t>> eventData;
    int specNo;
    int64_t _pTime;
    std::map<std::string, std::string> entries;
    std::map<std::string, std::string>::iterator it;

    std::vector<int> _pixelId;
    std::vector<float> _timeOfFlight;
    std::vector<float> _pulseTime;
    std::map<int, int> spectraMapping = getSpectraMapping(file, bm);
    std::pair<double, int64_t> _pTOF;

    file->openGroup(m_entryName, "NXentry");
    file->openGroup("event_data", "NXcollection");
 
    if (bm)
    {
        if (numBank==0)
        {
            // Look for all the banks
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                if (name.substr(0, 4) == "bank")
                {
                    file->openGroup(name, it->second);
                    int _num = atoi(name.substr(4,1).c_str());
                    file->readData("event_pixel_id",_pixelId);
                    file->readData("event_time_of_flight",_timeOfFlight);
                    file->readData("event_pulse_time",_pulseTime);
                    std::vector<double>_tof(_timeOfFlight.begin(),_timeOfFlight.end());
                    for (size_t i=0; i<_pixelId.size(); ++i)
                    {
                        specNo = spectraMapping[_pixelId[i]+512*_num];
                        _pTime = startTime + int64_t(_pulseTime[i]*1000);                 
                        _pTOF = std::make_pair(_tof[i], _pTime);
                        eventData.insert(std::multimap<int, std::pair<double, int64_t>>::value_type(specNo,_pTOF));
                    }
                    file->closeGroup();
                }
            }
        }else{
            // Look for specilized banks
            for (auto j = m_bankNames.begin(); j != m_bankNames.end(); ++j)
            {
                std::string name = *j;
                file->openGroup(name, "NXevent_data");
                int _num = atoi(name.substr(4,1).c_str());
                file->readData("event_pixel_id",_pixelId);
                file->readData("event_time_of_flight",_timeOfFlight);
                file->readData("event_pulse_time",_pulseTime);
                std::vector<double>_tof(_timeOfFlight.begin(),_timeOfFlight.end());
                    for (size_t i=0; i<_pixelId.size(); ++i)
                    {
                        specNo = spectraMapping[_pixelId[i]+512*_num];
                        _pTime = startTime + int64_t(_pulseTime[i]*1000);
                        _pTOF = std::make_pair(_tof[i], _pTime);
                        eventData.insert(std::multimap<int, std::pair<double, int64_t>>::value_type(specNo,_pTOF));
                    }
                file->closeGroup();
            }
            }
    }else{
         if (numMonitor==0)
        {
            // Look for all the monitors
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;

                if (name.substr(0, 7) == "monitor")
                {
                    file->openGroup(name, it->second);  
                    int _num = atoi(name.substr(7,1).c_str());
                    file->readData("event_pixel_id",_pixelId);
                    file->readData("event_time_of_flight",_timeOfFlight);
                    file->readData("event_pulse_time",_pulseTime);
                    std::vector<double>_tof(_timeOfFlight.begin(),_timeOfFlight.end());
                    for (size_t i=0; i<_pixelId.size(); ++i)
                    {
                        specNo = spectraMapping[_pixelId[i]+256*_num];
                        _pTime = startTime + int64_t(_pulseTime[i]*1000);
                        _pTOF = std::make_pair(_tof[i], _pTime);
                        eventData.insert(std::multimap<int, std::pair<double, int64_t>>::value_type(specNo,_pTOF));
                    }
                    file->closeGroup();
                }
            }
        }else{
             for (auto j = m_monitorNames.begin(); j != m_monitorNames.end(); ++j)
             {
                std::string name = *j;
                file->openGroup(name, "NXevent_data");
                int _num = atoi(name.substr(7,1).c_str());
                file->readData("event_pixel_id",_pixelId);
                file->readData("event_time_of_flight",_timeOfFlight);
                file->readData("event_pulse_time",_pulseTime);
                std::vector<double>_tof(_timeOfFlight.begin(),_timeOfFlight.end());
                for (size_t i=0; i<_pixelId.size(); ++i)
                {
                    specNo = spectraMapping[_pixelId[i]+256*_num];
                    _pTime = startTime + int64_t(_pulseTime[i]*1000);
                    _pTOF = std::make_pair(_tof[i], _pTime);
                    eventData.insert(std::multimap<int, std::pair<double, int64_t>>::value_type(specNo,_pTOF));
                }
                file->closeGroup();
             }
             }
        }
        file->closeGroup();
        file->closeGroup();
        return eventData;
}

//-------------------------------------------------------------------------------------------------
/**
* Return Bank list NX/entry/instrument/bank*
* @param[in] file :: nxs file
* @return bankNames :: bank list
*/
std::vector<std::string> LoadCSNSNexus::getBankNames(::NeXus::File *file)
{
    std::vector<std::string> bankNames;

    // Open the default data group 'entry'
    file->openGroup(m_entryName, "NXentry");
    // Also pop into the instrument
    file->openGroup("instrument", "NXinstrument");

    // Look for all the banks
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        if (it->first.substr(0,4)=="bank" && it->second=="NXdetector")
        {
            bankNames.push_back(it->first);
        }
    }

    file->closeGroup();
    file->closeGroup();
    return bankNames;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the pixel offset
* @param[in] file :: nxs file
* @return pixelOffset :: [x,y]
*/

std::vector<std::vector<float> > LoadCSNSNexus::getPixelOffset(::NeXus::File *file, const std::string &bankName)
{
    std::vector<std::vector<float> > pixelOffset;

//    size_t numBank = m_bankNames.size();
//    size_t numMonitor = m_monitorNames.size();

    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup(bankName, "NXdetector");

    std::vector<float> _tmp;
    file->readData("x_pixel_offset",_tmp);
    pixelOffset.push_back(_tmp);
    file->readData("y_pixel_offset",_tmp);
    pixelOffset.push_back(_tmp);

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();

    return pixelOffset;
}
//-------------------------------------------------------------------------------------------------
/**
 * Creates a time series property from the currently opened log entry. It is
 * assumed to
 * have been checked to have a time field and the value entry's name is given as
 * an argument
 * @param file :: A reference to the file handle
 * @param prop_name :: The name of the property
 * @returns A pointer to a new property containing the time series
 */
Kernel::Property *LoadCSNSNexus::createTimeSeries(::NeXus::File *file,
        const std::string &prop_name) const
{
    file->openData("time");
    //----- Start time is an ISO8601 string date and time. ------
    std::string start;
    try{
        file->getAttr("start", start);}
    catch (::NeXus::Exception &){
        start="2017-12-31T00:00:00+08:00";}
    // Convert to date and time
    Kernel::DateAndTime start_time = Kernel::DateAndTime(start);
    std::string time_units;
    file->getAttr("units", time_units);
    if (time_units.compare("second") < 0 && time_units != "s" &&
            time_units != "minutes") // Can be s/second/seconds/minutes
    {
        file->closeData();
        throw ::NeXus::Exception("Unsupported time unit '" + time_units + "'");
    }
    //--- Load the seconds into a double array ---
    std::vector<double> time_double;
    try
    {
        file->getDataCoerce(time_double);
    }
    catch (::NeXus::Exception &e)
    {
        g_log.warning() << "Log entry's time field could not be loaded: '"
                        << e.what() << "'.\n";
        file->closeData();
        throw;
    }
    file->closeData(); // Close time data
    g_log.debug() << "   done reading \"time\" array\n";
    // Convert to seconds if needed
    if (time_units == "minutes")
    {
        std::transform(time_double.begin(), time_double.end(), time_double.begin(),
                       std::bind2nd(std::multiplies<double>(), 60.0));
    }
    // Now the values: Could be a string, int or double
    file->openData("value");
    // Get the units of the property
    std::string value_units("");
    try
    {
        file->getAttr("units", value_units);
    }
    catch (::NeXus::Exception &)
    {
        // Ignore missing units field.
        value_units = "";
    }
    // Now the actual data
    ::NeXus::Info info = file->getInfo();
    // Check the size
    if (size_t(info.dims[0]) != time_double.size())
    {
        file->closeData();
        throw ::NeXus::Exception("Invalid value entry for time series");
    }
    if (file->isDataInt()) // Int type
    {
        std::vector<int> values;
        try
        {
            file->getDataCoerce(values);
            file->closeData();
        }
        catch (::NeXus::Exception &)
        {
            file->closeData();
            throw;
        }
        // Make an int TSP
        auto tsp = new TimeSeriesProperty<int>(prop_name);
        tsp->create(start_time, time_double, values);
        tsp->setUnits(value_units);
        g_log.debug() << "   done reading \"value\" array\n";
        return tsp;
    }
    else if (info.type == ::NeXus::CHAR)
    {
        std::string values;
        const int64_t item_length = info.dims[1];
        try
        {
            const int64_t nitems = info.dims[0];
            const int64_t total_length = nitems * item_length;
            boost::scoped_array<char> val_array(new char[total_length]);
            file->getData(val_array.get());
            file->closeData();
            values = std::string(val_array.get(), total_length);
        }
        catch (::NeXus::Exception &)
        {
            file->closeData();
            throw;
        }
        // The string may contain non-printable (i.e. control) characters, replace
        // these
        std::replace_if(values.begin(), values.end(), iscntrl, ' ');
        auto tsp = new TimeSeriesProperty<std::string>(prop_name);
        std::vector<DateAndTime> times;
        DateAndTime::createVector(start_time, time_double, times);
        const size_t ntimes = times.size();
        for (size_t i = 0; i < ntimes; ++i)
        {
            std::string value_i =
                std::string(values.data() + i * item_length, item_length);
            tsp->addValue(times[i], value_i);
        }
        tsp->setUnits(value_units);
        g_log.debug() << "   done reading \"value\" array\n";
        return tsp;
    }
    else if (info.type == ::NeXus::FLOAT32 || info.type == ::NeXus::FLOAT64)
    {
        std::vector<double> values;
        try
        {
            file->getDataCoerce(values);
            file->closeData();
        }
        catch (::NeXus::Exception &)
        {
            file->closeData();
            throw;
        }
        auto tsp = new TimeSeriesProperty<double>(prop_name);
        tsp->create(start_time, time_double, values);
        tsp->setUnits(value_units);
        g_log.debug() << "   done reading \"value\" array\n";
        return tsp;
    }
    else
    {
        throw ::NeXus::Exception(
            "Invalid value type for time series. Only int, double or strings are "
            "supported");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the pixel_id
* @param[in] file :: nxs file
* @param[in] bm :: true means bank, false means monitor
* @return pixelId :: pixel_id
*/

std::vector<uint32_t> LoadCSNSNexus::getPixelId(::NeXus::File *file, bool bm)
{
    size_t numBank = m_bankNames.size();
    size_t numMonitor = m_monitorNames.size();
    int _num;
    std::vector<int> _tmp;
    std::vector<int> tmp;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");

    std::map<std::string, std::string> entries;
    std::map<std::string, std::string>::iterator it;

    if(bm)
    {
         if (numBank==0)
        {
            // Look for all the banks
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                if (name.substr(0, 4) == "bank")
                {
                    file->openGroup(name, it->second);
                    _num = atoi(name.substr(4,1).c_str());
                    file->readData("pixel_id",_tmp);
                    int tsize = int(_tmp.size());
                    for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                        tmp.push_back((*i)+tsize*_num);
                    file->closeGroup();
                }
            }
        }else{
           // Look for spcilized banks
            for (auto j = m_bankNames.begin(); j != m_bankNames.end(); ++j)
            {
                std::string name = *j;
                _num = atoi(name.substr(4,1).c_str());
                file->openGroup(name, "NXdetector");
                file->readData("pixel_id",_tmp);
                int tsize = int(_tmp.size());
                for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                    tmp.push_back((*i)+tsize*_num);
                file->closeGroup();
            }
            }
    }else{
        if (numMonitor==0)
        {
            // Look for all the monitors
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                if (name.substr(0, 7) == "monitor")
                {
                    file->openGroup(name, it->second);
                    _num = atoi(name.substr(7,1).c_str());
                    file->readData("pixel_id",_tmp);
                    int tsize = int(_tmp.size());
                    for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                        tmp.push_back((*i)+tsize*_num);
                    file->closeGroup();
                }
            }
        }else{
            // Look for spcilized monitors
            for (auto j = m_monitorNames.begin(); j != m_monitorNames.end(); ++j)
            {
                std::string name = *j;
                _num = atoi(name.substr(7,1).c_str());
                file->openGroup(name, "NXdetector");
                file->readData("pixel_id",_tmp);
                int tsize = int(_tmp.size());
                for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                    tmp.push_back((*i)+tsize*_num);
                file->closeGroup();
            }
            }
        }
    std::vector<uint32_t>pixelId(tmp.begin(), tmp.end());
    sort(pixelId.begin(),pixelId.end());
    file->closeGroup();
    file->closeGroup();
    return pixelId;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the spectraMapping
* @param[in] file :: nxs file
* @param[in] bankName :: bank name
* @return spectraMapping :: spectrum and detecotId mapping
*/
std::map<int, int> LoadCSNSNexus::getSpectraMapping(::NeXus::File *file, bool bm)
{
    std::map<int, int> spectraMapping;
    std::vector<uint32_t> _PId = LoadCSNSNexus::getPixelId(file, bm);
    std::vector<int> PId(_PId.begin(), _PId.end());
    int p = 1;
    for (auto i = PId.begin(); i != PId.end(); ++i)
    {
        spectraMapping.insert(std::map<int, int>::value_type(*i,p));
        p++;
    }
    return spectraMapping;
}


//-------------------------------------------------------------------------------------------------
/**
 * Load an log entry
 * @param file:: NeXus file
 * @param workspace :: A pointer to the workspace to store the logs
 */
void LoadCSNSNexus::loadLog( ::NeXus::File *file, boost::shared_ptr<API::MatrixWorkspace> workspace) const
//void LoadCSNSNexus::loadLog( ::NeXus::File *file, EventWorkspace_sptr workspace) const
{
    file->openGroup(m_entryName, "NXentry");
    try{file->openGroup("logs", "NXcollection");}
    catch (::NeXus::Exception &e){
        g_log.warning() <<"Read logs: " << e.what()<<"\n";
        file->closeGroup();
        return;
    }
    // Validate the NX log class.
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        std::string logName=it->first;
        if (it->second == "NXlog")
        {
            file->openGroup(it->first, "NXlog");
            std::map<std::string, std::string> subEntries = file->getEntries();
            if ((subEntries.find("value") == subEntries.end()) || (subEntries.find("time") == subEntries.end()))
            {   g_log.warning() << "Invalid NXlog entry " << logName
                                << " found. Did not contain 'value' and 'time'.\n";}
            else{
                try{
                    if (!(workspace->run().hasProperty(logName)))
                    {
                        g_log.debug()<<logName<<"\n";
                        Kernel::Property *logValue = createTimeSeries(file, logName);
                        workspace->mutableRun().addProperty(logValue, true);
                    }
                }
                catch (::NeXus::Exception &e)
                {
                    g_log.warning() << "NXlog entry " << logName
                                    << " gave an error when loading:'" << e.what() << "'.\n";
                }
            }
            file->closeGroup();
        }//if
        if (it->second == "NXcollection")
        {
            std::string groupName=it->first;
            file->openGroup(it->first, "NXcollection");

            std::map<std::string, std::string> subEntries = file->getEntries();
            std::map<std::string, std::string>::iterator subit;
            for (subit = subEntries.begin(); subit != subEntries.end(); ++subit)
            {
                if (subit->second == "NXlog")
                {
                    file->openGroup(subit->first, "NXlog");
                    std::map<std::string, std::string> subEntries = file->getEntries();
                    if ((subEntries.find("value") == subEntries.end()) || (subEntries.find("time") == subEntries.end()))
                    {
                        g_log.warning() << "Invalid NXlog entry " << subit->first
                                        << " found. Did not contain 'value' and 'time'.\n";
                    }
                    else
                    {
                        try
                        {
                            if (!(workspace->run().hasProperty(subit->first)))
                            {
                                g_log.debug()<<subit->first<<"\n";
                                Kernel::Property *logValue = createTimeSeries(file, groupName+"_"+subit->first);
                                workspace->mutableRun().addProperty(logValue, true);
                            }
                        }
                        catch (::NeXus::Exception &e)
                        {
                            g_log.warning() << "NXlog entry " << subit->first
                                            << " gave an error when loading:'" << e.what() << "'.\n";
                        }
                    }
                    file->closeGroup();
                }
            }
            file->closeGroup();
        }//if
    }//for
    file->closeGroup();
    file->closeGroup();
    std::cout<<"current path: "<<file->getPath()<<std::endl;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the CSNS NeXus definiton version
* @param[in] filename :: nxs file name
* @return version :: CSNS NeXus definition version
*/
std::string LoadCSNSNexus::getVersion(::NeXus::File *file)
{
    std::string version="";

    file->openGroup(m_entryName, "NXentry");

    std::map<std::string, std::string> entries = file->getEntries();

    if (entries.find("version") != entries.end())
    {
        file->readData("version", version);
    }
    else
    {
        throw std::runtime_error("No version in the NXS file!");
    }

    file->closeGroup();
    return version;
}

//-------------------------------------------------------------------------------------------------
/**
* Return true if has event data
* @param[in] file :: nxs file
* @return hasEventData :: true|false
*/
bool LoadCSNSNexus::hasEventData(::NeXus::File *file)
{
    bool hasEventData=false;

    file->openGroup(m_entryName, "NXentry");
    try
    {
        file->openGroup("event_data", "NXcollection");
        std::map<std::string, std::string> entries = file->getEntries();
        std::map<std::string, std::string>::iterator it;
        for (it = entries.begin(); it != entries.end(); ++it)
        {
            if (it->second=="NXevent_data")
            {
                hasEventData = true;
                break;
            }
        }
        file->closeGroup();

    }
    catch(::NeXus::Exception)
    {
        hasEventData=false;
    }

    file->closeGroup();
    return hasEventData;
}

//-------------------------------------------------------------------------------------------------
/**
* Return true if has histogram data
* @param[in] file :: nxs file
* @return hasEventData :: true|false
*/
bool LoadCSNSNexus::hasHistogramData(::NeXus::File *file)
{
    bool hasHistogramData=false;

    file->openGroup(m_entryName, "NXentry");
    try
    {
        file->openGroup("histogram_data", "NXcollection");
        std::map<std::string, std::string> entries = file->getEntries();
        std::map<std::string, std::string>::iterator it;
        for (it = entries.begin(); it != entries.end(); ++it)
        {
            if (it->second=="NXdata")
            {
                hasHistogramData = true;
                break;
            }
        }
        file->closeGroup();

    }
    catch(::NeXus::Exception)
    {
        hasHistogramData=false;
    }

    file->closeGroup();
    return hasHistogramData;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the start time of experiment
* @param[in] filename :: nxs file name
* @return startTime :: start time
*/
Kernel::DateAndTime LoadCSNSNexus::getStartTime(::NeXus::File *file)
{
    std::string startTime;
    file->openGroup(m_entryName, "NXentry");
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("start_time") != entries.end())
    {
        file->readData("start_time", startTime);
    }
    else
    {
        throw std::runtime_error("No start time in the NXS file!");
    }
    file->closeGroup();
    return Kernel::DateAndTime(startTime);;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the end time of experiment
* @param[in] filename :: nxs file name
* @return endTime :: end time
*/
Kernel::DateAndTime LoadCSNSNexus::getEndTime(::NeXus::File *file)
{
    std::string endTime;
    file->openGroup(m_entryName, "NXentry");
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("end_time") != entries.end())
    {
        file->readData("end_time", endTime);
    }
    else
    {
        throw std::runtime_error("No end time in the NXS file!");
    }
    file->closeGroup();
    return Kernel::DateAndTime(endTime);
}
//-------------------------------------------------------------------------------------------------
/**
* Return sample information
* @param[in] filename :: nxs file name
* @return sampleInfo :: sample information
*/
CSNSsample LoadCSNSNexus::getSampleInfo(::NeXus::File *file)
{
    CSNSsample sampleInfo;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument","NXinstrument");
    file->openGroup("sample","NXsample");
    file->readData("name",sampleInfo.name);
    file->readData("type",sampleInfo.type);
    file->readData("situation",sampleInfo.situation);
    file->readData("chemical_formula",sampleInfo.chemical_formula);
    file->readData("density",sampleInfo.density);
    file->readData("mass",sampleInfo.mass);
    file->readData("distance",sampleInfo.distance);

    file->openGroup("geometry","NXgeometry");
    file->openGroup("shape","NXshape");
    file->readData("shape",sampleInfo.shape);
    file->readData("size",sampleInfo.size);

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    return sampleInfo;   
}

//-------------------------------------------------------------------------------------------------
/**
* load event data in workspace
* @param[in] numPixels :: pixel numbers
* @param[in] numBins :: time of flight numbers
*/
/*
template <typename T>
void LoadCSNSNexus::loadEventData(::NeXus::File *file, T ws)
{
    Kernel::DateAndTime start_time = getStartTime(file);
    int64_t sTime = start_time.totalNanoseconds();
    std::multimap<int,std::pair<int64_t, double>> m_eventData = getEventData(file, sTime, true);
    double tof;
    int64_t pulseTime; 
//    int numpixels = static_cast<int>(numPixels);
//    int numbins = static_cast<int>(numBins);
    int p = 0;
//    size_t pix = m_eventData->first.size();    
    std::multimap<int,std::pair<int64_t, double>>::iterator it;
    for (it = m_eventData.begin(); it != m_eventData.end(); it++){
       // ws->getEventList(p).addDetectorID(it->first-1);
       // ws->getEventList(p).setSpectrumNo(it->first);
        for (size_t j = 0; j != m_eventData.count(it->first); j++){
            pulseTime = it->second.first;
            tof = it->second.second;
            ws->getEventList(p) += TofEvent(tof, pulseTime);            
        }
       // ws->getEventList(p).addDetectorID(p);
        ws->getEventList(p).setSpectrumNo(it->first);
        p++;
    }
}
*/

void LoadCSNSNexus::exec()
{
    // The input properties
    std::string filename = getPropertyValue("Filename");
    m_entryName = getPropertyValue("Entryname");
    m_bankNames = getProperty("Bankname");
    m_monitorNames = getProperty("Monitorname");

    bool m_loadEvent = getProperty("Loadevent");
    if (m_loadEvent)
    {
        g_log.debug() <<"Event data will be loaded.\n";
    }
    else
    {
        g_log.debug() <<"Histogram data will be loaded.\n";
    }
    m_spec_min = getProperty("SpectrumMin");
    m_spec_max = getProperty("SpectrumMax");

    auto prog = new Progress(this, 0.0, 1.0, 10);
    prog->doReport("Start to load Nexus");
    // Open nxs file
    auto file = new ::NeXus::File(filename);
    // Find the entry name we want.
    std::string newEntryName = getEntryName(file);
    if (newEntryName != m_entryName)
    {
        throw std::runtime_error(m_entryName+" not found in the NXS file!");
    }
    std::vector<std::string> bankNames= getBankNames(file);
    g_log.debug() <<bankNames.size() << " banks found." << std::endl;
    prog->setNumSteps(bankNames.size() + 5);
    // Get CSNS Nexus Version
    std::string m_version = getVersion(file);
    g_log.debug()<<"CSNS Nexus version is: " <<m_version<<"\n";
/*
    // has Event data
    if (hasEventData(file))
    {
        cout<<"has Event data\n";
    }
    // has Histogram data
    if (hasHistogramData(file))
    {
        cout<<"has Histogram data\n";
    }
*/
    prog->doReport("Counting pixels");
    m_bNumPixels = getNumPixels(file, true);
    m_mNumPixels = getNumPixels(file, false);
    cout<<"pixel number in banks is: "<<m_bNumPixels<<endl;
    //cout<<"pixel number in monitors is: "<<m_mNumPixels<<endl;
    g_log.debug() <<m_bNumPixels <<" pixels found."<< std::endl;

    prog->doReport("Counting TOF bins");
    m_bNumBins = getNumBins(file, true);
    m_mNumBins = getNumBins(file, false);
    g_log.debug() <<m_numBins <<" TOF bins found."<< std::endl;
    //std::cout<<"num of bin in banks are: "<<m_bNumBins<<std::endl;
    //std::cout<<"num of bin in monitors are: "<<m_mNumBins<<std::endl;

    // test getPixelId
    //std::vector<uint32_t> pixelId_bank = getPixelId(file, true);
    //std::vector<uint32_t> pixelId_monitor = getPixelId(file, false);
    //cout<<"the pixelId size banks are: "<<pixelId_bank.size()<<endl;
    //cout<<"the pixelId size monitors are: "<<pixelId_monitor.size()<<endl;
    //cout<<"the bank pixelId in 2 is: "<<pixelId_bank[2]<<endl;
    //cout<<"the bank pixelId in 512 is: "<<pixelId_bank[512]<<endl;
    //cout<<"the monitor pixelId in 2 is: "<<pixelId_monitor[2]<<endl;
    //cout<<"the monitor pixelId in 256 is: "<<pixelId_monitor[256]<<endl;


    // create event workspace
    EventWorkspace_sptr WS(new EventWorkspace);
    setProperty("OutputWorkspace", WS);
    // Get start time and end time
    Kernel::DateAndTime start_time = getStartTime(file);
    Kernel::DateAndTime end_time = getEndTime(file);
    //int64_t sTime = start_time.totalNanoseconds();

    WS->mutableRun().setStartAndEndTime(start_time, end_time);
    // load info into workspace 
    CSNSsample m_sampleInfo = getSampleInfo(file);
    WS->setTitle(m_sampleInfo.name);

    //load event data
   // loadEventData(file, WS);
    WS->initialize(m_bNumPixels,1,1);
    int64_t sTime = start_time.totalNanoseconds();
    std::multimap<int,std::pair<double, int64_t>> m_eventData = getEventData(file, sTime, true);
    double m_tof;
    int64_t m_pulseTime;
    std::multimap<int,std::pair<double, int64_t>>::iterator it;

    for (int i = 0; i != m_bNumPixels; i++)
    {
        it = m_eventData.find(i+1);
        for (size_t j = 0; j != m_eventData.count(i+1); j++, it++)
        {        
            m_pulseTime = it->second.second;
            m_tof = it->second.first;
            WS->getEventList(i) += TofEvent(m_tof, m_pulseTime);
        } 
        WS->getEventList(i).setSpectrumNo(i+1);
    }
/* 
    //test and confirmed the correction of m_eventData
    typedef std::multimap<int, std::pair<double, int64_t>>::iterator it_multimap;
    it_multimap i1 = m_eventData.find(344);
    for (int k = 0; k != m_eventData.count(344); k++, i1++)
    {
        cout<<"the spcetrum No is "<<i1->first<<" and the tof is: "<<i1->second.first<<endl;
    }     
*/    
     // set units in X and Y axis 
    WS->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");
    WS->setYUnit("Counts");


/*
    std::map<int, int> spectraMapping_bank = getSpectraMapping(file, true);
    std::map<int, int> spectraMapping_monitor = getSpectraMapping(file, false);
    std::cout<<"total number of spectrum in banks is: "<<spectraMapping_bank.size()<<std::endl;
    std::cout<<"total number of spectrum in monitor is: "<<spectraMapping_monitor.size()<<std::endl;
    std::cout<<"Mapping of detector ID 1060  to spectrum no in bank is: "<<spectraMapping_bank[1060]<<std::endl;
    std::cout<<"Mapping of detector ID 255  to spectrum no in monitor  is: "<<spectraMapping_monitor[255]<<std::endl;
    int elem_bank = spectraMapping_bank[1060];
    int elem_monitor = spectraMapping_monitor[255];

    std::multimap<int,std::pair<int64_t,double>> m_eventData_bank = getEventData(file, sTime, true);
    //std::multimap<int,std::pair<int64_t,double>> m_eventData_monitor = getEventData(file, sTime, false);
    //typedef std::multimap<int, std::pair<int64_t,double>>::iterator it_multimap;
    //it_multimap _specNo_bank = m_eventData_bank.find(elem_bank);
    //it_multimap _specNo_monitor = m_eventData_monitor.find(elem_monitor);
    //std::cout<<"spectra No "<<_specNo_bank->first<<" in bank has "<<m_eventData_bank.count(elem_bank)<<" event datas! They are: "<<std::endl;
    //std::cout<<"spectra No "<<_specNo_monitor->first<<" in monitor has "<<m_eventData_monitor.count(elem_monitor)<<" event datas! They are: "<<std::endl;
*/

    
    //load instrument
//    std::vector<std::vector<float>>pixelOffset = getPixelOffset(file,bankNames[0]);
//    m_xOffset = pixelOffset[0];
//    m_yOffset = pixelOffset[1];
//    std::cout<<"xoffsets size is "<<m_xOffset.size()<<std::endl;

/*
    //create 2DWorkspace
    // Start with a dummy WS just to hold the logs and load the instrument
    prog->doReport("Creating workspace");
    // Bank workspace
    MatrixWorkspace_sptr WS1 = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D", m_bNumPixels, m_bNumBins + 1, m_bNumBins));
    // Monitor workspace
    MatrixWorkspace_sptr WS2 = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D", m_mNumPixels, m_mNumBins + 1, m_mNumBins));

*/

    // Load log of source, chopper and SE
    prog->doReport("Loading time log");
    loadLog(file, WS);
    //std::cout<<"current path: "<<file->getPath()<<std::endl;


/*
    // Set to the output
    //setProperty("OutputWorkspace", boost::dynamic_pointer_cast<MatrixWorkspace>(WS));
    //setProperty("OutputWorkspace", WS);
    WorkspaceGroup_sptr wksp_group(new WorkspaceGroup);

    wksp_group->addWorkspace(WS1);
    wksp_group->addWorkspace(WS2);
    setProperty("OutputWorkspace",
                boost::dynamic_pointer_cast<Workspace>(wksp_group));
    delete prog;
*/


    // close nxs file
    file->close();
    delete file;
}

} // namespace DataHandling
} // namespace Mantid
