#include "MantidAPI/FileProperty.h"
#include "MantidAPI/MatrixWorkspace.h"
#include "MantidAPI/RegisterFileLoader.h"
#include "MantidDataHandling/LoadEventNexus.h"
#include "MantidDataHandling/LoadCSNSNexus.h"
#include "MantidKernel/ArrayProperty.h"
#include "MantidKernel/BoundedValidator.h"
#include "MantidKernel/cow_ptr.h"
#include <nexus/NeXusFile.hpp>
#include <boost/algorithm/string/detail/classification.hpp>
#include <boost/algorithm/string/split.hpp>

namespace Mantid {
namespace DataHandling {

DECLARE_NEXUS_FILELOADER_ALGORITHM(LoadCSNSNexus)

using namespace Kernel;
using namespace API;
using namespace DataObjects;

LoadCSNSNexus::LoadCSNSNexus()
    : m_numPixels(0), m_signalNo(0), pulseTimes(0), m_numBins(0), m_spec_min(0),
      m_spec_max(0), m_dataField(""), m_axisField(""), m_xUnits(""),
      m_fileMutex(), m_assumeOldFile(false) {}

//-------------------------------------------------------------------------------------------------
/// Initialisation method.
void LoadCSNSNexus::init() {
  declareProperty(
      new FileProperty("Filename", "", FileProperty::Load, {".nxs"}),
      "The name of the NeXus file to load");
  declareProperty(new WorkspaceProperty<MatrixWorkspace>("OutputWorkspace", "",
                                                         Direction::Output),
                  "The name of the Workspace2D to create.");
  declareProperty("Signal", 1,
                  "Number of the signal to load from the file. Default is 1 = "
                  "time_of_flight.\n"
                  "Some NXS files have multiple data fields giving binning in "
                  "other units (e.g. d-spacing or momentum).\n"
                  "Enter the right signal number for your desired field.");
  auto mustBePositive = boost::make_shared<BoundedValidator<int>>();
  mustBePositive->setLower(1);
  declareProperty(
      new PropertyWithValue<specid_t>("SpectrumMin", 1, mustBePositive),
      "The index number of the first spectrum to read.  Only used if\n"
      "spectrum_max is set.");
  declareProperty(
      new PropertyWithValue<specid_t>("SpectrumMax", Mantid::EMPTY_INT(),
                                      mustBePositive),
      "The number of the last spectrum to read. Only used if explicitly\n"
      "set.");
}

//-------------------------------------------------------------------------------------------------
/**
 * Return the confidence with with this algorithm can load the file
 * @param descriptor A descriptor for the file
 * @returns An integer specifying the confidence level. 0 indicates it will not
 * be used
 */
int LoadCSNSNexus::confidence(Kernel::NexusDescriptor &descriptor) const {
  int confidence(0);
  if (descriptor.pathOfTypeExists("/csns", "NXentry")) {
    const bool hasEventData = descriptor.classTypeExists("NXevent_data");
    const bool hasHistogramData = descriptor.classTypeExists("NXdata");
    if (hasHistogramData && hasEventData)
      // Event data = this is event NXS
      confidence = 20;
    else if (hasHistogramData && !hasEventData)
      // No event data = this is the one
      confidence = 80;
    else
      // No data ?
      confidence = 10;
  }
  return confidence;
}

// Function object for remove_if STL algorithm
namespace {
// Check the numbers supplied are not in the range and erase the ones that are
struct range_check {
  range_check(specid_t min, specid_t max, detid2index_map id_to_wi)
      : m_min(min), m_max(max), m_id_to_wi(id_to_wi) {}

  bool operator()(specid_t x) {
    specid_t wi = static_cast<specid_t>((m_id_to_wi)[x]);
    return (wi + 1 < m_min || wi + 1 > m_max);
  }

private:
  specid_t m_min;
  specid_t m_max;
  detid2index_map m_id_to_wi;
};
}

void LoadCSNSNexus::exec() {
  // The input properties
  std::string filename = getPropertyValue("Filename");
  m_signalNo = getProperty("Signal");
  m_spec_min = getProperty("SpectrumMin");
  m_spec_max = getProperty("SpectrumMax");

  auto prog = new Progress(this, 0.0, 1.0, 10);
  prog->doReport("Start to load Nexus");
  g_log.debug() << "Start to load Nexus"<< std::endl;

  m_numPixels=512;
  m_numBins=5000;
  // Start with a dummy WS just to hold the logs and load the instrument
  MatrixWorkspace_sptr WS = WorkspaceFactory::Instance().create(
      "Workspace2D", m_numPixels, m_numBins + 1, m_numBins);
  
  // Set to the output
  setProperty("OutputWorkspace", WS);
  
  delete prog;
}

} // namespace DataHandling
} // namespace Mantid
