#include "MantidDataHandling/LoadCSNSNexus.h"
#include "MantidDataHandling/EventWorkspaceCollection.h"

#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_real.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/shared_array.hpp>
#include <boost/scoped_array.hpp>
#include <boost/function.hpp>
#include <functional>

#include "MantidKernel/ArrayProperty.h"
#include "MantidKernel/ThreadPool.h"
#include "MantidKernel/UnitFactory.h"
#include "MantidKernel/ThreadSchedulerMutexes.h"
#include "MantidKernel/BoundedValidator.h"
#include "MantidKernel/VisibleWhenProperty.h"
#include "MantidKernel/TimeSeriesProperty.h"
#include "MantidGeometry/Instrument/RectangularDetector.h"
#include "MantidAPI/FileProperty.h"
#include "MantidAPI/MemoryManager.h"
#include "MantidAPI/RegisterFileLoader.h"
#include "MantidAPI/SpectrumDetectorMapping.h"
#include "MantidKernel/Timer.h"

using std::cout;
using std::endl;
using std::map;
using std::string;
using std::vector;

namespace Mantid
{
namespace DataHandling
{

DECLARE_NEXUS_FILELOADER_ALGORITHM(LoadCSNSNexus)

using namespace Kernel;
using namespace Geometry;
using namespace API;
using namespace DataObjects;

LoadCSNSNexus::LoadCSNSNexus()
    : m_bTotalPixels(0),m_mTotalPixels(0), pulseTimes(0), m_numBins(0), m_spec_min(0),
      m_spec_max(0), m_fileMutex() {}

//-------------------------------------------------------------------------------------------------
/// Initialisation method.
void LoadCSNSNexus::init()
{
    declareProperty(
        new FileProperty("Filename", "", FileProperty::Load, {".nxs"}),
        "The name of the NeXus file to load");
    declareProperty(new WorkspaceProperty<API::Workspace>("OutputWorkspace", "",
                    Direction::Output),
                    "The name of the Workspace2D to create.");
    declareProperty(
        new PropertyWithValue<string>("Entryname", "csns",Direction::Input), "Optional: Name of entry (default csns)");

    declareProperty(
        new ArrayProperty<string>("Bankname", Direction::Input),
        "Optional: A comma-separated list of bank to read. (default all banks)");
    declareProperty(
        new ArrayProperty<string>("Monitorname", Direction::Input),
        "Optional: A comma-separated list of monitor to read. (default all monitors)");
    declareProperty(
        new PropertyWithValue<bool>("Loadevent", true, Direction::Input),
        "Default true: load event data mode, false: Load histogram data.");

    auto mustBePositive = boost::make_shared<BoundedValidator<int>>();
    mustBePositive->setLower(1);
    declareProperty(
        new PropertyWithValue<specid_t>("SpectrumMin", 1, mustBePositive),
        "The index number of the first spectrum to read.  Only used if spectrum_max is set.");
    declareProperty(
        new PropertyWithValue<specid_t>("SpectrumMax", Mantid::EMPTY_INT(),
                                        mustBePositive),
        "The number of the last spectrum to read. Only used if explicitly set.");
}

//-------------------------------------------------------------------------------------------------
/**
 * Return the confidence with with this algorithm can load the file
 * @param descriptor A descriptor for the file
 * @returns An integer specifying the confidence level. 0 indicates it will not
 * be used
 */
int LoadCSNSNexus::confidence(Kernel::NexusDescriptor &descriptor) const
{
    int confidence(0);
    if (descriptor.pathOfTypeExists("/csns", "NXentry"))
    {
        const bool hasEventData = descriptor.classTypeExists("NXevent_data");
        const bool hasHistogramData = descriptor.classTypeExists("NXdata");
        if (hasHistogramData && hasEventData)
            // Event data = this is event NXS
            confidence = 20;
        else if (hasHistogramData && !hasEventData)
            // No event data = this is the one
            confidence = 80;
        else if (!hasHistogramData && hasEventData)
            // Histogram Data only
            confidence = 40;
        else
            // No data ?
            confidence = 10;
    }
    return confidence;
}
//-------------------------------------------------------------------------------------------------
/**
 * Return the entry name
 * @param[in] file :: nxs file
 * @returns newEntry :: entry name
 */
std::string LoadCSNSNexus::getEntryName(::NeXus::File *file)
{
    std::string newEntry="";
    std::map<std::string, std::string> entries = file->getEntries();//(name,NXclass)
    if (entries.empty())
    {
        throw std::runtime_error("No entries in the NXS file!");
    }
    else if (entries.find("csns") != entries.end())
    {
        newEntry = "csns"; //if find csns,enwEntry is csns.
    }
    else
    {
        newEntry = entries.begin()->first; //get entry name not csns
    }
    return newEntry;
}
//-------------------------------------------------------------------------------------------------
/**
* Return Bank ot Monitor list NX/entry/instrument/bank* or monitor*
* @param[in] file :: nxs file
* @param[in] bm :: true means bank and false means monitor.
* @return detectorNames :: bank or monitor list
*/
std::vector<std::string> LoadCSNSNexus::getDetectorNames(::NeXus::File *file, bool bm)
{
    std::vector<std::string> detectorNames;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    if(bm)
    {
        for (it = entries.begin(); it != entries.end(); ++it)
        {
            if (it->first.substr(0,4)=="bank" && it->second=="NXdetector")
                detectorNames.push_back(it->first);
        }
    }else
    {
        for (it = entries.begin(); it != entries.end(); ++it)
        {
            if (it->first.substr(0,7)=="monitor" && it->second=="NXdetector")
                detectorNames.push_back(it->first);
        }
    }
    file->closeGroup();
    file->closeGroup();
    return detectorNames;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the CSNS NeXus definiton version
* @param[in] filename :: nxs file name
* @return version :: CSNS NeXus definition version
*/
std::string LoadCSNSNexus::getVersion(::NeXus::File *file)
{
    std::string version="";
    file->openGroup(m_entryName, "NXentry");
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("version") != entries.end())
    {
        file->readData("version", version);
    }
    else
    {
        throw std::runtime_error("No version in the NXS file!");
    }
    file->closeGroup();
    return version;
}
//-------------------------------------------------------------------------------------------------
/**
* Return sample information
* @param[in] filename :: nxs file name
* @return sampleInfo :: sample information
*/
CSNSsample LoadCSNSNexus::getSampleInfo(::NeXus::File *file)
{
//must consider if the data not exist!!!
    CSNSsample sampleInfo;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument","NXinstrument");
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("sample") != entries.end())
    {
        file->openGroup("sample", "NXsample");
    }
    else
    {
        throw std::runtime_error("No sample in the NXS file!");
    }
    std::map<std::string, std::string> entries2 = file->getEntries();
    if (entries2.find("name") != entries2.end())
    {
        file->readData("name",sampleInfo.name);
    }else
    {
        throw std::runtime_error("No sample name in the NXS file!");
    }
    if (entries2.find("type") != entries2.end())
    {
        file->readData("type",sampleInfo.type);
    }else
    {
        throw std::runtime_error("No sample type in the NXS file!");
    }
    if (entries2.find("situation") != entries2.end())
    {
        file->readData("situation",sampleInfo.situation);
    }else
    {
        throw std::runtime_error("No sample situation in the NXS file!");
    }
    if (entries2.find("chemical_formula") != entries2.end())
    {
        file->readData("chemical_formula",sampleInfo.chemical_formula);
    }else
    {
        throw std::runtime_error("No sample chemical formula in the NXS file!");
    }
    if (entries2.find("density") != entries2.end())
    {
        file->readData("density",sampleInfo.density);
    }else
    {
        throw std::runtime_error("No sample density in the NXS file!");
    }
    if (entries2.find("mass") != entries2.end())
    {
        file->readData("mass",sampleInfo.mass);
    }else
    {
        throw std::runtime_error("No sample mass in the NXS file!");
    }
    if (entries2.find("distance") != entries2.end())
    {
        file->readData("distance",sampleInfo.distance);
    }else
    {
        throw std::runtime_error("No sample distance in the NXS file!");
    }
    if (entries2.find("geometry") != entries2.end())
    {
        file->openGroup("geometry","NXgeometry");
        std::map<std::string, std::string> entries3 = file->getEntries();
        if(entries3.find("shape") != entries3.end())
        {
            file->openGroup("shape", "NXshape");
            std::map<std::string, std::string> entries4 = file->getEntries();
            if(entries4.find("shape") != entries4.end())
            {
                file->readData("shape",sampleInfo.shape);
            }else
            {
                throw std::runtime_error("No sample shape in the NXS file!");
            }
            if(entries4.find("size") != entries4.end())
            {
                file->readData("size",sampleInfo.size);
            }else
            {
                throw std::runtime_error("No sample size in the NXS file!");
            }
            
        }else
        {
            throw std::runtime_error("No sample shape entry in the NXS file!");
        }
    }else
    {
        throw std::runtime_error("No sample geometry in the NXS file!");
    }
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    return sampleInfo;   
}
//-------------------------------------------------------------------------------------------------
/**
* Return the start time of experiment
* @param[in] filename :: nxs file name
* @return startTime :: start time
*/
Kernel::DateAndTime LoadCSNSNexus::getStartTime(::NeXus::File *file)
{
    std::string startTime;
    file->openGroup(m_entryName, "NXentry");
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("start_time") != entries.end())
    {
        file->readData("start_time", startTime);
    }
    else
    {
        throw std::runtime_error("No start time in the NXS file!");
    }
    file->closeGroup();
    return Kernel::DateAndTime(startTime);;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the end time of experiment
* @param[in] filename :: nxs file name
* @return endTime :: end time
*/
Kernel::DateAndTime LoadCSNSNexus::getEndTime(::NeXus::File *file)
{
    std::string endTime;
    file->openGroup(m_entryName, "NXentry");
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("end_time") != entries.end())
    {
        file->readData("end_time", endTime);
    }
    else
    {
        throw std::runtime_error("No end time in the NXS file!");
    }
    file->closeGroup();
    return Kernel::DateAndTime(endTime);
}
//-------------------------------------------------------------------------------------------------
/**
* Return the pixel_id of the specified banks or monitors.
* @param[in] file :: nxs file
* @param[in] bm :: true means bank, false means monitor
* @return pixelId :: pixel_id from all (or choose) banks (or monitros).
*/

std::vector<uint32_t> LoadCSNSNexus::getPixelId(::NeXus::File *file, bool bm)
{
    size_t numBank = m_bankNames.size();
    size_t numMonitor = m_monitorNames.size();
    int _num;
    std::vector<int> _tmp;
    std::vector<int> tmp;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");

    std::map<std::string, std::string> entries;
    std::map<std::string, std::string>::iterator it;

    if(bm)
    {
         if (numBank==0)
        {
            // Look for all the banks
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                if (name.substr(0, 4) == "bank")
                {
                    file->openGroup(name, it->second);
                    _num = atoi(name.substr(4,1).c_str()); //get the series number of the monitor
                    file->readData("pixel_id",_tmp);
                    int tsize = int(_tmp.size());
                    for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                        tmp.push_back((*i)+tsize*_num); //make sure different num monitor has different pixID
                    file->closeGroup();
                }
            }
        }else{
           // Look for spcilized banks
            for (auto j = m_bankNames.begin(); j != m_bankNames.end(); ++j)
            {
                std::string name = *j;
                _num = atoi(name.substr(4,1).c_str());
                file->openGroup(name, "NXdetector");
                file->readData("pixel_id",_tmp);
                int tsize = int(_tmp.size());
                for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                    tmp.push_back((*i)+tsize*_num);
                file->closeGroup();
            }
            }
    }else{
        if (numMonitor==0)
        {
            // Look for all the monitors
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                if (name.substr(0, 7) == "monitor")
                {
                    file->openGroup(name, it->second);
                    _num = atoi(name.substr(7,1).c_str());
                    file->readData("pixel_id",_tmp);
                    int tsize = int(_tmp.size());
                    for (auto i = _tmp.begin(); i != _tmp.end(); ++i) 
                        tmp.push_back((*i)+tsize*_num); 
                    file->closeGroup();
                }
            }
        }else{
            // Look for spcilized monitors
            for (auto j = m_monitorNames.begin(); j != m_monitorNames.end(); ++j)
            {
                std::string name = *j;
                _num = atoi(name.substr(7,1).c_str());
                file->openGroup(name, "NXdetector");
                file->readData("pixel_id",_tmp);
                int tsize = int(_tmp.size());
                for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                    tmp.push_back((*i)+tsize*_num);
                file->closeGroup();
            }
            }
        }
    std::vector<uint32_t>pixelId(tmp.begin(), tmp.end());
    sort(pixelId.begin(),pixelId.end());
    file->closeGroup();
    file->closeGroup();
    return pixelId;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the time bins, banks and monitors have the same time bins 
* @param[in] filen :: nxs file
* @return TOF :: time of flight
*/
std::vector<uint32_t> LoadCSNSNexus::getTimeBin(::NeXus::File *file)
{
    std::vector<int> _tmp;
  //  std::vector<uint32_t> timeBin;
    std::map<std::string, std::string> entries;
    std::map<std::string, std::string> entries2;
    std::map<std::string, std::string>::iterator it;

    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
//    file->openGroup("bank0", "NXdetector");
//    cout<<"successful open bank group"<<endl;
//    file->readData("time_of_flight",_tmp);
//    std::vector<uint32_t>timeBin(_tmp.begin(), _tmp.end());
//    file->closeGroup();
    

    entries = file->getEntries();
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        std::string name = it->first;
       // if (name.substr(0, 4) == "bank" || name.substr(0,7)=="monitor")
        if (name.substr(0, 4) == "bank")
        {
            file->openGroup(name, it->second);
            cout<<"successful open bank group"<<endl;
            entries2 = file->getEntries();
            if (entries2.find("time_of_flight") != entries2.end())
            {
                //file->readData("time_of_flight",timeBin);
                file->readData("time_of_flight",_tmp);
                cout<<"successful read timebin"<<endl;
            }
            else
            {
                throw std::runtime_error("No time_of_flight in the NXS file!");
            }
            file->closeGroup();
            break;
        }
    }
    std::vector<uint32_t>timeBin(_tmp.begin(), _tmp.end());


    file->closeGroup();
    file->closeGroup();
    return timeBin;
}
//-------------------------------------------------------------------------------------------------
/**
* Return true if has event data
* @param[in] file :: nxs file
* @return hasEventData :: true|false
*/
bool LoadCSNSNexus::hasEventData(::NeXus::File *file)
{
    bool hasEventData=false;
    file->openGroup(m_entryName, "NXentry");
    try
    {
        file->openGroup("event_data", "NXcollection");
        std::map<std::string, std::string> entries = file->getEntries();
        std::map<std::string, std::string>::iterator it;
        for (it = entries.begin(); it != entries.end(); ++it)
        {
            if (it->second=="NXevent_data")
            {
                hasEventData = true;
                break;
            }
        }
        file->closeGroup();
    }
    catch(::NeXus::Exception)
    {
        hasEventData=false;
    }
    file->closeGroup();
    return hasEventData;
}
//-------------------------------------------------------------------------------------------------
/**
* Return true if has histogram data
* @param[in] file :: nxs file
* @return hasEventData :: true|false
*/
bool LoadCSNSNexus::hasHistogramData(::NeXus::File *file)
{
    bool hasHistogramData=false;
    file->openGroup(m_entryName, "NXentry");
    try
    {
        file->openGroup("histogram_data", "NXcollection");
        std::map<std::string, std::string> entries = file->getEntries();
        std::map<std::string, std::string>::iterator it;
        for (it = entries.begin(); it != entries.end(); ++it)
        {
            if (it->second=="NXdata")
            {
                hasHistogramData = true;
                break;
            }
        }
        file->closeGroup();
    }
    catch(::NeXus::Exception)
    {
        hasHistogramData=false;
    }
    file->closeGroup();
    return hasHistogramData;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the spectraMapping
* @param[in] file :: nxs file
* @param[in] bm :: true means bank, false means monitor
* @return spectraMapping :: mapping(pixel_id, spectrumNo from 1)
*/
//std::map<int, int> LoadCSNSNexus::getSpectraMapping(::NeXus::File *file, bool bm)
std::map<uint32_t, uint32_t> LoadCSNSNexus::getSpectraMapping(::NeXus::File *file, bool bm)
{
    std::map<uint32_t, uint32_t> spectraMapping;
    std::vector<uint32_t> PId = LoadCSNSNexus::getPixelId(file, bm);
    uint32_t p = 1;
    for (auto i = PId.begin(); i != PId.end(); ++i)
    {
        spectraMapping.insert(std::map<uint32_t, uint32_t>::value_type(*i,p));
        p++;
    }
    return spectraMapping;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the pixel positions
* @param[in] file :: nxs file
* @param[in] bankName :: which bank you want to obtain the pixel positions
* @return pixelOffset :: [x,y]
*/
std::vector<std::vector<float> > LoadCSNSNexus::getPixelOffset(::NeXus::File *file, const std::string &bankName)
{
    std::vector<std::vector<float> > pixelOffset;
//    size_t numBank = m_bankNames.size();
//    size_t numMonitor = m_monitorNames.size();
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup(bankName, "NXdetector");

    std::vector<float> _tmp;
    file->readData("x_pixel_offset",_tmp);
    pixelOffset.push_back(_tmp);
    file->readData("y_pixel_offset",_tmp);
    pixelOffset.push_back(_tmp);

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();

    return pixelOffset;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the value in histogram data
* @param[in] file :: nxs file
* @param[in] bm :: true for banks false for monitors
* @return value :: value in nexus file is 3-dimension data.
*/
/*
std::vector<uint32_t> LoadCSNSNexus::getValue(::NeXus::File *file, bool bm)
{
    size_t numBank = m_bankNames.size();
    size_t numMonitor = m_monitorNames.size();
    std::vector<int> _tmp;
    std::vector<int> tmp;

    std::map<std::string, std::string> entries;
    std::map<std::string, std::string>::iterator it;

    file->openGroup(m_entryName, "NXentry");
    file->openGroup("histogram_data", "NXcollection");

    if (bm)
    {
        if (numBank==0)
        {
            // Look for all the banks
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                if (name.substr(0, 4) == "bank")
                {
                    file->openGroup(name, it->second);
                    file->readData("value", _tmp);
                    for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                        tmp.push_back((*i));
                    file->closeGroup();
                }
            }
        }else{
            // Look for all the banks
            for (auto j = m_bankNames.begin(); j != m_bankNames.end(); ++j)
            {
                std::string name = *j;
                file->openGroup(name, "NXdata");
                file->readData("value", _tmp);
                for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                    tmp.push_back((*i));
                file->closeGroup();
            }
            }

    }else
    {
        if (numMonitor==0)
        {
            // Look for all the banks
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                if (name.substr(0, 7) == "monitor")
                {
                    file->openGroup(name, it->second);
                    file->readData("value", _tmp);
                    for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                        tmp.push_back((*i));
                    file->closeGroup();
                }
            }
        }else{
            // Look for all the monitors
            for (auto j = m_monitorNames.begin(); j != m_monitorNames.end(); ++j)
            {
                std::string name = *j;
                file->openGroup(name, "NXdata");
                file->readData("value", _tmp);
                for (auto i = _tmp.begin(); i != _tmp.end(); ++i)
                    tmp.push_back((*i));
                file->closeGroup();
            }
        }
    }
    std::vector<uint32_t>value(tmp.begin(), tmp.end());
    file->closeGroup();
    file->closeGroup();
    return value;
}
*/
//-------------------------------------------------------------------------------------------------
/**
* Return the event data
* @param[in] file :: nxs file
* @param[in] startTime :: start time in nxs file
* @param[in] pNum :: the number of pixels in one bank.
* @param[in] bm :: true is bank and false is monitor
* @return eventData :: event data include (pixel_id, pulse_time, tof). As a multimap, one pixel_id has several pairs of (pulsetime, tof).
* pulse time unit is nanoseconds
*/
std::multimap<uint32_t, std::pair<uint64_t, float>> LoadCSNSNexus::getEventData(::NeXus::File *file, uint64_t startTime, int pNum)
{
    size_t numBank = m_bankNames.size();
  //  size_t numMonitor = m_monitorNames.size();
    std::multimap<uint32_t, std::pair<uint64_t, float>> eventData;
    uint32_t specNo;
    uint64_t _pTime;
    std::map<std::string, std::string> entries;
    std::map<std::string, std::string>::iterator it;
 
    std::vector<int> _tmp1;
    std::vector<int> _index;
    std::vector<int64_t> _tmp2;
    std::vector<int64_t> _fulltmp2;
    std::vector<int>::iterator it_index;
    std::vector<int64_t>::iterator it_tmp2;

    std::vector<float> _timeOfFlight;
    std::map<uint32_t, uint32_t> spectraMapping = getSpectraMapping(file, true);
    std::pair<uint64_t, float> _pTOF;
    
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("event_data", "NXcollection");
    if (numBank==0)
        {
            // Look for all the banks
            entries = file->getEntries();
            for (it = entries.begin(); it != entries.end(); ++it)
            {
                std::string name = it->first;
                if (name.substr(0, 4) == "bank")
                {
                    file->openGroup(name, it->second);
                    int _num = atoi(name.substr(4,1).c_str());
                    //file->readData("event_pixel_id",_pixelId);
                    file->readData("event_pixel_id",_tmp1);
                    file->readData("event_time_of_flight",_timeOfFlight);
                    cout<<"successful in read time of flight"<<endl;
                // rearrange pulse time
                    file->readData("event_index", _index);                    
                    file->readData("event_pulse_time",_tmp2);
                    uint32_t n1 = 0;
                    it_tmp2 = _tmp2.begin();
                    for(it_index=_index.begin(); it_index != _index.end(); ++it_index)
                    {
                        if(*it_index == 0){
                            _fulltmp2.push_back(*it_tmp2);
                            n1++;
                        }else{
                                int delta = (*it_index)-(*(it_index-1));
                                if(delta != 0){
                                    for(int i = 0; i!= delta; i++){
                                        _fulltmp2.push_back(*(it_tmp2+n1));
                                    }
                                    n1++;
                               //     cout<<"when delta !=0, n1="<<n1<<endl;
                                }else {     n1++;}
                             }
                    }
                    cout<<"the size of pulsetime: "<<_fulltmp2.size()<<endl;
                   std::vector<uint32_t> _pixelId(_tmp1.begin(), _tmp1.end());
                   std::vector<uint64_t> _pulseTime(_fulltmp2.begin(), _fulltmp2.end());
                    cout<<"successful in put data into _pixelId and _pulseTime"<<endl;
                    std::vector<float>_tof(_timeOfFlight.begin(),_timeOfFlight.end());
                    for (size_t i=0; i<_pixelId.size(); ++i)
                    {
                        specNo = spectraMapping[_pixelId[i]+pNum*_num];
                       // cout<<"successful in spectraMapping"<<endl;
                        _pTime = startTime + _pulseTime[i];                 
                        _pTOF = std::make_pair(_tof[i],_pTime);
                        eventData.insert(std::multimap<uint32_t, std::pair<uint64_t, float>>::value_type(specNo,_pTOF));
                    }
                    file->closeGroup();//close bank
                }
            }
        }else{
            // Look for specilized banks
            for (auto j = m_bankNames.begin(); j != m_bankNames.end(); ++j)
            {
                std::string name = *j;
                file->openGroup(name, "NXevent_data");
                int _num = atoi(name.substr(4,1).c_str());
              //  file->readData("event_pixel_id",_pixelId);
                 file->readData("event_pixel_id",_tmp1);
                 file->readData("event_time_of_flight",_timeOfFlight);
                 cout<<"successful in read time of flight"<<endl;
                 file->readData("event_index", _index);
                    file->readData("event_pulse_time",_tmp2);
                    uint32_t n1 = 0;
                    it_tmp2 = _tmp2.begin();
                    for(it_index=_index.begin(); it_index != _index.end(); ++it_index)
                    {
                        if(*it_index == 0){
                            _fulltmp2.push_back(*it_tmp2);
                            n1++;
                        }else{
                                int delta = (*it_index)-(*(it_index-1));
                                if(delta != 0){
                                    for(int i = 0; i!= delta; i++){
                                        _fulltmp2.push_back(*(it_tmp2+n1));
                                    }
                                    n1++;
                         //           cout<<"when delta !=0, n1="<<n1<<endl;
                                }else {     n1++;}
                             }
                    }
                    cout<<"the size of pulsetime: "<<_fulltmp2.size()<<endl;
                   std::vector<uint32_t> _pixelId(_tmp1.begin(), _tmp1.end());
                   std::vector<uint64_t> _pulseTime(_fulltmp2.begin(), _fulltmp2.end());
                    cout<<"successful in put data into _pixelId and _pulseTime"<<endl;
                std::vector<float>_tof(_timeOfFlight.begin(),_timeOfFlight.end());
                    for (size_t i=0; i<_pixelId.size(); ++i)
                    {
                        specNo = spectraMapping[_pixelId[i]+pNum*_num];
                   // cout<<"successful in spectraMapping"<<endl;
                        _pTime = startTime + _pulseTime[i];
                         _pTOF = std::make_pair(_tof[i],_pTime);
                        eventData.insert(std::multimap<uint32_t, std::pair<uint64_t, float>>::value_type(specNo,_pTOF));
                    }
                file->closeGroup();
            }
            }

        file->closeGroup();
        file->closeGroup();
        return eventData;
}
//-------------------------------------------------------------------------------------------------
/**
 * Creates a time series property from the currently opened log entry. It is
 * assumed to
 * have been checked to have a time field and the value entry's name is given as
 * an argument
 * @param file :: A reference to the file handle
 * @param prop_name :: The name of the property
 * @returns A pointer to a new property containing the time series
 */
Kernel::Property *LoadCSNSNexus::createTimeSeries(::NeXus::File *file,
        const std::string &prop_name) const
{
    file->openData("time");
    //----- Start time is an ISO8601 string date and time. ------
    std::string start;
    try{
        file->getAttr("start", start);}
    catch (::NeXus::Exception &){
        start="2017-12-31T00:00:00+08:00";}
    // Convert to date and time
    Kernel::DateAndTime start_time = Kernel::DateAndTime(start);
    std::string time_units;
    file->getAttr("units", time_units);
    if (time_units.compare("second") < 0 && time_units != "s" &&
            time_units != "minutes") // Can be s/second/seconds/minutes
    {
        file->closeData();
        throw ::NeXus::Exception("Unsupported time unit '" + time_units + "'");
    }
    //--- Load the seconds into a double array ---
    std::vector<double> time_double;
    try
    {
        file->getDataCoerce(time_double);
    }
    catch (::NeXus::Exception &e)
    {
        g_log.warning() << "Log entry's time field could not be loaded: '"
                        << e.what() << "'.\n";
        file->closeData();
        throw;
    }
    file->closeData(); // Close time data
    g_log.debug() << "   done reading \"time\" array\n";
    // Convert to seconds if needed
    if (time_units == "minutes")
    {
        std::transform(time_double.begin(), time_double.end(), time_double.begin(),
                       std::bind2nd(std::multiplies<double>(), 60.0));
    }
    // Now the values: Could be a string, int or double
    file->openData("value");
    // Get the units of the property
    std::string value_units("");
    try
    {
        file->getAttr("units", value_units);
    }
    catch (::NeXus::Exception &)
    {
        // Ignore missing units field.
        value_units = "";
    }
    // Now the actual data
    ::NeXus::Info info = file->getInfo();
    // Check the size
    if (size_t(info.dims[0]) != time_double.size())
    {
        file->closeData();
        throw ::NeXus::Exception("Invalid value entry for time series");
    }
    if (file->isDataInt()) // Int type
    {
        std::vector<int> values;
        try
        {
            file->getDataCoerce(values);
            file->closeData();
        }
        catch (::NeXus::Exception &)
        {
            file->closeData();
            throw;
        }
        // Make an int TSP
        auto tsp = new TimeSeriesProperty<int>(prop_name);
        tsp->create(start_time, time_double, values);
        tsp->setUnits(value_units);
        g_log.debug() << "   done reading \"value\" array\n";
        return tsp;
    }
    else if (info.type == ::NeXus::CHAR)
    {
        std::string values;
        const int64_t item_length = info.dims[1];
        try
        {
            const int64_t nitems = info.dims[0];
            const int64_t total_length = nitems * item_length;
            boost::scoped_array<char> val_array(new char[total_length]);
            file->getData(val_array.get());
            file->closeData();
            values = std::string(val_array.get(), total_length);
        }
        catch (::NeXus::Exception &)
        {
            file->closeData();
            throw;
        }
        // The string may contain non-printable (i.e. control) characters, replace
        // these
        std::replace_if(values.begin(), values.end(), iscntrl, ' ');
        auto tsp = new TimeSeriesProperty<std::string>(prop_name);
        std::vector<DateAndTime> times;
        DateAndTime::createVector(start_time, time_double, times);
        const size_t ntimes = times.size();
        for (size_t i = 0; i < ntimes; ++i)
        {
            std::string value_i =
                std::string(values.data() + i * item_length, item_length);
            tsp->addValue(times[i], value_i);
        }
        tsp->setUnits(value_units);
        g_log.debug() << "   done reading \"value\" array\n";
        return tsp;
    }
    else if (info.type == ::NeXus::FLOAT32 || info.type == ::NeXus::FLOAT64)
    {
        std::vector<double> values;
        try
        {
            file->getDataCoerce(values);
            file->closeData();
        }
        catch (::NeXus::Exception &)
        {
            file->closeData();
            throw;
        }
        auto tsp = new TimeSeriesProperty<double>(prop_name);
        tsp->create(start_time, time_double, values);
        tsp->setUnits(value_units);
        g_log.debug() << "   done reading \"value\" array\n";
        return tsp;
    }
    else
    {
        throw ::NeXus::Exception(
            "Invalid value type for time series. Only int, double or strings are "
            "supported");
    }
}
/*-----------------------------------------------------------------------------------------------------------
 * Load an log entry
 * @param file:: NeXus file
 * @param workspace :: A pointer to the workspace to store the logs
 */
void LoadCSNSNexus::loadLog( ::NeXus::File *file, boost::shared_ptr<API::MatrixWorkspace> workspace) const
//void LoadCSNSNexus::loadLog( ::NeXus::File *file, EventWorkspace_sptr workspace) const
{
    file->openGroup(m_entryName, "NXentry");
    try{file->openGroup("logs", "NXcollection");}
    catch (::NeXus::Exception &e){
        g_log.warning() <<"Read logs: " << e.what()<<"\n";
        file->closeGroup();
        return;
    }
    // Validate the NX log class.
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        std::string logName=it->first;
        if (it->second == "NXlog")
        {
            file->openGroup(it->first, "NXlog");
            std::map<std::string, std::string> subEntries = file->getEntries();
            if ((subEntries.find("value") == subEntries.end()) || (subEntries.find("time") == subEntries.end()))
            {   g_log.warning() << "Invalid NXlog entry " << logName
                                << " found. Did not contain 'value' and 'time'.\n";}
            else{
                try{
                    if (!(workspace->run().hasProperty(logName)))
                    {
                        g_log.debug()<<logName<<"\n";
                        Kernel::Property *logValue = createTimeSeries(file, logName);
                        workspace->mutableRun().addProperty(logValue, true);
                    }
                }
                catch (::NeXus::Exception &e)
                {
                    g_log.warning() << "NXlog entry " << logName
                                    << " gave an error when loading:'" << e.what() << "'.\n";
                }
            }
            file->closeGroup();
        }//if
        if (it->second == "NXcollection")
        {
            std::string groupName=it->first;
            file->openGroup(it->first, "NXcollection");

            std::map<std::string, std::string> subEntries = file->getEntries();
            std::map<std::string, std::string>::iterator subit;
            for (subit = subEntries.begin(); subit != subEntries.end(); ++subit)
            {
                if (subit->second == "NXlog")
                {
                    file->openGroup(subit->first, "NXlog");
                    std::map<std::string, std::string> subEntries = file->getEntries();
                    if ((subEntries.find("value") == subEntries.end()) || (subEntries.find("time") == subEntries.end()))
                    {
                        g_log.warning() << "Invalid NXlog entry " << subit->first
                                        << " found. Did not contain 'value' and 'time'.\n";
                    }
                    else
                    {
                        try
                        {
                            if (!(workspace->run().hasProperty(subit->first)))
                            {
                                g_log.debug()<<subit->first<<"\n";
                                Kernel::Property *logValue = createTimeSeries(file, groupName+"_"+subit->first);
                                workspace->mutableRun().addProperty(logValue, true);
                            }
                        }
                        catch (::NeXus::Exception &e)
                        {
                            g_log.warning() << "NXlog entry " << subit->first
                                            << " gave an error when loading:'" << e.what() << "'.\n";
                        }
                    }
                    file->closeGroup();
                }
            }
            file->closeGroup();
        }//if
    }//for
    file->closeGroup();
    file->closeGroup();
}
//-------------------------------------------------------------------------------------------------
/**
* Load event data into workspace
* @param[in] filename :: nxs file name
* @param[in] workspace :: workspace name
* @param[in] sartTime :: start time of the experiment
* @param[in] totalPixels :: total pixel numbers
* @param[in] pNum :: pixel number of each bank or monitor
* @param[in] timeBin :: time of flight
*/
/*
void LoadCSNSNexus::loadEventData(::NeXus::File *file, boost::shared_ptr<DataObjects::EventWorkspace> workspace,
                                    size_t totalPixels, Kernel::DateAndTime startTime, int pNum,
                                    std::vector<uint32_t> timeBin)
{
    workspace->initialize(totalPixels,1,1);
    uint64_t sTime = startTime.totalNanoseconds();
    std::multimap<uint32_t,std::pair<uint64_t, float>> m_eventData = getEventData(file, sTime, pNum);
    cout<<"successful get event data in multimap"<<endl;
    float m_tof;
    uint64_t m_pulseTime;
    std::multimap<uint32_t,std::pair<uint64_t, float>>::iterator it;
    int numPixels = int(totalPixels);
    //numPixels is total numbers of all pixels.
    for (int i = 0; i != numPixels; i++)
    {
        it = m_eventData.find(i+1);
        for (size_t j = 0; j != m_eventData.count(i+1); j++, it++)
        {
            m_pulseTime = it->second.first;
            m_tof = it->second.second;
            workspace->getEventList(i) += TofEvent(m_tof, m_pulseTime);
        }
        workspace->getEventList(i).setSpectrumNo(i+1);
    }
    Kernel::cow_ptr<MantidVec> axis; // MantidVec = std::vector<double>. axis point x axis.
    MantidVec &xRef = axis.access(); //if data is shared, creates a copy of data,return new copy of *this.
    xRef.resize(2); //set new column size of 2
    xRef[0] = 0;
    xRef[1] = *(timeBin.end()-1); //set tof-max
    workspace->setAllX(axis); //set X size. setAllX(const HistogramData::BinEdges &x)
}
*/
//-------------------------------------------------------------------------------------------------
/**
* Load histgram data into workspace
* @param[in] filename :: nxs file name
* @param[in] workspace :: workspace name
* @param[in] totalPixels :: total pixel numbers
* @param[in] timeBin :: time of flight
*/
/*
void LoadCSNSNexus::loadHistData(::NeXus::File *file, boost::shared_ptr<API::MatrixWorkspace> workspace, size_t totalPixels, std::vector<uint64_t> timeBin)
{
    std::vector<uint32_t> m_value = getValue(file,true);
    size_t numBins = timeBin.size();
    MantidVecPtr x, e;
    MantidVec &xRef = x.access();
    MantidVec &eRef = e.access();
    xRef.resize(numBins);
    eRef.resize(numBins, 0); // Value of 0 in all error
    //cout<<"how many tofs: "<<timeBin.size()<<endl;
    // set X value
    int st1 = 0;
    for (std::vector<uint64_t>::iterator it = timeBin.begin(); it != timeBin.end(); ++it)  
    {
        xRef[st1] = (*it);
        st1++;
    }
    // set Y value
    size_t hist = 0;
    std::vector<uint32_t>::iterator it_start = m_value.begin();
    std::vector<uint32_t>::iterator it_end = it_start + numBins ;
    std::vector<uint64_t>::iterator it_tof = timeBin.begin();
    while(hist < totalPixels){
    MantidVec &Y = workspace->dataY(hist);
    Y.assign(it_start, it_end);
    it_start += numBins;
    it_end += numBins;
    workspace->setX(hist, x);
    ++hist;
    }
}
*/
//-------------------------------------------------------------------------------------------------
void LoadCSNSNexus::exec()
{
    // The input properties
    std::string filename = getPropertyValue("Filename");
    m_entryName = getPropertyValue("Entryname");
    m_bankNames = getProperty("Bankname");
    m_monitorNames = getProperty("Monitorname");
    cout<<"the number of bank is: "<<m_bankNames.size()<<endl;
    //cout<<"the number of monitors is: "<<m_monitorNames.size()<<endl;
    bool m_loadEvent = getProperty("Loadevent");
    if (m_loadEvent)
    {
        g_log.debug() <<"Event data will be loaded.\n";
    }
    else
    {
        g_log.debug() <<"Histogram data will be loaded.\n";
    }
    m_spec_min = getProperty("SpectrumMin");
    m_spec_max = getProperty("SpectrumMax");
    auto prog = new Progress(this, 0.0, 1.0, 10);
    prog->doReport("Start to load Nexus");

    // Open nxs file
    auto file = new ::NeXus::File(filename);
    // make sure that the entry name in NXS file is csns.
    std::string newEntryName = getEntryName(file);
    cout<<"Get Entry Name: "<< newEntryName<<endl;

    if (newEntryName != m_entryName)
    {
        throw std::runtime_error(m_entryName+" not found in the NXS file!");
    }
    //if the input is empty, get all detectors from NXS file.
    int bankNum, monitorNum;
    if(!m_bankNames.size())
    {
        std::vector<std::string> bankNames= getDetectorNames(file, true);
        bankNum = int(bankNames.size());
    }else
    {
        bankNum = int(m_bankNames.size());
    }
    if(!m_monitorNames.size())
    {
        std::vector<std::string>  monitorNames= getDetectorNames(file, false);
        monitorNum = int(monitorNames.size());
    }else
    {
        monitorNum = int(m_monitorNames.size());
    }
    cout<<"the number of banks: "<<bankNum<<endl;
    cout<<"the number of monitors: "<<monitorNum<<endl;

   // g_log.debug() <<bankNum << " banks found." << std::endl;
   // g_log.debug() <<monitorNum << " monitors found." << std::endl;
   // prog->setNumSteps(bankNum + 5);//why?

    //Get version
    std::string m_version = getVersion(file);
    cout<<"CSNS Nexus version is: "<< m_version<<endl;
    g_log.debug()<<"CSNS Nexus version is: " <<m_version<<"\n";
    //Get sample information
    CSNSsample m_sampleInfo = getSampleInfo(file);
    cout<<"Have get sample Info: "<<endl;

     // Get start time and end time
    Kernel::DateAndTime start_time = getStartTime(file);
    Kernel::DateAndTime end_time = getEndTime(file);
    cout<<"the start_time is "<<start_time<<endl;

    prog->doReport("Counting pixels");
    std::vector<uint32_t> pixelId_bank = getPixelId(file, true);
//    std::vector<uint32_t> pixelId_monitor = getPixelId(file, false);
    m_bTotalPixels = pixelId_bank.size();
//    m_mTotalPixels = pixelId_monitor.size();
    g_log.debug() <<m_bTotalPixels <<" pixels found."<< std::endl;
    //test pixelID
    cout<<"the pixel number in the banks: "<<m_bTotalPixels<<endl;
    //cout<<"the pixel number in the monitors: "<<m_mTotalPixels<<endl;

    bPixelNum = int(m_bTotalPixels/bankNum);
//    mPixelNum = int(m_mTotalPixels/monitorNum);

    prog->doReport("Counting TOF bins");
   // m_bNumBins = getNumBins(file, true);
    std::vector<uint32_t> m_timeBin = getTimeBin(file);
    m_bNumBins = m_timeBin.size();
    cout<<"the bins is: "<<m_bNumBins<<endl;
   // m_mNumBins = getNumBins(file, false);
   // g_log.debug() <<m_bNumBins <<" TOF bins found."<< std::endl;

/*
    if(m_loadEvent)
    {
        if (hasEventData(file))
        {
            cout<<"has Event data\n";
        }else
        {
            throw std::runtime_error(" No event data in the NXS file!");
        }

        EventWorkspace_sptr WS(new EventWorkspace);
        setProperty("OutputWorkspace", WS);

        cout<<"successful create event WS"<<endl;
        loadEventData(file, WS, m_bTotalPixels, start_time, bPixelNum, m_timeBin);
        
        WS->mutableRun().setStartAndEndTime(start_time, end_time);
        WS->setTitle(m_sampleInfo.name);
        WS->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");
        WS->setYUnit("Counts");

        // Load log of source, chopper and SE    
        prog->doReport("Loading time log");
        loadLog(file, WS);
    }else
    {
         if (hasHistogramData(file))
        {
            cout<<"has Histogram data\n";
        }else
        {
            throw std::runtime_error(" No histgram data in the NXS file!");
        }
*/
/*
        prog->doReport("Creating workspace");
        // Bank workspace
        MatrixWorkspace_sptr WS1 = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D", m_bTotalPixels, m_bNumBins + 1, m_bNumBins));
        // Monitor workspace
        MatrixWorkspace_sptr WS2 = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D", m_mTotalPixels, m_bNumBins + 1, m_bNumBins));
        // make group
        WorkspaceGroup_sptr wksp_group(new WorkspaceGroup);
        wksp_group->addWorkspace(WS1);
        wksp_group->addWorkspace(WS2);
        // set outputworkspace
        setProperty("OutputWorkspace",boost::dynamic_pointer_cast<Workspace>(wksp_group));
        loadHistData(file, WS1, m_bTotalPixels, m_timeBin);
        WS1->mutableRun().setStartAndEndTime(start_time, end_time);
        WS1->setTitle(m_sampleInfo.name);
        WS1->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");
        WS1->setYUnit("Counts");
*/

       // Load log of source, chopper and SE    
//        prog->doReport("Loading time log");
//        loadLog(file, WS1);
//    }

//----------------------------------------- test---------------------------------------------------------
    EventWorkspace_sptr WS(new EventWorkspace);
    setProperty("OutputWorkspace", WS);
    WS->initialize(bPixelNum,1,1);
    WS->mutableRun().setStartAndEndTime(start_time, end_time);
    // load info into workspace 
   // CSNSsample m_sampleInfo = getSampleInfo(file);
    WS->setTitle(m_sampleInfo.name);
    // set units in X and Y axis 
    WS->getAxis(0)->unit() = Kernel::UnitFactory::Instance().create("TOF");    
    WS->setYUnit("Counts");

    // test getSpectraMapping
    std::map<uint32_t, uint32_t> spectraMapping_bank = getSpectraMapping(file, true);
//    std::map<int, int> spectraMapping_monitor = getSpectraMapping(file, false);
    std::cout<<"total number of spectrum in banks is: "<<spectraMapping_bank.size()<<std::endl;
//    std::cout<<"total number of spectrum in monitor is: "<<spectraMapping_monitor.size()<<std::endl;
    if (spectraMapping_bank.count(510))
    std::cout<<"Mapping of detector ID 510  to spectrum no in bank is: "<<spectraMapping_bank[510]<<std::endl;
    if (spectraMapping_bank.count(515))
    std::cout<<"Mapping of detector ID 515  to spectrum no in bank is: "<<spectraMapping_bank[515]<<std::endl;
    if (spectraMapping_bank.count(1025))
    std::cout<<"Mapping of detector ID 1025  to spectrum no in bank is: "<<spectraMapping_bank[1025]<<std::endl;
//    if (spectraMapping_monitor.count(255))
//    std::cout<<"Mapping of detector ID 255  to spectrum no in monitor is: "<<spectraMapping_monitor[255]<<std::endl;
//    if (spectraMapping_monitor.count(260))
//    std::cout<<"Mapping of detector ID 260  to spectrum no in monitor  is: "<<spectraMapping_monitor[260]<<std::endl;

// test getEventData

    uint64_t sTime = start_time.totalNanoseconds();
    std::multimap<uint32_t,std::pair<uint64_t, float>> m_eventData_bank = getEventData(file, sTime, bPixelNum);
    cout<<"the size of m_eventData_bank is: "<<m_eventData_bank.size()<<endl;
    typedef std::multimap<uint32_t,std::pair<uint64_t, float>>::iterator it_multimap;
    // event data in banks
    it_multimap _specNo_bank1 = m_eventData_bank.find(510);
    if(_specNo_bank1 != m_eventData_bank.end())
        std::cout<<"spectra No "<<_specNo_bank1->first<<" in bank has "<<m_eventData_bank.count(510)<<" event datas!"<<std::endl;

    it_multimap _specNo_bank2 = m_eventData_bank.find(515);
    if(_specNo_bank2 != m_eventData_bank.end())
        std::cout<<"spectra No "<<_specNo_bank2->first<<" in bank has "<<m_eventData_bank.count(515)<<" event datas! "<<std::endl;

    it_multimap _specNo_bank3 = m_eventData_bank.find(1025);
    if(_specNo_bank3 != m_eventData_bank.end())
        std::cout<<"spectra No "<<_specNo_bank3->first<<" in bank has "<<m_eventData_bank.count(1025)<<" event datas! "<<std::endl;
    

    //load instrument
//    std::vector<std::vector<float>>pixelOffset = getPixelOffset(file,bankNames[0]);
//    m_xOffset = pixelOffset[0];
//    m_yOffset = pixelOffset[1];
//    std::cout<<"xoffsets size is "<<m_xOffset.size()<<std::endl;

    // close nxs file
    file->close();
    delete file;
}

} // namespace DataHandling
} // namespace Mantid
