#ifndef MANTID_DATAHANDLING_LOADCSNSNEXUS_H_
#define MANTID_DATAHANDLING_LOADCSNSNEXUS_H_

//----------------------------------------------------------------------
// Includes
//----------------------------------------------------------------------
#include "MantidAPI/IFileLoader.h"
#include "MantidDataObjects/EventWorkspace.h"
#include <nexus/NeXusFile.hpp>
#include <nexus/NeXusException.hpp>
#include "MantidDataObjects/Events.h"
#include "MantidAPI/WorkspaceGroup.h"
#include "MantidKernel/TimeSeriesProperty.h"
#include "MantidDataHandling/EventWorkspaceCollection.h"
#include <memory>
#include <boost/lexical_cast.hpp>


namespace Mantid
{

namespace DataHandling
{
/**
 Loads a NeXus file that conforms to the CSNSNexus instrument definition format and
 stores it in a 2D workspace.

 Copyright &copy; 2007-8 ISIS Rutherford Appleton Laboratory, NScD Oak Ridge
 National Laboratory & European Spallation Source

 This file is part of Mantid.

 Mantid is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 Mantid is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.

 File change history is stored at: <https://github.com/mantidproject/mantid>
 */
struct CSNSsample
{
    std::string name;
    std::string type;
    std::string situation;
    std::string chemical_formula;
    std::string shape;
    float density;    
    float mass;    
    float distance;    
    std::vector<float> size ;    
};


class DLLExport LoadCSNSNexus
    : public API::IFileLoader<Kernel::NexusDescriptor>
{
public:
    /// Default Constructor
    LoadCSNSNexus();

    // Destructor
    virtual ~LoadCSNSNexus() {}

    /// Algorithm's name for identification overriding a virtual method
    virtual const std::string name() const
    {
        return "LoadCSNSNexus";
    }

    /// Summary of algorithms purpose
    virtual const std::string summary() const
    {
        return "Loads a NeXus file confirming to the CSNSNexus format";
    }

    /// Algorithm's version for identification overriding a virtual method
    virtual int version() const
    {
        return 1;
    }

    /// Algorithm's category for identification overriding a virtual method
    virtual const std::string category() const
    {
        return "DataHandling\\Nexus";
    }

    ///variables
    std::string m_entryName; //entry name
    std::vector<std::string> m_bankNames; //specified bank names
    std::vector<std::string> m_monitorNames; //specified monitor names
    int bPixelNum, mPixelNum; // pixel numbers in one detector
    size_t m_bTotalPixels;/// total Number of pixels in banks
    size_t m_mTotalPixels;/// Number of pixels in monitors
    size_t m_bNumBins;/// Number of bins in banks
    size_t m_mNumBins;/// Number of bins in monitor
    std::vector<float> m_xOffset;/// x_pixel_offset
    std::vector<float> m_yOffset;/// y_pixel_offset


    //get methods
    std::string getEntryName(::NeXus::File *file); //get entry name
    /// get CSNS Nexus version
    std::string getVersion(::NeXus::File *file); //get CSNS Nexus version

    /// Return true if nxs has histogram data
    bool hasHistogramData(::NeXus::File *file);/// Return true if nxs has histogram data

    /// Return true if nxs has event data
    bool hasEventData(::NeXus::File *file); /// Return true if nxs has event data

    /// List bank or monitor names
    std::vector<std::string> getDetectorNames(::NeXus::File *file, bool bm); /// List bank or monitor names

    /// Return number of pixels of all banks
    size_t getNumPixels(::NeXus::File *file, bool bm); /// Return number of pixels of all banks

    /// Return number of bins of TOF/d/Q
    size_t getNumBins(::NeXus::File *file, bool bm); /// Return number of bins of TOF/d/Q

    /// Load Log to workspace
    void loadLog(::NeXus::File *file, boost::shared_ptr<API::MatrixWorkspace> workspace) const; /// Load Log to workspace
   // void loadLog(::NeXus::File *file, EventWorkspace_sptr workspace) const;

    /// Create a time series property
    Kernel::Property *createTimeSeries(::NeXus::File *file, const std::string &prop_name) const; /// Create a time series property

    /// Return pixel offset of x and y
    std::vector<std::vector<float>> getPixelOffset(::NeXus::File *file, const std::string &bankName);/// Return pixel offset of x and y

    /// get pixel_id
    std::vector<uint32_t> getPixelId(::NeXus::File *file, bool bm);/// get pixel_id

    /// get time bins
    std::vector<uint32_t> getTimeBin(::NeXus::File *file);
   // std::vector<int> getTimeBin(::NeXus::File *file);

    /// get event pixel_id
//    std::vector<int> getEventPixelId(::NeXus::File *file, bool bm);

    /// Returns a confidence value that this algorithm can load a file
    virtual int confidence(Kernel::NexusDescriptor &descriptor) const;/// Returns a confidence value that this algorithm can load a file
    /// get start_time
    Kernel::DateAndTime getStartTime(::NeXus::File *file);/// get start_time

    /// get end_time
    Kernel::DateAndTime getEndTime(::NeXus::File *file);/// get end_time
    /// get instrument name
    std::string getInstrumentName(::NeXus::File *file);/// get instrument name

    /// get value
  //  std::vector<uint32_t> getValue(::NeXus::File *file, bool bm);

    /// get event data
    std::multimap<uint32_t, std::pair<uint64_t, float>> getEventData(::NeXus::File *file, uint64_t startTime, int pNum);
    //std::multimap<int, std::pair<double, int64_t>> getEventData(::NeXus::File *file, int64_t startTime, int pNum, bool bm);

    /// get spectra mapping
    std::map<uint32_t, uint32_t> getSpectraMapping(::NeXus::File *file, bool bm);
    //std::map<int, int> getSpectraMapping(::NeXus::File *file, bool bm);


    /// get sample information
    CSNSsample getSampleInfo(::NeXus::File *file);

    /// load event data
    //template <typename T>
    //void loadEventData(::NeXus::File *file, boost::shared_ptr<DataObjects::EventWorkspace> workspace,
    //void loadEventData(::NeXus::File *file, boost::shared_ptr<DataObjects::EventWorkspace> workspace, size_t totalPixels, Kernel::DateAndTime startTime, int pNum, std::vector<uint32_t> timeBin);
    
    // load histgram data
   //void loadHistData(::NeXus::File *file, boost::shared_ptr<API::MatrixWorkspace> workspace, size_t totalPixels, std::vector<uint64_t> timeBin);
  // void loadHistData(::NeXus::File *file, boost::shared_ptr<API::MatrixWorkspace> workspace, size_t totalPixels, std::vector<int> timeBin);




protected:
    void init();
    void exec();


    /// List of the absolute time of each pulse
    std::vector<Kernel::DateAndTime> pulseTimes;

    /// Number of bins
    size_t m_numBins;

    /// Interval of chunk
    specid_t m_spec_min, m_spec_max;

    /// Mutex to avoid simultaneous file access
    Kernel::Mutex m_fileMutex;

};

} // namespace DataHandling
} // namespace Mantid

#endif /* MANTID_DATAHANDLINGLOADCSNSNEXUS_H_ */
