//----------------------------------------------------------------------
// Includes
//----------------------------------------------------------------------
#include "MantidDataHandling/LoadCSNSNexus.h"
#include "MantidDataHandling/EventWorkspaceCollection.h"

#include <boost/random/mersenne_twister.hpp>
#include <boost/random/uniform_real.hpp>
#include <boost/shared_ptr.hpp>
#include <boost/shared_array.hpp>
#include <boost/scoped_array.hpp>
#include <boost/function.hpp>
#include <functional>

#include "MantidKernel/ArrayProperty.h"
#include "MantidKernel/ThreadPool.h"
#include "MantidKernel/UnitFactory.h"
#include "MantidKernel/ThreadSchedulerMutexes.h"
#include "MantidKernel/BoundedValidator.h"
#include "MantidKernel/VisibleWhenProperty.h"
#include "MantidKernel/TimeSeriesProperty.h"
#include "MantidGeometry/Instrument/RectangularDetector.h"
#include "MantidAPI/FileProperty.h"
#include "MantidAPI/MemoryManager.h"
#include "MantidAPI/RegisterFileLoader.h"
#include "MantidAPI/SpectrumDetectorMapping.h"
#include "MantidKernel/Timer.h"

using std::cout;
using std::endl;
using std::map;
using std::string;
using std::vector;

namespace Mantid
{
namespace DataHandling
{

DECLARE_NEXUS_FILELOADER_ALGORITHM(LoadCSNSNexus)

using namespace Kernel;
using namespace Geometry;
using namespace API;
using namespace DataObjects;

LoadCSNSNexus::LoadCSNSNexus()
    : m_numPixels(0), pulseTimes(0), m_numBins(0), m_spec_min(0),
      m_spec_max(0), m_fileMutex() {}

//-------------------------------------------------------------------------------------------------
/// Initialisation method.
void LoadCSNSNexus::init()
{
    declareProperty(
        new FileProperty("Filename", "", FileProperty::Load, {".nxs"}),
        "The name of the NeXus file to load");
    declareProperty(new WorkspaceProperty<API::Workspace>("OutputWorkspace", "",
                    Direction::Output),
                    "The name of the Workspace2D to create.");
    declareProperty(
        new PropertyWithValue<string>("Entryname", "csns",Direction::Input), "Optional: Name of entry (default csns)");

    declareProperty(
        new ArrayProperty<string>("Bankname", Direction::Input),
        "Optional: A comma-separated list of bank to read. (default all banks)");
    declareProperty(
        new PropertyWithValue<bool>("Loadevent", true, Direction::Input),
        "Default true: load event data mode, false: Load histogram data.");

    auto mustBePositive = boost::make_shared<BoundedValidator<int>>();
    mustBePositive->setLower(1);
    declareProperty(
        new PropertyWithValue<specid_t>("SpectrumMin", 1, mustBePositive),
        "The index number of the first spectrum to read.  Only used if spectrum_max is set.");
    declareProperty(
        new PropertyWithValue<specid_t>("SpectrumMax", Mantid::EMPTY_INT(),
                                        mustBePositive),
        "The number of the last spectrum to read. Only used if explicitly set.");
}

//-------------------------------------------------------------------------------------------------
/**
 * Return the confidence with with this algorithm can load the file
 * @param descriptor A descriptor for the file
 * @returns An integer specifying the confidence level. 0 indicates it will not
 * be used
 */
int LoadCSNSNexus::confidence(Kernel::NexusDescriptor &descriptor) const
{
    int confidence(0);
    if (descriptor.pathOfTypeExists("/csns", "NXentry"))
    {
        const bool hasEventData = descriptor.classTypeExists("NXevent_data");
        const bool hasHistogramData = descriptor.classTypeExists("NXdata");
        if (hasHistogramData && hasEventData)
            // Event data = this is event NXS
            confidence = 20;
        else if (hasHistogramData && !hasEventData)
            // No event data = this is the one
            confidence = 80;
        else if (!hasHistogramData && hasEventData)
            // Histogram Data only
            confidence = 40;
        else
            // No data ?
            confidence = 10;
    }
    return confidence;
}

//-------------------------------------------------------------------------------------------------
/**
 * Return entry name
 * @param[in] file :: nxs file
 * @returns newEntry :: entry name
 */
std::string LoadCSNSNexus::getEntryName(::NeXus::File *file)
{
    std::string newEntry="";

    std::map<std::string, std::string> entries = file->getEntries();

    if (entries.empty())
    {
        throw std::runtime_error("No entries in the NXS file!");
    }
    else if (entries.find("csns") != entries.end())
    {
        newEntry = "csns";
    }
    else
    {
        newEntry = entries.begin()->first;
    }

    return newEntry;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the pixel number
* @param[in] file :: nxs file
* @return numPixels
*/
size_t LoadCSNSNexus::getNumPixels(::NeXus::File *file)
{
    size_t numPixels=0;
    size_t newPixels=0;
    
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    // Look for all the banks
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        std::string name = it->first;

        if (name.substr(0, 4) == "bank")
        {
            // cout<<"name is:"<<name<<endl;
            // file->openGroup(name, "NXdetector");
            file->openGroup(name, it->second);

            std::map<std::string, std::string> entries = file->getEntries();

            if (entries.find("pixel_id") != entries.end())
            {
                file->openData("pixel_id");
                std::vector<int64_t> dims = file->getInfo().dims;
                file->closeData();

                if (!dims.empty())
                {
                    newPixels = 1;
                    for (size_t i = 0; i < dims.size(); i++)
                        newPixels *= dims[i];
                    // cout<<"newPixels is: "<<newPixels<<endl;
                    numPixels += newPixels;
                }
                // cout<<"numPixels is:"<<numPixels<<endl;
            }
            else
            {
                throw std::runtime_error("No pixels in the NXS file!");
            }
            file->closeGroup();
        }
    }
    file->closeGroup();
    file->closeGroup();
    return numPixels;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the pixel number
* @param[in] filen :: nxs file
* @return numBins :: bins of TOF
*/
size_t LoadCSNSNexus::getNumBins(::NeXus::File *file)
{
    size_t numBins=0;

    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    // Look for all the banks
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        std::string name = it->first;
        if (name.substr(0, 4) == "bank")
        {
            file->openGroup(name, it->second);
            std::map<std::string, std::string> entries = file->getEntries();
            if (entries.find("time_of_flight") != entries.end())
            {
                file->openData("time_of_flight");
                std::vector<int64_t> dims = file->getInfo().dims;
                file->closeData();
                numBins = dims[0];
                break;
            }
        }
    }
    return numBins;
}

//-------------------------------------------------------------------------------------------------
/**
* Return Bank list NX/entry/instrument/bank*
* @param[in] file :: nxs file
* @return bankNames :: bank list
*/
std::vector<std::string> LoadCSNSNexus::getBankNames(::NeXus::File *file)
{
    std::vector<std::string> bankNames;

    // Open the default data group 'entry'
    file->openGroup(m_entryName, "NXentry");
    // Also pop into the instrument
    file->openGroup("instrument", "NXinstrument");

    // Look for all the banks
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        if (it->first.substr(0,4)=="bank" && it->second=="NXdetector")
        {
            bankNames.push_back(it->first);
        }
    }

    return bankNames;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the pixel offset
* @param[in] file :: nxs file
* @return pixelOffset :: [x,y]
*/

std::vector<std::vector<float> > LoadCSNSNexus::getPixelOffset(::NeXus::File *file, const std::string &bankName)
{
    std::vector<std::vector<float> > pixelOffset;

    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup(bankName, "NXdetector");

    std::vector<float> _tmp;
    file->readData("x_pixel_offset",_tmp);
    pixelOffset.push_back(_tmp);
    file->readData("y_pixel_offset",_tmp);
    pixelOffset.push_back(_tmp);

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();

    return pixelOffset;
}
//-------------------------------------------------------------------------------------------------
/**
 * Creates a time series property from the currently opened log entry. It is
 * assumed to
 * have been checked to have a time field and the value entry's name is given as
 * an argument
 * @param file :: A reference to the file handle
 * @param prop_name :: The name of the property
 * @returns A pointer to a new property containing the time series
 */
Kernel::Property *LoadCSNSNexus::createTimeSeries(::NeXus::File *file,
        const std::string &prop_name) const
{
    file->openData("time");
    //----- Start time is an ISO8601 string date and time. ------
    std::string start;
    try
    {
        file->getAttr("start", start);
    }
    catch (::NeXus::Exception &)
    {
        start="2017-12-31T00:00:00+08:00";
    }

    // Convert to date and time
    Kernel::DateAndTime start_time = Kernel::DateAndTime(start);
    std::string time_units;
    file->getAttr("units", time_units);
    if (time_units.compare("second") < 0 && time_units != "s" &&
            time_units != "minutes") // Can be s/second/seconds/minutes
    {
        file->closeData();
        throw ::NeXus::Exception("Unsupported time unit '" + time_units + "'");
    }
    //--- Load the seconds into a double array ---
    std::vector<double> time_double;
    try
    {
        file->getDataCoerce(time_double);
    }
    catch (::NeXus::Exception &e)
    {
        g_log.warning() << "Log entry's time field could not be loaded: '"
                        << e.what() << "'.\n";
        file->closeData();
        throw;
    }
    file->closeData(); // Close time data
    g_log.debug() << "   done reading \"time\" array\n";
    // Convert to seconds if needed
    if (time_units == "minutes")
    {
        std::transform(time_double.begin(), time_double.end(), time_double.begin(),
                       std::bind2nd(std::multiplies<double>(), 60.0));
    }
    // Now the values: Could be a string, int or double
    file->openData("value");
    // Get the units of the property
    std::string value_units("");
    try
    {
        file->getAttr("units", value_units);
    }
    catch (::NeXus::Exception &)
    {
        // Ignore missing units field.
        value_units = "";
    }

    // Now the actual data
    ::NeXus::Info info = file->getInfo();
    // Check the size
    if (size_t(info.dims[0]) != time_double.size())
    {
        file->closeData();
        throw ::NeXus::Exception("Invalid value entry for time series");
    }
    if (file->isDataInt()) // Int type
    {
        std::vector<int> values;
        try
        {
            file->getDataCoerce(values);
            file->closeData();
        }
        catch (::NeXus::Exception &)
        {
            file->closeData();
            throw;
        }
        // Make an int TSP
        auto tsp = new TimeSeriesProperty<int>(prop_name);
        tsp->create(start_time, time_double, values);
        tsp->setUnits(value_units);
        g_log.debug() << "   done reading \"value\" array\n";
        return tsp;
    }
    else if (info.type == ::NeXus::CHAR)
    {
        std::string values;
        const int64_t item_length = info.dims[1];
        try
        {
            const int64_t nitems = info.dims[0];
            const int64_t total_length = nitems * item_length;
            boost::scoped_array<char> val_array(new char[total_length]);
            file->getData(val_array.get());
            file->closeData();
            values = std::string(val_array.get(), total_length);
        }
        catch (::NeXus::Exception &)
        {
            file->closeData();
            throw;
        }
        // The string may contain non-printable (i.e. control) characters, replace
        // these
        std::replace_if(values.begin(), values.end(), iscntrl, ' ');
        auto tsp = new TimeSeriesProperty<std::string>(prop_name);
        std::vector<DateAndTime> times;
        DateAndTime::createVector(start_time, time_double, times);
        const size_t ntimes = times.size();
        for (size_t i = 0; i < ntimes; ++i)
        {
            std::string value_i =
                std::string(values.data() + i * item_length, item_length);
            tsp->addValue(times[i], value_i);
        }
        tsp->setUnits(value_units);
        g_log.debug() << "   done reading \"value\" array\n";
        return tsp;
    }
    else if (info.type == ::NeXus::FLOAT32 || info.type == ::NeXus::FLOAT64)
    {
        std::vector<double> values;
        try
        {
            file->getDataCoerce(values);
            file->closeData();
        }
        catch (::NeXus::Exception &)
        {
            file->closeData();
            throw;
        }
        auto tsp = new TimeSeriesProperty<double>(prop_name);
        tsp->create(start_time, time_double, values);
        tsp->setUnits(value_units);
        g_log.debug() << "   done reading \"value\" array\n";
        return tsp;
    }
    else
    {
        throw ::NeXus::Exception(
            "Invalid value type for time series. Only int, double or strings are "
            "supported");
    }
}
/**
* Return the pixel_id
* @param[in] file :: nxs file
* @param[in] bankName :: bank name
* @return pixelId :: pixel_id
*/
std::vector<uint32_t> LoadCSNSNexus::getPixelId(::NeXus::File *file, const std::string &bankName)
{
    std::vector<int> tmp;
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup(bankName, "NXdetector");

    file->readData("pixel_id",tmp);
    std::vector<uint32_t> pixelId(tmp.begin(), tmp.end());

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();

    return pixelId;
}

//-------------------------------------------------------------------------------------------------
/**
 * Load an log entry
 * @param file:: NeXus file
 * @param workspace :: A pointer to the workspace to store the logs
 */
void LoadCSNSNexus::loadNXLog(
    ::NeXus::File *file,
    boost::shared_ptr<API::MatrixWorkspace> workspace) const
{

    file->openGroup(m_entryName, "NXentry");

    try
    {
        file->openGroup("logs", "NXcollection");
    }
    catch (::NeXus::Exception &e)
    {
        g_log.warning() <<"Read logs: " << e.what()<<"\n";
        file->closeGroup();
        return;
    }

    // Validate the NX log class.
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        std::string logName=it->first;
        if (it->second == "NXlog")
        {
            file->openGroup(it->first, "NXlog");

            std::map<std::string, std::string> subEntries = file->getEntries();
            if ((subEntries.find("value") == subEntries.end()) || (subEntries.find("time") == subEntries.end()))
            {
                g_log.warning() << "Invalid NXlog entry " << logName
                                << " found. Did not contain 'value' and 'time'.\n";
            }
            else
            {
                try
                {
                    if (!(workspace->run().hasProperty(logName)))
                    {
                        g_log.debug()<<logName<<"\n";
                        Kernel::Property *logValue = createTimeSeries(file, logName);
                        workspace->mutableRun().addProperty(logValue, true);
                    }
                }
                catch (::NeXus::Exception &e)
                {
                    g_log.warning() << "NXlog entry " << logName
                                    << " gave an error when loading:'" << e.what() << "'.\n";
                }

            }
            file->closeGroup();
        }//if
        if (it->second == "NXcollection")
        {
            std::string groupName=it->first;
            file->openGroup(it->first, "NXcollection");

            std::map<std::string, std::string> subEntries = file->getEntries();
            std::map<std::string, std::string>::iterator subit;
            for (subit = subEntries.begin(); subit != subEntries.end(); ++subit)
            {
                if (subit->second == "NXlog")
                {
                    file->openGroup(subit->first, "NXlog");
                    std::map<std::string, std::string> subEntries = file->getEntries();
                    if ((subEntries.find("value") == subEntries.end()) || (subEntries.find("time") == subEntries.end()))
                    {
                        g_log.warning() << "Invalid NXlog entry " << subit->first
                                        << " found. Did not contain 'value' and 'time'.\n";
                    }
                    else
                    {
                        try
                        {
                            if (!(workspace->run().hasProperty(subit->first)))
                            {
                                g_log.debug()<<subit->first<<"\n";
                                Kernel::Property *logValue = createTimeSeries(file, groupName+"_"+subit->first);
                                workspace->mutableRun().addProperty(logValue, true);
                            }
                        }
                        catch (::NeXus::Exception &e)
                        {
                            g_log.warning() << "NXlog entry " << subit->first
                                            << " gave an error when loading:'" << e.what() << "'.\n";
                        }

                    }
                    file->closeGroup();
                }

            }
            file->closeGroup();
        }//if
    }//for
    file->closeGroup();
    file->closeGroup();
    std::cout<<"current path: "<<file->getPath()<<std::endl;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the CSNS NeXus definiton version
* @param[in] filename :: nxs file name
* @return version :: CSNS NeXus definition version
*/
std::string LoadCSNSNexus::getVersion(::NeXus::File *file)
{
    std::string version="";

    file->openGroup(m_entryName, "NXentry");

    std::map<std::string, std::string> entries = file->getEntries();

    if (entries.find("version") != entries.end())
    {
        file->readData("version", version);
    }
    else
    {
        throw std::runtime_error("No version in the NXS file!");
    }

    file->closeGroup();
    return version;
}

void LoadCSNSNexus::exec()
{
    // The input properties
    std::string filename = getPropertyValue("Filename");
    m_entryName = getPropertyValue("Entryname");
    std::vector<std::string> m_bankNames = getProperty("Bankname");
    bool m_loadEvent = getProperty("Loadevent");
    if (m_loadEvent){
        g_log.debug() <<"Event data will be loaded.\n";
    }else{
        g_log.debug() <<"Histogram data will be loaded.\n";
    }
    m_spec_min = getProperty("SpectrumMin");
    m_spec_max = getProperty("SpectrumMax");

    auto prog = new Progress(this, 0.0, 1.0, 10);
    prog->doReport("Start to load Nexus");

    // Open nxs file
    auto file = new ::NeXus::File(filename);
    // Find the entry name we want.
    std::string newEntryName = getEntryName(file);
    if (newEntryName != m_entryName)
    {
        throw std::runtime_error(m_entryName+" not found in the NXS file!");
    }
    std::cout<<"current path: "<<file->getPath()<<std::endl;

    // Get CSNS Nexus Version
    std::string m_version = getVersion(file);
    g_log.debug()<<"CSNS Nexus version is: " <<m_version<<"\n";
    std::cout<<"current path: "<<file->getPath()<<std::endl;

    prog->doReport("Counting pixels");
    m_numPixels = getNumPixels(file);
    g_log.debug() <<m_numPixels <<" pixels found."<< std::endl;
    std::cout<<"current path: "<<file->getPath()<<std::endl;

    prog->doReport("Counting TOF bins");
    m_numBins = getNumBins(file);
    g_log.debug() <<m_numBins <<" TOF bins found."<< std::endl;
    std::cout<<"current path: "<<file->getPath()<<std::endl;

    std::vector<std::string> bankNames= getBankNames(file);
    g_log.debug() <<bankNames.size() << " banks found." << std::endl;
    prog->setNumSteps(bankNames.size() + 5);
    std::cout<<"current path: "<<file->getPath()<<std::endl;

//load instrument
    std::vector<std::vector<float>>pixelOffset = getPixelOffset(file,bankNames[0]);
    m_xOffset = pixelOffset[0];
    m_yOffset = pixelOffset[1];
    std::cout<<"xoffsets size is "<<m_xOffset.size()<<std::endl;
    std::vector<uint32_t> m_pixelId = getPixelId(file, bankNames[0]);
    std::cout<<"pixel_id size is "<<m_pixelId.size()<<std::endl;
    std::cout<<"current path: "<<file->getPath()<<std::endl;

    // Start with a dummy WS just to hold the logs and load the instrument
    prog->doReport("Creating workspace");
    MatrixWorkspace_sptr WS = boost::dynamic_pointer_cast<MatrixWorkspace>( WorkspaceFactory::Instance().create("Workspace2D", m_numPixels, m_numBins + 1, m_numBins));
    std::cout<<"current path: "<<file->getPath()<<std::endl;

    // Load log of source, chopper and SE
    prog->doReport("Loading time log");
    loadNXLog(file, WS);
    std::cout<<"current path: "<<file->getPath()<<std::endl;

    file->openPath("/csns/logs/good_frame");
    Kernel::Property *logValue1 = createTimeSeries(file, "good_frame");
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    std::cout<<"current path: "<<file->getPath()<<std::endl;
    //
    file->openPath("/csns/logs/proton_charge");
    Kernel::Property *logValue2 = createTimeSeries(file, "proton_charge");
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    //
    file->openPath("/csns/logs/sample_environment/temperature");
    Kernel::Property *logValue3 = createTimeSeries(file, "temperature");
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    //
    file->openPath("/csns/logs/disk_chopper0/phase");
    Kernel::Property *logValue4 = createTimeSeries(file, "phase");
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    file->closeGroup();

    (void)logValue1;
    (void)logValue2;
    (void)logValue3;
    (void)logValue4;

    // close nxs file
    file->close();
    // Set to the output
    //setProperty("OutputWorkspace", boost::dynamic_pointer_cast<MatrixWorkspace>(WS));
    setProperty("OutputWorkspace", WS);

    delete prog;
}

} // namespace DataHandling
} // namespace Mantid
