#ifndef MANTID_DATAHANDLING_LOADCSNSNEXUS_H_
#define MANTID_DATAHANDLING_LOADCSNSNEXUS_H_

//----------------------------------------------------------------------
// Includes
//----------------------------------------------------------------------
#include "MantidAPI/IFileLoader.h"
#include "MantidDataObjects/EventWorkspace.h"
#include <nexus/NeXusFile.hpp>
#include <nexus/NeXusException.hpp>
#include "MantidDataObjects/Events.h"
#include "MantidAPI/WorkspaceGroup.h"
#include "MantidKernel/TimeSeriesProperty.h"
#include "MantidDataHandling/EventWorkspaceCollection.h"
#include <memory>
#include <boost/lexical_cast.hpp>


namespace Mantid
{

namespace DataHandling
{
/**
 Loads a NeXus file that conforms to the CSNSNexus instrument definition format and
 stores it in a 2D workspace.

 Copyright &copy; 2007-8 ISIS Rutherford Appleton Laboratory, NScD Oak Ridge
 National Laboratory & European Spallation Source

 This file is part of Mantid.

 Mantid is free software; you can redistribute it and/or modify
 it under the terms of the GNU General Public License as published by
 the Free Software Foundation; either version 3 of the License, or
 (at your option) any later version.

 Mantid is distributed in the hope that it will be useful,
 but WITHOUT ANY WARRANTY; without even the implied warranty of
 MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 GNU General Public License for more details.

 You should have received a copy of the GNU General Public License
 along with this program.  If not, see <http://www.gnu.org/licenses/>.

 File change history is stored at: <https://github.com/mantidproject/mantid>
 */
struct CSNSsample
{
    std::string name;
    std::string type;
    std::string situation;
    std::string chemical_formula;
    std::string shape;
    float density;    
    float mass;    
    float distance;    
    std::vector<float> size ;    
};


class DLLExport LoadCSNSNexus
    : public API::IFileLoader<Kernel::NexusDescriptor>
{
public:
    /// Default Constructor
    LoadCSNSNexus();

    // Destructor
    virtual ~LoadCSNSNexus() {}

    /// Algorithm's name for identification overriding a virtual method
    virtual const std::string name() const
    {
        return "LoadCSNSNexus";
    }

    /// Summary of algorithms purpose
    virtual const std::string summary() const
    {
        return "Loads a NeXus file confirming to the CSNSNexus format";
    }

    /// Algorithm's version for identification overriding a virtual method
    virtual int version() const
    {
        return 1;
    }

    /// Algorithm's category for identification overriding a virtual method
    virtual const std::string category() const
    {
        return "DataHandling\\Nexus";
    }

    /// get entry name
    std::string getEntryName(::NeXus::File *file);

    /// get CSNS Nexus version
    std::string getVersion(::NeXus::File *file);

    /// Return true if nxs has histogram data
    bool hasHistogramData(::NeXus::File *file);

    /// Return true if nxs has event data
    bool hasEventData(::NeXus::File *file);

    /// List bank names
    std::vector<std::string> getBankNames(::NeXus::File *file);

    /// Return number of pixels of all banks
    size_t getNumPixels(::NeXus::File *file, bool bm);

    /// Return number of bins of TOF/d/Q
    size_t getNumBins(::NeXus::File *file, bool bm);

    /// Load Log to workspace
    void loadLog(::NeXus::File *file, boost::shared_ptr<API::MatrixWorkspace> workspace) const;
   // void loadLog(::NeXus::File *file, EventWorkspace_sptr workspace) const;

    /// Create a time series property
    Kernel::Property *createTimeSeries(::NeXus::File *file, const std::string &prop_name) const;

    /// Return pixel offset of x and y
    std::vector<std::vector<float>> getPixelOffset(::NeXus::File *file, const std::string &bankName);

    /// get pixel_id
    std::vector<uint32_t> getPixelId(::NeXus::File *file, bool bm);

    /// get event pixel_id
//    std::vector<int> getEventPixelId(::NeXus::File *file, bool bm);

    /// Returns a confidence value that this algorithm can load a file
    virtual int confidence(Kernel::NexusDescriptor &descriptor) const;

    ///entry name
    std::string m_entryName;

    /// get start_time
    Kernel::DateAndTime getStartTime(::NeXus::File *file);

    /// get end_time
    Kernel::DateAndTime getEndTime(::NeXus::File *file);
    /// get instrument name
    std::string getInstrumentName(::NeXus::File *file);

    ///bank name
    std::vector<std::string> m_bankNames;

    ///monitor name
    std::vector<std::string> m_monitorNames;

    /// Number of pixels in banks
    size_t m_bNumPixels;
    /// Number of pixels in monitors
    size_t m_mNumPixels;
    /// Number of bins in banks
    size_t m_bNumBins;
    /// Number of bins in monitor
    size_t m_mNumBins;

    /// x_pixel_offset
    std::vector<float> m_xOffset;

    /// y_pixel_offset
    std::vector<float> m_yOffset;

    /// get event data
    std::multimap<int, std::pair<int64_t, double>> getEventData(::NeXus::File *file, int64_t startTime, bool bm);

    /// get spectra mapping
    std::map<int, int> getSpectraMapping(::NeXus::File *file, bool bm);

    /// get sample information
    CSNSsample getSampleInfo(::NeXus::File *file);

    /// load event data
    void loadEventData(::NeXus::File *file, size_t numPixels);


protected:
    void init();
    void exec();


    /// List of the absolute time of each pulse
    std::vector<Kernel::DateAndTime> pulseTimes;

    /// Number of bins
    size_t m_numBins;

    /// Interval of chunk
    specid_t m_spec_min, m_spec_max;

    /// Mutex to avoid simultaneous file access
    Kernel::Mutex m_fileMutex;

};

} // namespace DataHandling
} // namespace Mantid

#endif /* MANTID_DATAHANDLINGLOADCSNSNEXUS_H_ */
