#include "MantidAPI/FileProperty.h"
#include "MantidAPI/MatrixWorkspace.h"
#include "MantidAPI/RegisterFileLoader.h"
#include "MantidDataHandling/LoadEventNexus.h"
#include "MantidDataHandling/LoadCSNSNexus.h"
#include "MantidKernel/ArrayProperty.h"
#include "MantidKernel/BoundedValidator.h"
#include "MantidKernel/cow_ptr.h"
#include <nexus/NeXusFile.hpp>
#include <boost/algorithm/string/detail/classification.hpp>
#include <boost/algorithm/string/split.hpp>
#include <boost/scoped_array.hpp>

namespace Mantid {
namespace DataHandling {

DECLARE_NEXUS_FILELOADER_ALGORITHM(LoadCSNSNexus)

using namespace Kernel;
using namespace API;
using namespace DataObjects;

LoadCSNSNexus::LoadCSNSNexus()
    : m_numPixels(0), m_signalNo(0), pulseTimes(0), m_numBins(0), m_spec_min(0),
      m_spec_max(0), m_dataField(""), m_axisField(""), m_xUnits(""),
      m_fileMutex(), m_assumeOldFile(false) {}

//-------------------------------------------------------------------------------------------------
/// Initialisation method.
void LoadCSNSNexus::init() {
  declareProperty(
      new FileProperty("Filename", "", FileProperty::Load, {".nxs"}),
      "The name of the NeXus file to load");
  declareProperty(new WorkspaceProperty<Workspace>("OutputWorkspace", "",
                                                         Direction::Output),
                  "The name of the Workspace2D to create.");

  declareProperty("Signal", 1,
                  "Number of the signal to load from the file. Default is 1 = "
                  "time_of_flight.\n"
                  "Some NXS files have multiple data fields giving binning in "
                  "other units (e.g. d-spacing or momentum).\n"
                  "Enter the right signal number for your desired field.");
  auto mustBePositive = boost::make_shared<BoundedValidator<int>>();
  mustBePositive->setLower(1);
  declareProperty(
      new PropertyWithValue<specid_t>("SpectrumMin", 1, mustBePositive),
      "The index number of the first spectrum to read.  Only used if\n"
      "spectrum_max is set.");
  declareProperty(
      new PropertyWithValue<specid_t>("SpectrumMax", Mantid::EMPTY_INT(),
                                      mustBePositive),
      "The number of the last spectrum to read. Only used if explicitly\n"
      "set.");
}

//-------------------------------------------------------------------------------------------------
/**
 * Return the confidence with with this algorithm can load the file
 * @param descriptor A descriptor for the file
 * @returns An integer specifying the confidence level. 0 indicates it will not
 * be used
 */
int LoadCSNSNexus::confidence(Kernel::NexusDescriptor &descriptor) const {
  int confidence(0);
  if (descriptor.pathOfTypeExists("/csns", "NXentry")) {
    const bool hasEventData = descriptor.classTypeExists("NXevent_data");
    const bool hasHistogramData = descriptor.classTypeExists("NXdata");
    if (hasHistogramData && hasEventData)
      // Event data = this is event NXS
      confidence = 20;
    else if (hasHistogramData && !hasEventData)
      // No event data = this is the one
      confidence = 80;
    else
      // No data ?
      confidence = 10;
  }
  return confidence;
}

//-------------------------------------------------------------------------------------------------
/**
 * Return entry name
 * @param[in] filename :: nxs file name
 * @returns newEntry :: entry name
 */
std::string LoadCSNSNexus::getEntryName(const std::string &filename)
{
    std::string newEntry="";

    auto file = new ::NeXus::File(filename);
    std::map<std::string, std::string> entries = file->getEntries();
    file->close();
    delete file;

    if (entries.empty())
    {
        throw std::runtime_error("No entries in the NXS file!");
    }else if (entries.find("csns") != entries.end()){
        newEntry = "csns";
    }else{
        newEntry = entries.begin()->first;
        //g_log.warning() << "Not CSNS NXS file!" << std::endl;    
    }

    m_entryName = newEntry;
    return newEntry;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the pixel number
* @param[in] filename :: nxs file name
* @return numPixels
*/
size_t LoadCSNSNexus::getNumPixels(const std::string &filename)
{
    size_t numPixels=0;
    size_t newPixels=0;

    auto file = new ::NeXus::File(filename);
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    // Look for all the banks
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        std::string name = it->first;

        if (name.substr(0, 4) == "bank")
        {
            // cout<<"name is:"<<name<<endl;
            // file->openGroup(name, "NXdetector");
            file->openGroup(name, it->second);

            std::map<std::string, std::string> entries = file->getEntries();

            if (entries.find("pixel_id") != entries.end())
            {
                file->openData("pixel_id");
                std::vector<int64_t> dims = file->getInfo().dims;
                file->closeData();

                if (!dims.empty())
                {
                    newPixels = 1;
                    for (size_t i = 0; i < dims.size(); i++)
                        newPixels *= dims[i];
                    // cout<<"newPixels is: "<<newPixels<<endl;
                    numPixels += newPixels;
                }
                // cout<<"numPixels is:"<<numPixels<<endl;
            }
            else
            {
                throw std::runtime_error("No pixels in the NXS file!");
            }
            file->closeGroup();
        }
    }
    return numPixels;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the pixel number
* @param[in] filename :: nxs file name
* @return numBins :: bins of TOF
*/
size_t LoadCSNSNexus::getNumBins(const std::string &filename)
{
    size_t numBins=0;

    auto file = new ::NeXus::File(filename);
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    // Look for all the banks
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        std::string name = it->first;
        if (name.substr(0, 4) == "bank")
        {
            file->openGroup(name, it->second);
            std::map<std::string, std::string> entries = file->getEntries();
            if (entries.find("time_of_flight") != entries.end())
            {
                file->openData("time_of_flight");
                std::vector<int64_t> dims = file->getInfo().dims;
                file->closeData();
                numBins = dims[0];
                break;
            }
        }
    }
    return numBins;
}

//-------------------------------------------------------------------------------------------------
/**
* Return Bank list NX/entry/instrument/bank*
* @param[in] filename :: nxs file name
* @return bankNames :: bank list
*/
std::vector<std::string> LoadCSNSNexus::getBankNames(const std::string &filename)
{
    std::vector<std::string> bankNames;

    // Create the root Nexus class
    auto file = new ::NeXus::File(filename);
    // Open the default data group 'entry'
    file->openGroup(m_entryName, "NXentry");
    // Also pop into the instrument
    file->openGroup("instrument", "NXinstrument");

    // Look for all the banks
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it)
    {
        if (it->first.substr(0,4)=="bank" && it->second=="NXdetector")
        {
            bankNames.push_back(it->first);
        }
    }
    file->close();
    delete file;

    return bankNames;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the pixel offset
* @param[in] filename :: nxs file name
* @param[in] bankName :: bank name
* @return pixelOffset :: [x,y]
*/

std::vector<std::vector<float> > LoadCSNSNexus::getPixelOffset(const std::string &filename, const std::string &bankName)
{
    std::vector<std::vector<float> > pixelOffset;

    auto file = new ::NeXus::File(filename);
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup(bankName, "NXdetector");

    std::vector<float> _tmp;
    file->readData("x_pixel_offset",_tmp);
    pixelOffset.push_back(_tmp);
    file->readData("y_pixel_offset",_tmp);
    pixelOffset.push_back(_tmp);

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();

    return pixelOffset;
}
//-------------------------------------------------------------------------------------------------
/**
 * Creates a time series property from the currently opened log entry. It is
 * assumed to
 * have been checked to have a time field and the value entry's name is given as
 * an argument
 * @param file :: A reference to the file handle
 * @param prop_name :: The name of the property
 * @returns A pointer to a new property containing the time series
 */
Kernel::Property *LoadCSNSNexus::createTimeSeries(::NeXus::File &file,
                                const std::string &prop_name) const {
  file.openData("time");
  //----- Start time is an ISO8601 string date and time. ------
  std::string start;
  try {
    file.getAttr("start", start);
  } catch (::NeXus::Exception &) {
    start="2017-12-31T00:00:00+08:00";
  }

  // Convert to date and time
  Kernel::DateAndTime start_time = Kernel::DateAndTime(start);
  std::string time_units;
  file.getAttr("units", time_units);
  if (time_units.compare("second") < 0 && time_units != "s" &&
      time_units != "minutes") // Can be s/second/seconds/minutes
  {
    file.closeData();
    throw ::NeXus::Exception("Unsupported time unit '" + time_units + "'");
  }
  //--- Load the seconds into a double array ---
  std::vector<double> time_double;
  try {
    file.getDataCoerce(time_double);
  } catch (::NeXus::Exception &e) {
    g_log.warning() << "Log entry's time field could not be loaded: '"
                    << e.what() << "'.\n";
    file.closeData();
    throw;
  }
  file.closeData(); // Close time data
  g_log.debug() << "   done reading \"time\" array\n";
  // Convert to seconds if needed
  if (time_units == "minutes") {
    std::transform(time_double.begin(), time_double.end(), time_double.begin(),
                   std::bind2nd(std::multiplies<double>(), 60.0));
  }
  // Now the values: Could be a string, int or double
  file.openData("value");
  // Get the units of the property
  std::string value_units("");
  try {
    file.getAttr("units", value_units);
  } catch (::NeXus::Exception &) {
    // Ignore missing units field.
    value_units = "";
  }

  // Now the actual data
  ::NeXus::Info info = file.getInfo();
  // Check the size
  if (size_t(info.dims[0]) != time_double.size()) {
    file.closeData();
    throw ::NeXus::Exception("Invalid value entry for time series");
  }
  if (file.isDataInt()) // Int type
  {
    std::vector<int> values;
    try {
      file.getDataCoerce(values);
      file.closeData();
    } catch (::NeXus::Exception &) {
      file.closeData();
      throw;
    }
    // Make an int TSP
    auto tsp = new TimeSeriesProperty<int>(prop_name);
    tsp->create(start_time, time_double, values);
    tsp->setUnits(value_units);
    g_log.debug() << "   done reading \"value\" array\n";
    return tsp;
  } else if (info.type == ::NeXus::CHAR) {
    std::string values;
    const int64_t item_length = info.dims[1];
    try {
      const int64_t nitems = info.dims[0];
      const int64_t total_length = nitems * item_length;
      boost::scoped_array<char> val_array(new char[total_length]);
      file.getData(val_array.get());
      file.closeData();
      values = std::string(val_array.get(), total_length);
    } catch (::NeXus::Exception &) {
      file.closeData();
      throw;
    }
    // The string may contain non-printable (i.e. control) characters, replace
    // these
    std::replace_if(values.begin(), values.end(), iscntrl, ' ');
    auto tsp = new TimeSeriesProperty<std::string>(prop_name);
    std::vector<DateAndTime> times;
    DateAndTime::createVector(start_time, time_double, times);
    const size_t ntimes = times.size();
    for (size_t i = 0; i < ntimes; ++i) {
      std::string value_i =
          std::string(values.data() + i * item_length, item_length);
      tsp->addValue(times[i], value_i);
    }
    tsp->setUnits(value_units);
    g_log.debug() << "   done reading \"value\" array\n";
    return tsp;
  } else if (info.type == ::NeXus::FLOAT32 || info.type == ::NeXus::FLOAT64) {
    std::vector<double> values;
    try {
      file.getDataCoerce(values);
      file.closeData();
    } catch (::NeXus::Exception &) {
      file.closeData();
      throw;
    }
    auto tsp = new TimeSeriesProperty<double>(prop_name);
    tsp->create(start_time, time_double, values);
    tsp->setUnits(value_units);
    g_log.debug() << "   done reading \"value\" array\n";
    return tsp;
  } else {
    throw ::NeXus::Exception(
        "Invalid value type for time series. Only int, double or strings are "
        "supported");
  }
}

//-------------------------------------------------------------------------------------------------
/**
* Return the pixel_id
* @param[in] filename :: nxs file name
* @param[in] bankName :: bank name
* @return pixelId :: pixel_id
*/
std::vector<uint32_t> LoadCSNSNexus::getPixelId(const std::string &filename, const std::string &bankName)
{
    std::vector<int> tmp;
    auto file = new ::NeXus::File(filename);
    file->openGroup(m_entryName, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup(bankName, "NXdetector");

    file->readData("pixel_id",tmp);
    std::vector<uint32_t> pixelId(tmp.begin(), tmp.end());

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();

    return pixelId;
}

// Function object for remove_if STL algorithm
namespace {
// Check the numbers supplied are not in the range and erase the ones that are
struct range_check {
  range_check(specid_t min, specid_t max, detid2index_map id_to_wi)
      : m_min(min), m_max(max), m_id_to_wi(id_to_wi) {}

  bool operator()(specid_t x) {
    specid_t wi = static_cast<specid_t>((m_id_to_wi)[x]);
    return (wi + 1 < m_min || wi + 1 > m_max);
  }

private:
  specid_t m_min;
  specid_t m_max;
  detid2index_map m_id_to_wi;
};
}

void LoadCSNSNexus::exec() {
  // The input properties
  std::string filename = getPropertyValue("Filename");
  m_signalNo = getProperty("Signal");
  m_spec_min = getProperty("SpectrumMin");
  m_spec_max = getProperty("SpectrumMax");

  auto prog = new Progress(this, 0.0, 1.0, 10);
  prog->doReport("Start to load Nexus");

  // Find the entry name we want.
  getEntryName(filename);
  g_log.debug() << "entry name is"<<m_entryName<<std::endl;
 
  prog->doReport("Counting pixels");
  m_numPixels = getNumPixels(filename);
  g_log.debug() <<m_numPixels <<" pixels found."<< std::endl;
  g_log.debug() <<m_numPixels <<" pixels found."<< std::endl;

  prog->doReport("Counting TOF bins");
  m_numBins = getNumBins(filename);
  g_log.debug() <<m_numBins <<" TOF bins found."<< std::endl;

  std::vector<std::string> bankNames= getBankNames(filename);
  g_log.debug() <<bankNames.size() << " banks found." << std::endl;
  prog->setNumSteps(bankNames.size() + 5);

  //load instrument
  std::vector<std::vector<float>>pixelOffset = getPixelOffset(filename,bankNames[0]);
  m_xOffset = pixelOffset[0];
  m_yOffset = pixelOffset[1];
  std::cout<<"xoffsets size is "<<m_xOffset.size()<<std::endl;
  m_pixelId = getPixelId(filename,bankNames[0]);
  std::cout<<"pixel_id size is "<<m_pixelId.size()<<std::endl;
  

  // Start with a dummy WS just to hold the logs and load the instrument
  prog->doReport("Creating workspace");
  DataObjects::Workspace2D_sptr WS = boost::dynamic_pointer_cast<DataObjects::Workspace2D>( WorkspaceFactory::Instance().create("Workspace2D", m_numPixels, m_numBins + 1, m_numBins));

  // Set name of outputworkspace
  // std::string wksname = getPropertyValue("OutputWorkspace");
  // setPropertyValue("OutputWorkspace", wksname);
  // Set to the output
  setProperty("OutputWorkspace", boost::dynamic_pointer_cast<Workspace>(WS));

  delete prog;
}

} // namespace DataHandling
} // namespace Mantid
