//#include "MantidAPI/FileProperty.h"
//#include "MantidAPI/MatrixWorkspace.h"
//#include "MantidAPI/RegisterFileLoader.h"
//#include "MantidDataHandling/LoadEventNexus.h"
//#include "MantidDataHandling/ReadCSNSNexus.h"
#include "../inc/MantidDataHandling/ReadCSNSNexus.h"
//#include "MantidKernel/ArrayProperty.h"
//#include "MantidKernel/BoundedValidator.h"
//#include "MantidKernel/cow_ptr.h"
//#include <nexus/NeXusFile.hpp>
//#include <boost/algorithm/string/detail/classification.hpp>
//#include <boost/algorithm/string/split.hpp>

#include <iostream>
#include <string>
#include <sstream>
#include <cstring>
#include <cstdlib>
#include <cstdio>
#include <vector>
#include <map>

using std::cout;
using std::endl;
using std::map;
using std::multimap;
using std::string;
using std::vector;

ReadCSNSNexus::ReadCSNSNexus(){}

//-------------------------------------------------------------------------------------------------
void ReadCSNSNexus::init() {}

void ReadCSNSNexus::loadBank(const std::string &nexusfilename,
                               const std::string &entry_name,
                               const std::string &bankName
                               ) {
}
//------------------------------------------------------------------------------------------------
/***load event data of specilized banks
@param[in] filename :: nxs file name
@param[in] bankName :: nxs file bankName name 
return eventData :: eventData(pixel_id||time_of_flight||pulse_time)
 */
std::vector<CSNSEventDataFormat> ReadCSNSNexus::getEventData(const std::string &filename, const std::string &bankName){
    std::vector<CSNSEventDataFormat> eventData;
    auto file = new ::NeXus::File(filename);
    std::string entry_name="entry";
    file->openGroup(entry_name, "NXentry");
    file->openGroup("event_data", "NXcollection");
    file->openGroup(bankName,"NXevent_data");
  
    std::vector<int> _pixelId; 
    file->readData("event_pixel_id",_pixelId);
    std::vector<float> _timeOfFlight; 
    file->readData("event_time_of_flight",_timeOfFlight);
    std::vector<float> _pulseTime; 
    file->readData("event_pulse_time",_pulseTime);
    
    CSNSEventDataFormat _eventDatatmp;
    for (size_t i=0; i<_pixelId.size(); ++i){
     	_eventDatatmp.pixelId = _pixelId[i];
     	_eventDatatmp.timeOfFlight = _timeOfFlight[i];
     	_eventDatatmp.pulseTime = _pulseTime[i];
	eventData.push_back(_eventDatatmp);
    }
    
    file->closeGroup(); 
    return eventData;
}

//-------------------------------------------------------------------------------------------------
//mantid::std::string NexusDescriptor::pathOfType(const std::string &type)
/*
bool ReadCSNSNexus::pathOfTypeExists(const std::string &path,
                                       const std::string &type) const {
  auto it = m_pathsToTypes.find(path);
  if (it != m_pathsToTypes.end()) {
    return (it->second == type);
  } else
    return false;
}
*/
//-------------------------------------------------------------------------------------------------
/*
int LoadCSNSNexus::confidence() const {
  int confidence(0); 
  if (descriptor.pathOfTypeExists("/entry", "NXentry") ||
      descriptor.pathOfTypeExists("/entry-state0", "NXentry")) {
    const bool hasEventData = descriptor.classTypeExists("NXevent_data");
    const bool hasData = descriptor.classTypeExists("NXdata");
    if (hasData && hasEventData)
      // Event data = this is event NXS
      confidence = 20;
    else if (hasData && !hasEventData)
      // No event data = this is the one
      confidence = 80;
    else
      // No data ?
      confidence = 10;
  }
  return confidence;
}
*/
//-------------------------------------------------------------------------------------------------
std::string ReadCSNSNexus::getEntryName(const std::string &filename) {
  std::string entry_name = "entry";
  auto file = new ::NeXus::File(filename);
  std::map<std::string, std::string> entries = file->getEntries();
  file->close();
  delete file;

  if (entries.empty())
    throw std::runtime_error("No entries in the NXS file!");

  if (entries.find(entry_name) == entries.end())
    throw std::runtime_error("Not CSNS NXS file!");
    entry_name = entries.begin()->first;

  return entry_name;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the CSNS NeXus definiton version
* @param[in] filename :: nxs file name
* @return version :: CSNS NeXus definition version 
*/
std::string ReadCSNSNexus::getVersion(const std::string &filename){
  std::string version;

  std::string entry_name = "entry";

  // Create the root Nexus class
  auto file = new ::NeXus::File(filename);
  // Open the default data group 'entry'
  file->openGroup(entry_name, "NXentry");

   std::map<std::string, std::string> entries = file->getEntries();

   if (entries.find("version") != entries.end()) {
          file->readData("version", version);
          return version;
    }else{
        throw std::runtime_error("No version in the NXS file!");
    }

}

//-------------------------------------------------------------------------------------------------
/**
* Return the Org NeXus version
* @param[in] filename :: nxs file name
* @return nexusVersion :: org NeXus version 
*/
std::string ReadCSNSNexus::getNexusVersion(const std::string &filename){
  std::string nexusVersion;

  // Create the root Nexus class
  auto file = new ::NeXus::File(filename);
  if (file->hasAttr("NeXus_version")){
    file->getAttr("NeXus_version", nexusVersion);
    return nexusVersion;
  }else{
    throw std::runtime_error("No Nexus version in the NXS file!");
  }  
}

//-------------------------------------------------------------------------------------------------
/**
* Return the HDF5 version
* @param[in] filename :: nxs file name
* @return hdf5Version :: HDF5 version 
*/
std::string ReadCSNSNexus::getHdf5Version(const std::string &filename){
  std::string hdf5Version;

  // Create the root Nexus class
  auto file = new ::NeXus::File(filename);
  if (file->hasAttr("HDF5_Version")){
    file->getAttr("HDF5_Version", hdf5Version);
    return hdf5Version;
  }else{
    throw std::runtime_error("No HDF5 version in the NXS file!");
  }
}

//-------------------------------------------------------------------------------------------------
/**
* Return the instrument name
* @param[in] filename :: nxs file name
* @return instrument name 
*/
std::string ReadCSNSNexus::getInstrumentName(const std::string &filename){
  std::string instrumentName;

  std::string entry_name = "entry";

  // Create the root Nexus class
  auto file = new ::NeXus::File(filename);
  // Open the default data group 'entry'
  file->openGroup(entry_name, "NXentry");

   std::map<std::string, std::string> entries = file->getEntries();

   if (entries.find("instrument_name") != entries.end()) {
          file->readData("instrument_name", instrumentName);
          return instrumentName;
    }else{
        throw std::runtime_error("No Instrument Name in the NXS file!");
    }

}

//-------------------------------------------------------------------------------------------------
/**
* Return the instrument beamline
* @param[in] filename :: nxs file name
* @return beamline :: beamline name 
*/
std::string ReadCSNSNexus::getBeamline(const std::string &filename){
  std::string beamline;

  std::string entry_name = "entry";

  // Create the root Nexus class
  auto file = new ::NeXus::File(filename);
  // Open the default data group 'entry'
  file->openGroup(entry_name, "NXentry");

   std::map<std::string, std::string> entries = file->getEntries();

   if (entries.find("beamline") != entries.end()) {
          file->readData("beamline", beamline);
          return beamline;
    }else{
        throw std::runtime_error("No Beamline Name in the NXS file!");
    }

}

//-------------------------------------------------------------------------------------------------
/**
* Return the instrument No
* @param[in] filename :: nxs file name
* @return beamlineNo :: beamline No 
*/
size_t ReadCSNSNexus::getBeamlineNo(const std::string &filename){
  std::string beamline;
  size_t beamlineNo;

  std::string entry_name = "entry";
    
  // Create the root Nexus class
  auto file = new ::NeXus::File(filename);
  // Open the default data group 'entry'
  file->openGroup(entry_name, "NXentry");

   std::map<std::string, std::string> entries = file->getEntries();

   if (entries.find("beamline") != entries.end()) {
          file->readData("beamline", beamline);
          beamlineNo = atoi(beamline.substr(2, 2).c_str()); 
          return beamlineNo;
    }else{
        throw std::runtime_error("No Beamline Name in the NXS file!");
    }

}

//-------------------------------------------------------------------------------------------------
/**
* Return the Run no
* @param[in] filename :: nxs file name
* @return run no 
*/
std::string ReadCSNSNexus::getRunNo(const std::string &filename){
    std::string runNo;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
  
    std::map<std::string, std::string> entries = file->getEntries();

    if (entries.find("run_no") != entries.end()) {
          file->readData("run_no", runNo);
          return runNo;
    }else{
        throw std::runtime_error("No Run No in the NXS file!");
    }
}

//-------------------------------------------------------------------------------------------------
/**
* Return the Accelerator Run no
* @param[in] filename :: nxs file name
* @return accelerator run no 
*/
std::string ReadCSNSNexus::getAcceleratorRunNo(const std::string &filename){
    std::string acceleratorRunNo;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
  
    std::map<std::string, std::string> entries = file->getEntries();

    if (entries.find("accelerator_run_no") != entries.end()) {
          file->readData("accelerator_run_no", acceleratorRunNo);
          return acceleratorRunNo;
    }else{
        throw std::runtime_error("No Accelerator Run No in the NXS file!");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the description
* @param[in] filename :: nxs file name
* @return description 
*/
std::string ReadCSNSNexus::getDescription(const std::string &filename){
    std::string description;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
  
    std::map<std::string, std::string> entries = file->getEntries();

    if (entries.find("description") != entries.end()) {
          file->readData("description",description);
          return description;
    }else{
        throw std::runtime_error("No description in the NXS file!");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the proposal id
* @param[in] filename :: nxs file name
* @return proposalId 
*/
std::string ReadCSNSNexus::getProposalId(const std::string &filename){
    std::string proposalId;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
  
    std::map<std::string, std::string> entries = file->getEntries();

    if (entries.find("proposal_id") != entries.end()) {
          file->readData("proposal_id",proposalId);
          return proposalId;
    }else{
        throw std::runtime_error("No proposal Id in the NXS file!");
    }
}

//-------------------------------------------------------------------------------------------------
/**
* Return if filename changed (true: changed; flase: not changed)
* @param[in] filename :: nxs file name
* @param[out] oldfilename :: original file name
* @return true: changed, flase: not changed
*/
bool ReadCSNSNexus::isFilenameChanged(const std::string &newfilename, std::string &oldfilename){

  // Create the root Nexus class
  auto file = new ::NeXus::File(newfilename);
  if (file->hasAttr("file_name")){
    file->getAttr("file_name", oldfilename);
    if (oldfilename == newfilename){
      return false;
    }else{
      return true;
    }
  }else{
    throw std::runtime_error("No file name in the NXS file!");
  }
}


//-------------------------------------------------------------------------------------------------
/**
* Return the create time of file
* @param[in] filename :: nxs file name
* @return createTime :: create time
*/
std::string ReadCSNSNexus::getCreateTime(const std::string &filename){
  std::string createTime;

  // Create the root Nexus class
  auto file = new ::NeXus::File(filename);
  if (file->hasAttr("file_time")){
    file->getAttr("file_time", createTime);
    return createTime;
  }else{
    throw std::runtime_error("No file create time in the NXS file!");
  }    
}

//-------------------------------------------------------------------------------------------------
/**
* Return the start time of experiment
* @param[in] filename :: nxs file name
* @return startTime :: start time
*/
std::string ReadCSNSNexus::getStartTime(const std::string &filename){
    std::string startTime;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");

    std::map<std::string, std::string> entries = file->getEntries();

    if (entries.find("start_time") != entries.end()) {
          file->readData("start_time", startTime);
          return startTime;
    }else{
        throw std::runtime_error("No start time in the NXS file!");
    }
}

//-------------------------------------------------------------------------------------------------
/**
* Return the end time of experiment
* @param[in] filename :: nxs file name
* @return endTime :: end time
*/
std::string ReadCSNSNexus::getEndTime(const std::string &filename){
    std::string endTime;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    
    std::map<std::string, std::string> entries = file->getEntries();
    
    if (entries.find("end_time") != entries.end()) {
          file->readData("end_time", endTime);
          return endTime;
    }else{
        throw std::runtime_error("No end time in the NXS file!");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the sample name of experiment
* @param[in] filename :: nxs file name
* @return sampleName :: sample name
*/
std::string ReadCSNSNexus::getSampleName(const std::string &filename){
    std::string sampleName;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("sample", "NXsample");
 
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("name") != entries.end()) {
          file->readData("name", sampleName);
          return sampleName;
    }else{
        throw std::runtime_error("No sample name in the NXS file!");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the sample chemical formula of experiment
* @param[in] filename :: nxs file name
* @return sampleChemicalFormula :: sample chemical formula
*/
std::string ReadCSNSNexus::getSampleChemicalFormula(const std::string &filename){
    std::string sampleChemicalFormula;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("sample", "NXsample");
 
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("chemical_formula") != entries.end()) {
          file->readData("chemical_formula",sampleChemicalFormula);
          return sampleChemicalFormula;
    }else{
        throw std::runtime_error("No sample chemical formula in the NXS file!");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the sample type of experiment
* @param[in] filename :: nxs file name
* @return sampleType :: sample type
*/
std::string ReadCSNSNexus::getSampleType(const std::string &filename){
    std::string sampleType;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("sample", "NXsample");
 
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("type") != entries.end()) {
          file->readData("type", sampleType);
          return sampleType;
    }else{
        throw std::runtime_error("No sample type in the NXS file!");
    }
}

//-------------------------------------------------------------------------------------------------
/**
* Return the sample situaiton of experiment
* @param[in] filename :: nxs file name
* @return sampleSituation :: sample situation
*/
std::string ReadCSNSNexus::getSampleSituation(const std::string &filename){
    std::string sampleSituation;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("sample", "NXsample");
 
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("situation") != entries.end()) {
          file->readData("situation", sampleSituation);
          return sampleSituation;
    }else{
        throw std::runtime_error("No sample situation in the NXS file!");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the sample density
* @param[in] filename :: nxs file name
* @return sampleDensity :: sample density
*/
float ReadCSNSNexus::getSampleDensity(const std::string &filename){
    float sampleDensity;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("sample", "NXsample");
 
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("density") != entries.end()) {
          file->readData("density", sampleDensity);
          return sampleDensity;
    }else{
        throw std::runtime_error("No sample density in the NXS file!");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the sample distance
* @param[in] filename :: nxs file name
* @return sampleDistance :: sample distance
*/
float ReadCSNSNexus::getSampleDistance(const std::string &filename){
    float sampleDistance;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("sample", "NXsample");
 
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("distance") != entries.end()) {
          file->readData("distance", sampleDistance);
          return sampleDistance;
    }else{
        throw std::runtime_error("No sample distance in the NXS file!");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the sample mass
* @param[in] filename :: nxs file name
* @return sampleMass :: sample mass
*/
float ReadCSNSNexus::getSampleMass(const std::string &filename){
    float sampleMass;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("sample", "NXsample");
 
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("mass") != entries.end()) {
          file->readData("mass", sampleMass);
          return sampleMass;
    }else{
        throw std::runtime_error("No sample mass in the NXS file!");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the sample shape
* @param[in] filename :: nxs file name
* @return sampleShape :: sample shape
*/
std::string ReadCSNSNexus::getSampleShape(const std::string &filename){
    std::string sampleShape;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("sample", "NXsample");
    file->openGroup("geometry", "NXgeometry");
    file->openGroup("shape", "NXshape");
 
    std::map<std::string, std::string> entries = file->getEntries();
    if (entries.find("shape") != entries.end()) {
          file->readData("shape", sampleShape);
          return sampleShape;
    }else{
        throw std::runtime_error("No sample shape in the NXS file!");
    }
}
//-------------------------------------------------------------------------------------------------
/**
* Return the sample size
* @param[in] filename :: nxs file name
* @return sampleSize :: sample size
*/
std::vector<float> ReadCSNSNexus::getSampleSize(const std::string &filename){
    std::vector<float> sampleSize;
    std::string entry_name = "entry";
    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("sample", "NXsample");
    file->openGroup("geometry", "NXgeometry");
    file->openGroup("shape", "NXshape");
 
    std::vector<float> _tmp;
    file->readData("size", sampleSize);
  //  sampleSize.push_back(_tmp);

    //cout<<file->getPath()<<endl; 

    return sampleSize;
}
//-------------------------------------------------------------------------------------------------
/**
* Return the pixel number
* @param[in] filename :: nxs file name
* @param[in] entry_name :: nxs file entry name
* @return numPixels 
*/
size_t ReadCSNSNexus::getNumPixels(const std::string &filename, const std::string &entry_name){
    size_t numPixels=0;
    size_t newPixels;

    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    // Look for all the banks
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it) {
        std::string name = it->first;

        if (name.substr(0, 4) == "bank") {
         // cout<<"name is:"<<name<<endl; 
         // file->openGroup(name, "NXdetector");
          file->openGroup(name, it->second);
         
          std::map<std::string, std::string> entries = file->getEntries();

          if (entries.find("pixel_id") != entries.end()) {
            file->openData("pixel_id");
    	    std::vector<int64_t> dims = file->getInfo().dims;
            file->closeData();

            if (!dims.empty()) {
              size_t newPixels = 1;
              for (size_t i = 0; i < dims.size(); i++)
                newPixels *= dims[i];
              // cout<<"newPixels is: "<<newPixels<<endl;
              numPixels += newPixels;
            }
             // cout<<"numPixels is:"<<numPixels<<endl;
            } else{
              throw std::runtime_error("No pixels in the NXS file!");
            }
          file->closeGroup();
          }
    }
    return numPixels;
}

//-------------------------------------------------------------------------------------------------
/**
* Return the pixel number
* @param[in] filename :: nxs file name
* @param[in] entry_name :: nxs file entry name
* @return numBins :: bins of TOF 
*/
size_t ReadCSNSNexus::getNumBins(const std::string &filename, const std::string &entry_name){
    size_t numBins=0;

    auto file = new ::NeXus::File(filename);
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    // Look for all the banks
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it) {
        std::string name = it->first;
        if (name.substr(0, 4) == "bank") {
            file->openGroup(name, it->second);
            std::map<std::string, std::string> entries = file->getEntries();
            if (entries.find("time_of_flight") != entries.end()) {
                file->openData("time_of_flight");
                std::vector<int64_t> dims = file->getInfo().dims;
                file->closeData();
                numBins = dims[0];
                break;
            }
        }
    }
    return numBins;
}

//-------------------------------------------------------------------------------------------------
/**
* Return proton charge
* @param[in] filename :: nxs file name
* @return protonChargeLog :: [time, value]
*/
std::vector<std::vector<float> > ReadCSNSNexus::getProtonChargeLog(const std::string &filename) {
    std::vector<std::vector<float> > protonChargeLog;    

    auto file = new ::NeXus::File(filename);

    std::string entry_name = "entry";
    file->openGroup(entry_name, "NXentry");
    file->openGroup("logs", "NXcollection");
    file->openGroup("proton_charge", "NXlog");

    std::vector<float> _tmp;
    file->readData("time", _tmp);
    protonChargeLog.push_back(_tmp);

    file->readData("value", _tmp);
    protonChargeLog.push_back(_tmp);
   
    file->closeGroup(); 
    file->closeGroup(); 
    file->closeGroup();
//    cout<<file->getPath()<<endl; 

    return protonChargeLog;

}

//-------------------------------------------------------------------------------------------------
/**
* Return sample environment list
* @param[in] filename :: nxs file name
* @return sampleEnvironment :: sample environment list
*/
std::vector<std::string> ReadCSNSNexus::getSampleEnvironment(const std::string &filename){
    std::vector<std::string > sampleEnvironment;

    auto file = new ::NeXus::File(filename);

    std::string entry_name = "entry";
    file->openGroup(entry_name, "NXentry");
    file->openGroup("logs", "NXcollection");
    file->openGroup("sample_environment", "NXcollection");

    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it) {
        std::string type = it->second;
        if (type == "NXlog") {
            sampleEnvironment.push_back(it->first);
        }
    }

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();

    return sampleEnvironment;
}


//-------------------------------------------------------------------------------------------------
/**
* Return sample Environment log
* @param[in] filename :: nxs file name
* @param[in] SEname :: sample Environment name
* @return sampleEnvironment :: [time, value]
*/
std::vector<std::vector<float> > ReadCSNSNexus::getSampleEnvironmentLog(const std::string &filename, const std::string &SEname) {
    std::vector<std::vector<float> > sampleEnvironment;

    auto file = new ::NeXus::File(filename);

    std::string entry_name = "entry";
    file->openGroup(entry_name, "NXentry");
    file->openGroup("logs", "NXcollection");
    file->openGroup("sample_environment", "NXcollection");

    std::map<std::string, std::string> entries = file->getEntries();

    if (entries.find(SEname) != entries.end()) {
        file->openGroup(SEname, "NXlog");

        std::vector<float> _tmp;
        file->readData("time", _tmp);
        sampleEnvironment.push_back(_tmp);

        file->readData("value", _tmp);
        sampleEnvironment.push_back(_tmp);
        file->closeGroup();
    }else{
        throw std::runtime_error("No "+SEname+" in the NXS file!");
    }

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();
    //cout<<file->getPath()<<endl;

    return sampleEnvironment;
}
//-------------------------------------------------------------------------------------------------
/**
* Return disk_chopper1 list
* @param[in] filename :: nxs file name
* @return diskChopper1 :: disk_chopper1 list
*/
std::vector<std::string> ReadCSNSNexus::getDiskChopper1(const std::string &filename){
    std::vector<std::string > diskChopper1;

    auto file = new ::NeXus::File(filename);

    std::string entry_name = "entry";
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("disk_chopper1", "NXdisk_chopper");

    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    for (it = entries.begin(); it != entries.end(); ++it) {
         diskChopper1.push_back(it->first);
        }

    file->closeGroup();
    file->closeGroup();
    file->closeGroup();

    return diskChopper1;
}


//-------------------------------------------------------------------------------------------------
/**
* Return disk_chopper1 log
* @param[in] filename :: nxs file name
* @param[in] DCname :: disk_chopper1 name
* @return diskChopper1 :: value
*/
float ReadCSNSNexus::getDiskChopper1Log(const std::string &filename, const std::string &DCname) {
   float diskChopper1Log;

    auto file = new ::NeXus::File(filename);

    std::string entry_name = "entry";
    file->openGroup(entry_name, "NXentry");
    file->openGroup("instrument", "NXinstrument");
    file->openGroup("disk_chopper1", "NXdisk_chopper");
    std::map<std::string, std::string> entries = file->getEntries();
          if (entries.find(DCname) != entries.end()) {
              file->readData(DCname, diskChopper1Log);
           }else{
                 throw std::runtime_error("No "+DCname+" in the NXS file!");
           }

    file->closeGroup();
    file->closeGroup();

    return diskChopper1Log;
}

//-------------------------------------------------------------------------------------------------
/**
* Return Good Frame Log
* @param[in] filename :: nxs file name
* @return goodFrameLog :: [time, value]
*/
std::vector<std::vector<float> > ReadCSNSNexus::getGoodFrameLog(const std::string &filename) {
    std::vector<std::vector<float> > goodFrameLog;    

    auto file = new ::NeXus::File(filename);

    std::string entry_name = "entry";
    file->openGroup(entry_name, "NXentry");
    file->openGroup("logs", "NXcollection");
    file->openGroup("good_frame", "NXlog");

    std::vector<float> _tmp;
    file->readData("time", _tmp);
    goodFrameLog.push_back(_tmp);

    file->readData("value", _tmp);
    goodFrameLog.push_back(_tmp);
   
    file->closeGroup(); 
    file->closeGroup(); 
    file->closeGroup();
//    cout<<file->getPath()<<endl; 

    return goodFrameLog;
}

//-------------------------------------------------------------------------------------------------
/**
* Return chopper phase Log
* @param[in] filename :: nxs file name
* @return chopperPhaseLog :: [time, value]
*/
std::vector<std::vector<float> > ReadCSNSNexus::getChopperPhaseLog(const std::string &filename) {
    std::vector<std::vector<float> > chopperPhaseLog;    

    auto file = new ::NeXus::File(filename);

    std::string entry_name = "entry";
    file->openGroup(entry_name, "NXentry");
    file->openGroup("logs", "NXcollection");
    file->openGroup("good_frame", "NXlog");

    std::vector<float> _tmp;
    file->readData("time", _tmp);
    chopperPhaseLog.push_back(_tmp);

    file->readData("value", _tmp);
    chopperPhaseLog.push_back(_tmp);
   
    file->closeGroup(); 
    file->closeGroup(); 
    file->closeGroup();
    //cout<<file->getPath()<<endl; 

    return chopperPhaseLog;
}
//-------------------------------------------------------------------------------------------------
/**
* Return users
* @param[in] filename :: nxs file name
* @return users :: users
*/
std::vector<CSNSuser> ReadCSNSNexus::getUsers(const std::string &filename) {
    std::vector<CSNSuser> users;

    auto file = new ::NeXus::File(filename);

    std::string entry_name = "entry";
    file->openGroup(entry_name, "NXentry");
    file->openGroup("users", "NXcollection");
    // Look for all the banks
    std::map<std::string, std::string> entries = file->getEntries();
    std::map<std::string, std::string>::iterator it;
    CSNSuser _user;
    std::string _tmp;
    for (it = entries.begin(); it != entries.end(); ++it) {
        std::string type = it->second;
        if (type == "NXuser") {
            file->openGroup(it->first, "NXuser");
            
            file->readData("address", _tmp);
            _user.address=_tmp;
            
            file->readData("affiliation", _tmp);
            _user.affiliation=_tmp;
            
            file->readData("email", _tmp);
            _user.email=_tmp;
            
            file->readData("facility_user_id", _tmp);
            _user.facility_user_id=_tmp;
            
            file->readData("name", _tmp);
            _user.name=_tmp;
            
            file->readData("role", _tmp);
            _user.role=_tmp;

            users.push_back(_user);

            file->closeGroup();
        }
    }
    file->closeGroup();
    return users;
}

//-------------------------------------------------------------------------------------------------
std::vector<std::string> ReadCSNSNexus::getBankNames(const std::string &filename) {
  std::vector<std::string> bankNames;

  std::string entry_name = "entry";

  // Create the root Nexus class
  auto file = new ::NeXus::File(filename);
  // Open the default data group 'entry'
  file->openGroup(entry_name, "NXentry");
  // Also pop into the instrument
  file->openGroup("instrument", "NXinstrument");

  // Look for all the banks
  std::map<std::string, std::string> entries = file->getEntries();
  std::map<std::string, std::string>::iterator it;
  for (it = entries.begin(); it != entries.end(); ++it) {
    if (it->first.substr(0,4)=="bank" and it->second=="NXdetector"){
    bankNames.push_back(it->first);
    }
  }
  return bankNames;
}

// exec
void ReadCSNSNexus::exec(const std::string &filename, size_t m_signalNo, size_t m_spec_min, size_t m_spec_max){

cout<<"====================================="<<endl;
std::string oldfilename;
if (isFilenameChanged(filename, oldfilename)){
    cout<<"Filename changed: from "<< oldfilename << " to "<<filename.substr(filename.find_last_of("/\\") + 1)<<endl;
}else{
    cout<<"Filename not changed"<<endl;
}

cout<<"====================================="<<endl;
std::string entry_name = getEntryName(filename);
cout<<"entry is: "<<entry_name<<endl;

cout<<"====================================="<<endl;
std::vector<std::string> bankNames = getBankNames(filename);
cout<<bankNames.size()<<" banks found!"<<endl;
for (auto i = bankNames.begin(); i != bankNames.end(); ++i){std::cout<<"    " << *i <<endl;}

cout<<"====================================="<<endl;
std::string instrumentName = getInstrumentName(filename);
cout<<"instrument name is: "<<instrumentName<<endl;

cout<<"====================================="<<endl;
std::string runNo = getRunNo(filename);
cout<<"run no is: "<<runNo<<endl;

cout<<"====================================="<<endl;
std::string acceleratorRunNo = getAcceleratorRunNo(filename);
cout<<"accelerator run no is: "<<acceleratorRunNo<<endl;

cout<<"====================================="<<endl;
std::string description = getDescription(filename);
cout<<"description is: "<<description<<endl;

cout<<"====================================="<<endl;
std::string proposalId = getProposalId(filename);
cout<<"proposal id is: "<<proposalId<<endl;

cout<<"====================================="<<endl;
size_t numPixels = getNumPixels(filename, entry_name);
cout<<"Total Pixel number is: "<<numPixels<<endl;

cout<<"====================================="<<endl;
size_t numBins = getNumBins(filename, entry_name);
cout<<"The TOF number is: "<<numBins<<endl;

cout<<"====================================="<<endl;
std::string version = getVersion(filename);
cout<<"CSNS NeXus version is: "<<version<<endl;

cout<<"====================================="<<endl;
std::string nexusVersion = getNexusVersion(filename);
cout<<"ORG NeXus version is: "<<nexusVersion<<endl;

cout<<"====================================="<<endl;
std::string hdf5Version = getHdf5Version(filename);
cout<<"HDF5 version is: "<<hdf5Version<<endl;

cout<<"====================================="<<endl;
std::string createTime = getCreateTime(filename);
cout<<"File create time is: "<<createTime<<endl;

cout<<"====================================="<<endl;
std::string startTime = getStartTime(filename);
cout<<"Start time is: "<<startTime<<endl;

cout<<"====================================="<<endl;
std::string endTime = getEndTime(filename);
cout<<"End time is: "<<endTime<<endl;

cout<<"====================================="<<endl;
std::string beamline = getBeamline(filename);
cout<<"The beamline is: "<<beamline<<endl;

cout<<"====================================="<<endl;
size_t beamlineNo = getBeamlineNo(filename);
cout<<"The beamline No. is: "<<beamlineNo<<endl;

cout<<"====================================="<<endl;
std::vector<CSNSuser> users = getUsers(filename);
cout<<users.size()<<" users found!"<<endl;
for (auto i = users.begin(); i != users.end(); ++i){cout<<"    "<< (*i).name  <<endl;}

cout<<"====================================="<<endl;
std::vector<std::vector<float> > protonChargeLog =getProtonChargeLog(filename);
cout<<protonChargeLog[1].size()<<" proton charge log found!"<<endl;
double protonChargeMin = *min_element(protonChargeLog[1].begin(), protonChargeLog[1].end());
double protonChargeMax = *max_element(protonChargeLog[1].begin(), protonChargeLog[1].end());
cout<<"    min proton charge is: "<<protonChargeMin<<endl;
cout<<"    max proton charge is: "<<protonChargeMax<<endl;

cout<<"====================================="<<endl;
std::vector<std::vector<float> > goodFrameLog =getGoodFrameLog(filename);
cout<<goodFrameLog[0].size()<<" pulse frame log found!"<<endl;
size_t numGoodFrame=0;
for (size_t j=0; j<goodFrameLog[1].size(); ++j){
    if (goodFrameLog[1][j] == 1.0){numGoodFrame+=1;}
}
cout<<"    "<<numGoodFrame<<"/"<<goodFrameLog[1].size()<<" good frame found!"<<endl;


cout<<"====================================="<<endl;
std::vector<std::vector<float> > chopperPhaseLog =getChopperPhaseLog(filename);
cout<<chopperPhaseLog[0].size()<<" chopper phase log found!"<<endl;
size_t numGoodChopper=0;
for (size_t j=0; j<chopperPhaseLog[1].size(); ++j){
    if (chopperPhaseLog[1][j] == 1.0){numGoodChopper+=1;}
}
cout<<"    "<<numGoodChopper<<"/"<<chopperPhaseLog[1].size()<<" good chopper phase found!"<<endl;

cout<<"====================================="<<endl;
std::vector<std::string> sampleEnvironment = getSampleEnvironment(filename);
cout<<sampleEnvironment.size()<<" sample environments found!"<<endl;
for (auto i = sampleEnvironment.begin(); i != sampleEnvironment.end(); ++i){std::cout<<"    " << *i <<endl;}

cout<<"====================================="<<endl;
std::vector<std::vector<float> > sampleEnvironmentLog = getSampleEnvironmentLog(filename, "temperature");
if (sampleEnvironmentLog.size()==2){
    for (size_t j=0; j<sampleEnvironmentLog[0].size(); ++j){
        cout<<"time||temperature : "<<sampleEnvironmentLog[0][j]<<"||"<<sampleEnvironmentLog[1][j]<<endl;
    }
}

cout<<"====================================="<<endl;
std::vector<std::string> diskChopper1 = getDiskChopper1(filename);
for (auto i = diskChopper1.begin(); i != diskChopper1.end(); ++i){std::cout<<"The Disk Chopper1 is: " << *i <<endl;}

cout<<"====================================="<<endl;
float diskChopper1Log = getDiskChopper1Log(filename, "distance");
cout<<"distance of disk chopper1 is :"<<diskChopper1Log<<endl;

cout<<"====================================="<<endl;
std::string sampleName = getSampleName(filename);
cout<<"Sample name is: "<<sampleName<<endl;

cout<<"====================================="<<endl;
std::string sampleChemicalFormular = getSampleChemicalFormula(filename);
cout<<"Sample chemical formula is: "<<sampleChemicalFormular<<endl;

cout<<"====================================="<<endl;
std::string sampleType = getSampleType(filename);
cout<<"Sample type is: "<<sampleType<<endl;

cout<<"====================================="<<endl;
std::string sampleSituation = getSampleSituation(filename);
cout<<"Sample situation is: "<<sampleSituation<<endl;

cout<<"====================================="<<endl;
float sampleDensity = getSampleDensity(filename);
cout<<"Sample density is: "<<sampleDensity<<endl;

cout<<"====================================="<<endl;
float sampleDistance = getSampleDistance(filename);
cout<<"Sample distance is: "<<sampleDistance<<endl;

cout<<"====================================="<<endl;
float sampleMass = getSampleMass(filename);
cout<<"Sample mass is: "<<sampleMass<<endl;

cout<<"====================================="<<endl;
std::string sampleShape = getSampleShape(filename);
cout<<"Sample shape is: "<<sampleShape<<endl;

cout<<"====================================="<<endl;
std::vector<float> sampleSize =getSampleSize(filename);
for (auto i = sampleSize.begin(); i != sampleSize.end(); ++i){std::cout<<"sample size is: " << *i <<endl;}

std::vector<CSNSEventDataFormat> eventData = getEventData(filename, "bank0");
for (auto i = eventData.begin(); i != eventData.end(); ++i){
     cout<<"EventData is:PID||TOF||PT  "<< (*i).pixelId<<"||"<<(*i).timeOfFlight<<"||"<<(*i).pulseTime<<endl;}

//mantid:: Create workspace
//MatrixWorkspace_sptr WS = WorkspaceFactory::Instance().create("Workspace2D", numPixels, numBins + 1, numBins);

//
}

int main(int argc, char** argv){
std::string filename="../../../Examples/BL18_GPPD_0000001.nxs";
size_t  m_signalNo = 1;
size_t  m_spec_min = 1;
size_t  m_spec_max = 1;


ReadCSNSNexus *readnxs = new ReadCSNSNexus();
readnxs->exec(filename, m_signalNo, m_spec_min, m_spec_max);

};
