/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package cn.ac.csns.nexus.csnsnexustemplate;

import java.math.BigDecimal;
import java.math.BigInteger;
import java.math.RoundingMode;
import java.text.SimpleDateFormat;
import java.time.Instant;
import java.util.Calendar;
import java.util.TimeZone;
import org.threeten.extra.scale.TaiInstant;
import org.threeten.extra.scale.UtcInstant;

/**
 *
 * @author lanjian
 */
public class CalendarTool {

    /**
     * Get current date and time as a formated string.
     *
     * @param format a string limit how date and time are returned. eg.
     * "yyyy-MM-dd'T'HH:mm:ss"
     * @param isTimeDiff a boolean limit if time zone offset is returned. Such
     * as +08:00 for China.
     * <ul>
     * <li><code>true</code> return the time zone offset</li>
     * <li><code>false</code> don't return the time zone offset</li>
     * </ul>
     * @return
     */
    public static String getFormatedCurrentDate(String format, Boolean isTimeDiff) {
        String result;
        int hour;
        int minute;
        int diff;
        String offset = "";
        TimeZone tz;

        Calendar calendar = Calendar.getInstance();
        SimpleDateFormat sdf = new SimpleDateFormat(format);
        result = sdf.format(calendar.getTime());

        if (isTimeDiff) {
            tz = calendar.getTimeZone();
            diff = tz.getRawOffset();
            diff = Math.abs(diff / 1000);
            hour = diff / 3600;
            minute = (diff - hour * 3600) / 60;
            offset = String.format("%02d:%02d", hour, minute);
            if (tz.getRawOffset() >= 0) {
                offset = "+" + offset;
            } else {
                offset = "-" + offset;
            }
        }
        return result + offset;
    }

    /**
     * Obtains an instance of UtcInstant from the number of TAI seconds from the
     * TAI epoch of 1958-01-01T00:00:00(TAI) with a nanosecond fraction of
     * second.
     *
     * The UtcInstant will represent exactly the same point on the time-line as
     * per the available leap-second rules. If the leap-second rules change then
     * conversion back to TAI may result in a different instant.
     *
     * @param taiSeconds the number of seconds from the epoch of
     * 1958-01-01T00:00:00(TAI)
     * @param nanoAdjustment the nanosecond adjustment to the number of seconds,
     * positive or negative
     * @return a UtcInstant representing the same instant, not null
     */
    public static UtcInstant getUtcInstantFromTaiSeconds(long taiSeconds, long nanoAdjustment) {
        return TaiInstant.ofTaiSeconds(taiSeconds, nanoAdjustment).toUtcInstant();
    }

    /**
     * Converts a TAI instant from TAI seconds to an Instant.
     *
     * Converting a TAI instant to UTC-SLS requires leap second rules. This
     * method uses the latest available system rules. The conversion first maps
     * from TAI to UTC, then converts to UTC-SLS.
     *
     * Conversion to an Instant will not be completely accurate near a leap
     * second in accordance with UTC-SLS.
     *
     * @param taiSeconds the number of seconds from the epoch of
     * 1958-01-01T00:00:00(TAI)
     * @param nanoAdjustment the nanosecond adjustment to the number of seconds,
     * positive or negative
     * @return an Instant representing the best approximation of this instant,
     * not null
     */
    public static Instant getInstantFromTaiSeconds(long taiSeconds, long nanoAdjustment) {
        return TaiInstant.ofTaiSeconds(taiSeconds, nanoAdjustment).toInstant();
    }

    /**
     * Obtains an instance of TaiInstant representing exactly the same point on
     * the time-line from an UTC datetime string.
     *
     * @param utcString an UTC datetime string. E.g. "2016-08-26T02:07:18.418Z"
     * @return the TAI instant
     */
    public static TaiInstant getTaiInstantFromUtc(String utcString) {
        return TaiInstant.of(UtcInstant.parse(utcString));
    }

    /**
     * Get total nanoseconds from TAI.
     *
     * The formula is taiSeconds*1000000000+nanoAdjustment.
     *
     * @param taiSeconds TAI seconds
     * @param nanoAdjustment TAI nano-seconds
     * @return an instance of BigInteger
     */
    public static BigInteger getTotalNanoSecondsFromTaiSeconds(long taiSeconds, long nanoAdjustment) {
        return new BigInteger(Long.toString(nanoAdjustment))
                .add(
                        new BigInteger(Long.toString(taiSeconds))
                        .multiply(new BigInteger("1000000000"))
                );
    }

    /**
     * Get total nanoseconds from TAI.
     *
     * The formula is taiSeconds*1000000000+taiNano.
     *
     * @param tai an instance of TaiInstant
     * @return an instance of BigInteger
     */
    public static BigInteger getTotalNanoSecondsFromTaiInstant(TaiInstant tai) {
        return getTotalNanoSecondsFromTaiSeconds(tai.getTaiSeconds(), tai.getNano());
    }

    /**
     * Get total nanoseconds from a UTC string.
     *
     * The UTC string is first converted to an UtcInstant and then to a
     * TaiInstant, finally the result of taiSeconds*1000000000+taiNano is
     * returned.
     *
     * @param utcString an UTC datetime string
     * @return an instance of BigInteger
     */
    public static BigInteger getTotalNanoSecondsFromUtcString(String utcString) {
        TaiInstant tai = TaiInstant.of(UtcInstant.parse(utcString));
        return getTotalNanoSecondsFromTaiInstant(tai);
    }

    /**
     * Get the reduced nanoseconds.
     *
     * The result is totalNanoSeconds/scale. If totalNanoSeconds/scale is too
     * big to fit in a long, only the low-order 64 bits are returned.
     *
     * @param totalNanoSeconds
     * @param scale
     * @return
     */
    public static long getReducedNanoSeconds(BigInteger totalNanoSeconds, int scale) {
        return new BigDecimal(totalNanoSeconds)
                .divide(new BigDecimal(Integer.toString(scale)))
                .setScale(0, RoundingMode.HALF_UP)
                .longValue();
    }

    /**
     * Convert a reduced nanoseconds to the most nearest point on the time-line.
     *
     * This conversion is not completely accurate, with the least resolution of
     * scale.
     *
     * @param reducedNanoSeconds reduced nanoseconds
     * @param scale
     * @return an instance of TaiInstant
     */
    public static TaiInstant getTaiInstantFromReducedNanoSeconds(long reducedNanoSeconds, int scale) {
        BigInteger totalNanoSeconds = new BigInteger(Long.toString(reducedNanoSeconds))
                .multiply(new BigInteger(Integer.toString(scale)));
        BigInteger[] result = totalNanoSeconds.divideAndRemainder(new BigInteger("1000000000"));
        return TaiInstant.ofTaiSeconds(result[0].longValue(), result[1].longValue());
    }

    /**
     * Convert a total nanoseconds to the exactly TAI time point on the
     * time-line.
     *
     * This conversion is not completely accurate, with the least resolution of
     * scale.
     *
     * @param totalNanoSeconds
     * @return
     */
    public static TaiInstant getTaiInstantFromNanoSeconds(BigInteger totalNanoSeconds) {
        BigInteger[] result = totalNanoSeconds.divideAndRemainder(new BigInteger("1000000000"));
        return TaiInstant.ofTaiSeconds(result[0].longValue(), result[1].longValue());
    }

    /**
     * Convert a total nanoseconds to the exactly TAI time point on the
     * time-line.
     *
     * This conversion is not completely accurate, with the least resolution of
     * scale.
     *
     * @param totalNanoSeconds
     * @return
     */
    public static TaiInstant getTaiInstantFromNanoSeconds(long totalNanoSeconds) {
        BigInteger[] result = new BigInteger(Long.toString(totalNanoSeconds))
                .divideAndRemainder(new BigInteger("1000000000"));
        return TaiInstant.ofTaiSeconds(result[0].longValue(), result[1].longValue());
    }
}
