/*
 * To change this license header, choose License Headers in Project Properties.
 * To change this template file, choose Tools | Templates
 * and open the template in the editor.
 */
package cn.ac.csns.portal.proposalsystem.facade;

import cn.ac.csns.portal.proposalsystem.exception.IpsException.IpsExceptionType;
import cn.ac.csns.portal.proposalsystem.exception.IpsException;
import java.util.List;
import javax.persistence.EntityExistsException;
import javax.persistence.EntityManager;
import javax.persistence.EntityNotFoundException;
import javax.persistence.PersistenceException;
import javax.persistence.TransactionRequiredException;

/**
 * Abstract class for entity facades.
 *
 * @author tangm <a href="mailto:tangm@ihep.ac.cn">Tang Ming</a>
 */
public abstract class AbstractFacade<T> {

    private final Class<T> entityClass;

    /**
     * Constructor.
     *
     * @param entityClass the entity class this facade operating.
     */
    public AbstractFacade(Class<T> entityClass) {
        this.entityClass = entityClass;
    }

    protected abstract EntityManager getEntityManager();

    /**
     * Make an instance managed and persistent.
     *
     * @param entity the entity.
     * @throws IpsException if the create fails.
     */
    public void create(T entity) throws IpsException {
        try {
            getEntityManager().persist(entity);
        } catch (IllegalArgumentException ex) {
            throw new IpsException(IpsExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (EntityExistsException ex) {
            throw new IpsException(IpsExceptionType.OBJECT_ALREADY_EXISTS,
                    ex.getLocalizedMessage());
        } catch (TransactionRequiredException ex) {
            throw new IpsException(IpsExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Merge the state of the given entity into the current persistence context.
     *
     * @param entity the entity.
     * @throws IpsException if edit or create fails.
     */
    public void edit(T entity) throws IpsException {
        try {
            getEntityManager().merge(entity);
        } catch (IllegalArgumentException ex) {
            throw new IpsException(IpsExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (TransactionRequiredException ex) {
            throw new IpsException(IpsExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Remove the entity instance.
     *
     * @param entity the entity.
     * @throws IpsException if the remove fails.
     */
    public void remove(T entity) throws IpsException {
        try {
            getEntityManager().remove(getEntityManager().merge(entity));
        } catch (IllegalArgumentException ex) {
            throw new IpsException(IpsExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (TransactionRequiredException ex) {
            throw new IpsException(IpsExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Synchronize the persistence context to the underlying database.
     *
     * @throws IpsException if the flush fails
     */
    public void flush() throws IpsException {
        try {
            getEntityManager().flush();
        } catch (IllegalArgumentException ex) {
            throw new IpsException(IpsExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (PersistenceException ex) {
            throw new IpsException(IpsExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Refresh the state of the instance from the database, overwriting changes
     * made to the entity, if any.
     *
     * @param entity the entity
     * @throws IpsException if the refresh fails
     */
    public void refresh(T entity) throws IpsException {
        try {
            getEntityManager().refresh(entity);
        } catch (IllegalArgumentException ex) {
            throw new IpsException(IpsExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (EntityNotFoundException ex) {
            throw new IpsException(IpsExceptionType.NO_SUCH_OBJECT_FOUND,
                    ex.getLocalizedMessage());
        } catch (TransactionRequiredException ex) {
            throw new IpsException(IpsExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Find by primary key. Search for an entity of the specified class and
     * primary key. If the entity instance is contained in the persistence
     * context, it is returned from there.
     *
     * @param id primary key
     * @return the found entity instance or null if the entity does not exist
     * @throws IpsException if id is not a valid primary key
     */
    public T find(Object id) throws IpsException {
        try {
            return getEntityManager().find(entityClass, id);
        } catch (IllegalArgumentException ex) {
            throw new IpsException(IpsExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Find all entity instances.
     * 
     * @return all entity instances
     * @throws IpsException if the find fails
     */
    public List<T> findAll() throws IpsException {
        try {
            javax.persistence.criteria.CriteriaQuery cq = getEntityManager().getCriteriaBuilder().createQuery();
            cq.select(cq.from(entityClass));
            return getEntityManager().createQuery(cq).getResultList();
        } catch (IllegalArgumentException ex) {
            throw new IpsException(IpsExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (Exception ex) {
            throw new IpsException(IpsExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Find entity instances in a specified range.
     *
     * @param range the range
     * @return the entity instances between range[0] and range[1]
     * @throws IpsException if the find fails
     */
    public List<T> findRange(int[] range) throws IpsException {
        try {
            javax.persistence.criteria.CriteriaQuery cq = getEntityManager().getCriteriaBuilder().createQuery();
            cq.select(cq.from(entityClass));
            javax.persistence.Query q = getEntityManager().createQuery(cq);
            q.setMaxResults(range[1] - range[0] + 1);
            q.setFirstResult(range[0]);
            return q.getResultList();
        } catch (IllegalArgumentException ex) {
            throw new IpsException(IpsExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (Exception ex) {
            throw new IpsException(IpsExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

    /**
     * Count the number of the entity instances.
     *
     * @return the count
     * @throws IpsException if the count fails
     */
    public int count() throws IpsException {
        try {
            javax.persistence.criteria.CriteriaQuery cq = getEntityManager().getCriteriaBuilder().createQuery();
            javax.persistence.criteria.Root<T> rt = cq.from(entityClass);
            cq.select(getEntityManager().getCriteriaBuilder().count(rt));
            javax.persistence.Query q = getEntityManager().createQuery(cq);
            return ((Long) q.getSingleResult()).intValue();
        } catch (IllegalArgumentException ex) {
            throw new IpsException(IpsExceptionType.BAD_PARAMETER,
                    ex.getLocalizedMessage());
        } catch (Exception ex) {
            throw new IpsException(IpsExceptionType.INTERNAL,
                    ex.getLocalizedMessage());
        }
    }

}
