package cn.ac.csns.portal.proposalsystem.managed;

import cn.ac.csns.portal.proposalsystem.entity.FacilityCycle;
import cn.ac.csns.portal.proposalsystem.entity.Grouping;
import cn.ac.csns.portal.proposalsystem.entity.Instrument;
import cn.ac.csns.portal.proposalsystem.entity.Investigation;
import cn.ac.csns.portal.proposalsystem.entity.InvestigationGroup;
import cn.ac.csns.portal.proposalsystem.entity.InvestigationInstrument;
import cn.ac.csns.portal.proposalsystem.entity.InvestigationType;
import cn.ac.csns.portal.proposalsystem.entity.InvestigationUser;
import cn.ac.csns.portal.proposalsystem.entity.Keyword;
import cn.ac.csns.portal.proposalsystem.entity.Sample;
import cn.ac.csns.portal.proposalsystem.entity.SampleType;
import cn.ac.csns.portal.proposalsystem.entity.Study;
import cn.ac.csns.portal.proposalsystem.entity.StudyInvestigation;
import cn.ac.csns.portal.proposalsystem.entity.User;
import cn.ac.csns.portal.proposalsystem.exception.IpsException;
import cn.ac.csns.portal.proposalsystem.facade.InvestigationFacade;
import java.io.Serializable;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Date;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.annotation.PostConstruct;
import javax.annotation.PreDestroy;
import javax.ejb.EJB;
import javax.faces.application.FacesMessage;
import javax.faces.context.ExternalContext;
import javax.faces.context.FacesContext;
import javax.inject.Named;
import javax.faces.view.ViewScoped;
import javax.inject.Inject;
import javax.transaction.Transactional;
import org.primefaces.event.RowEditEvent;

/**
 *
 * @author tangm <a href="mailto:tangm@ihep.ac.cn">Tang Ming</a>
 */
@Named
@ViewScoped
public class EditProposalController implements Serializable {

    private static final String KEYWORDS_SEPARATOR = ",";

    private Investigation current;
    private Instrument preferredInstrument;
    private List<InvestigationType> investigationTypes;
    private List<FacilityCycle> facilityCycles;
    private List<Instrument> instruments;
    private List<SampleType> sampleTypes;
    private String keywordsStr;
    private String currentUser;
    private Study selectedStudy;
    private User selectedUser;
    private Grouping selectedGroup;
    private boolean primaryGroup = false;
    private boolean primaryUser = false;
    private boolean keywordsChanged = false;
    private boolean preferredInstrumentChanged = false;
    private boolean samplesChanged = false;
    private boolean studyInvestigationChanged = false;
    private boolean investigationUsersChanged = false;
    private boolean investigationGroupsChanged = false;

    @EJB
    private InvestigationFacade facade;
    @Inject
    private AppDataController appData;

    /**
     * Creates a new instance of EditProposalController
     */
    public EditProposalController() {
    }

    @PostConstruct
    private void init() {
        String message;
        ExternalContext externalContext = FacesContext.getCurrentInstance().getExternalContext();
        currentUser = externalContext.getRemoteUser();
        current = (Investigation) externalContext.getSessionMap().get("currentEditInvestigation");
        if (current == null) {
            message = "No proposal found.";
            FacesContext.getCurrentInstance().addMessage(null,
                    new FacesMessage(FacesMessage.SEVERITY_INFO, message, message));
            return;
        }
        keywordsStr = getKeywordsStrFromInvestigation(current);
        investigationTypes = appData.getInvestigationTypes().stream()
                .filter(it -> it.getFacility().equals(current.getFacility()))
                .collect(Collectors.toList());
        facilityCycles = appData.getFacilityCycles().stream()
                .filter(fc -> fc.getFacility().equals(current.getFacility()))
                .sorted((o1, o2) -> o2.getCreateTime().compareTo(o1.getCreateTime()))
                .collect(Collectors.toList());
        instruments = appData.getInstruments().stream()
                .filter(ist -> ist.getFacility().equals(current.getFacility()))
                .sorted((o1, o2) -> o1.getName().compareToIgnoreCase(o2.getName()))
                .collect(Collectors.toList());
        if (current.getInvestigationInstruments().size() > 0) {
            preferredInstrument = current.getInvestigationInstruments().get(0).getInstrument();
        } else {
            preferredInstrumentChanged = true;
        }
    }

    @PreDestroy
    private void clear() {
        preferredInstrumentChanged = false;
        keywordsChanged = false;
        samplesChanged = false;
        studyInvestigationChanged = false;
        primaryGroup = false;
        primaryUser = false;
        investigationGroupsChanged = false;
        investigationUsersChanged = false;
        FacesContext.getCurrentInstance().getExternalContext()
                .getSessionMap().remove("currentEditInvestigation");
    }

    public void onKeywordsChange() {
        keywordsChanged = true;
    }

    public void onPreferredInstrumentChange() {
        preferredInstrumentChanged = true;
    }

    public void onRowEdit(RowEditEvent event) {
        Sample edited = (Sample) event.getObject();
        current.getSamples().stream()
                .filter((s) -> (s.getId().equals(edited.getId())))
                .forEach((s) -> {
                    s.setModId(currentUser);
                    s.setModTime(new Date());
                });
        samplesChanged = true;
    }

    public void onRowCancel(RowEditEvent event) {
    }

    public void removeSample(Sample sample) {
        current.getSamples().remove(sample);
        samplesChanged = true;
    }

    public void removeStudyInvestigation(StudyInvestigation studyInvestigation) {
        current.getStudyInvestigations().remove(studyInvestigation);
        studyInvestigationChanged = true;
    }

    public void addStudyInvestigation() {
        if (selectedStudy == null) {
            return;
        }
        for (StudyInvestigation si : current.getStudyInvestigations()) {
            if (si.getStudy().equals(selectedStudy)) {
                return;
            }
        }
        Date now = new Date();
        StudyInvestigation newsi = new StudyInvestigation();
        newsi.setCreateId(currentUser);
        newsi.setCreateTime(now);
        newsi.setModId(currentUser);
        newsi.setModTime(now);
        newsi.setInvestigation(current);
        newsi.setStudy(selectedStudy);
        current.getStudyInvestigations().add(newsi);
        studyInvestigationChanged = true;
    }

    private void updateStudyInvestigations(Investigation investigation) throws IpsException {
        if (studyInvestigationChanged) {
            investigation.getStudyInvestigations().clear();
            facade.flush();
            investigation.setStudyInvestigations(current.getStudyInvestigations());
        }
    }

    public void addInvestigationGroup() {
        if (selectedGroup == null) {
            return;
        }
        for (InvestigationGroup ig : current.getInvestigationGroups()) {
            if (selectedGroup.equals(ig.getGroup())) {
                return;
            }
        }
        Date now = new Date();
        InvestigationGroup newig = new InvestigationGroup();
        newig.setCreateId(currentUser);
        newig.setCreateTime(now);
        newig.setModId(currentUser);
        newig.setModTime(now);
        newig.setGroupRole("Common");
        if (primaryGroup) {
            newig.setGroupRole("Primary");
        }
        newig.setGroup(selectedGroup);
        newig.setInvestigation(current);
        current.getInvestigationGroups().add(newig);
        primaryGroup = false;
        investigationGroupsChanged = true;
    }

    public void removeInvestigationGroup(InvestigationGroup investigationGroup) {
        current.getInvestigationGroups().remove(investigationGroup);
        investigationGroupsChanged = true;
    }

    private void updateInvestigationGroups(Investigation investigation) throws IpsException {
        if (investigationGroupsChanged) {
            investigation.getInvestigationGroups().clear();
            facade.flush();
            investigation.setInvestigationGroups(current.getInvestigationGroups());
        }
    }

    public void addInvestigationUser() {
        if (selectedUser == null) {
            return;
        }
        for (InvestigationUser iu : current.getInvestigationUsers()) {
            if (selectedUser.equals(iu.getUser())) {
                return;
            }
        }
        Date now = new Date();
        InvestigationUser newiu = new InvestigationUser();
        newiu.setCreateId(currentUser);
        newiu.setCreateTime(now);
        newiu.setModId(currentUser);
        newiu.setModTime(now);
        newiu.setUserRole("Participator");
        if (primaryUser) {
            newiu.setUserRole("Primary");
        }
        newiu.setUser(selectedUser);
        newiu.setInvestigation(current);
        current.getInvestigationUsers().add(newiu);
        primaryUser = false;
        investigationUsersChanged = true;
    }

    public void removeInvestigationUser(InvestigationUser investigationUser) {
        current.getInvestigationUsers().remove(investigationUser);
        investigationUsersChanged = true;
    }

    private void updateInvestigationUsers(Investigation investigation) throws IpsException {
        if (investigationUsersChanged) {
            investigation.getInvestigationUsers().clear();
            facade.flush();
            investigation.setInvestigationUsers(current.getInvestigationUsers());
        }
    }

    @Transactional
    public String doUpdate() {
        String message;
        try {
            Investigation investigationEditing = facade.find(current.getId());
            investigationEditing.setModTime(new Date());
            investigationEditing.setFacilityCycle(current.getFacilityCycle());
            investigationEditing.setName(current.getName());
            investigationEditing.setTitle(current.getTitle());
            investigationEditing.setType(current.getType());
            updateKeywords(investigationEditing);
            investigationEditing.setSummary(current.getSummary());
            updatePreferredInstrument(investigationEditing, preferredInstrument);
            updateSamples(investigationEditing);
            updateStudyInvestigations(investigationEditing);
            updateInvestigationGroups(investigationEditing);
            updateInvestigationUsers(investigationEditing);
        } catch (IpsException ex) {
            Logger.getLogger(EditProposalController.class.getName())
                    .log(Level.SEVERE, ex.getLocalizedMessage(), ex);
            message = "Cannot retrive proposal";
            FacesContext.getCurrentInstance().addMessage(null,
                    new FacesMessage(FacesMessage.SEVERITY_FATAL, message, message));
        }
        return "/index?faces-redirect=true";
    }

    public String doCancle() {
        clear();
        return "/index?faces-redirect=true";
    }

    public List<SampleType> getSampleTypes() {
        if (current != null && current.getFacility() != null) {
            return appData.getSampleTypes().stream()
                    .filter(st -> st.getFacility().equals(current.getFacility()))
                    .sorted((SampleType o1, SampleType o2)
                            -> o1.getName().compareToIgnoreCase(o2.getName()))
                    .collect(Collectors.toList());
        }
        return new ArrayList<>();
    }

    private List<String> processKeywordsStr() {
        return Arrays.<String>asList(keywordsStr.split(KEYWORDS_SEPARATOR))
                .stream().map(String::trim)
                .filter(item -> !item.isEmpty())
                .distinct().collect(Collectors.toList());
    }

    private List<Keyword> createInvetigationKeywords(List<String> keywordsList, Investigation investigation) {
        List<Keyword> keywords = new ArrayList<>();
        Date now = new Date();
        keywordsList.stream().forEach(item -> {
            Keyword keyword = new Keyword();
            keyword.setCreateId(currentUser);
            keyword.setCreateTime(now);
            keyword.setModId(currentUser);
            keyword.setModTime(now);
            keyword.setName(item);
            keyword.setInvestigation(investigation);
            keywords.add(keyword);
        });
        return keywords;
    }

    private void updateKeywords(Investigation investigation) throws IpsException {
        if (keywordsChanged) {
            clearKeywords(investigation);
            List<Keyword> keywords = createInvetigationKeywords(
                    processKeywordsStr(), investigation);
            investigation.setKeywords(keywords);
        }
    }

    private void updatePreferredInstrument(Investigation investigation, Instrument instrument)
            throws IpsException {
        if (preferredInstrumentChanged) {
            investigation.getInvestigationInstruments().clear();
            facade.flush();
            InvestigationInstrument ii = new InvestigationInstrument();
            Date now = new Date();
            ii.setCreateId(currentUser);
            ii.setCreateTime(now);
            ii.setModId(currentUser);
            ii.setModTime(now);
            ii.setInstrument(instrument);
            ii.setInvestigation(investigation);
            investigation.getInvestigationInstruments().add(ii);
        }
    }

    private void updateSamples(Investigation investigation) throws IpsException {
        if (samplesChanged) {
            investigation.getSamples().clear();
            facade.flush();
            investigation.setSamples(current.getSamples());
        }
    }

    private void clearKeywords(Investigation investigation) throws IpsException {
        investigation.getKeywords().clear();
        facade.flush();
    }

    private String getKeywordsStrFromInvestigation(Investigation investigaiton) {
        StringBuilder sb = new StringBuilder();
        investigaiton.getKeywords().stream().forEach(item -> {
            sb.append(item.getName());
            sb.append(", ");
        });
        return sb.toString();
    }

    public Investigation getCurrent() {
        return current;
    }

    public void setCurrent(Investigation current) {
        this.current = current;
    }

    public List<InvestigationType> getInvestigationTypes() {
        return investigationTypes;
    }

    public String getKeywordsStr() {
        return keywordsStr;
    }

    public Instrument getPreferredInstrument() {
        return preferredInstrument;
    }

    public void setPreferredInstrument(Instrument preferredInstrument) {
        this.preferredInstrument = preferredInstrument;
    }

    public void setKeywordsStr(String keywordsStr) {
        this.keywordsStr = keywordsStr;
    }

    public List<FacilityCycle> getFacilityCycles() {
        return facilityCycles;
    }

    public List<Instrument> getInstruments() {
        return instruments;
    }

    public Study getSelectedStudy() {
        return selectedStudy;
    }

    public void setSelectedStudy(Study selectedStudy) {
        this.selectedStudy = selectedStudy;
    }

    public User getSelectedUser() {
        return selectedUser;
    }

    public void setSelectedUser(User selectedUser) {
        this.selectedUser = selectedUser;
    }

    public Grouping getSelectedGroup() {
        return selectedGroup;
    }

    public void setSelectedGroup(Grouping selectedGroup) {
        this.selectedGroup = selectedGroup;
    }

    public boolean isPrimaryGroup() {
        return primaryGroup;
    }

    public void setPrimaryGroup(boolean primaryGroup) {
        this.primaryGroup = primaryGroup;
    }

    public boolean isPrimaryUser() {
        return primaryUser;
    }

    public void setPrimaryUser(boolean primaryUser) {
        this.primaryUser = primaryUser;
    }

}
