package cn.ac.csns.icat;

import java.net.MalformedURLException;
import java.net.URL;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.List;
import java.util.Properties;
import java.util.logging.Level;
import java.util.logging.Logger;
import java.util.stream.Collectors;
import javax.xml.namespace.QName;
import org.icatproject.EntityBaseBean;
import org.icatproject.Grouping;
import org.icatproject.ICAT;
import org.icatproject.ICATService;
import org.icatproject.IcatException_Exception;
import org.icatproject.Login.Credentials;
import org.icatproject.Login.Credentials.Entry;
import org.icatproject.PublicStep;
import org.icatproject.Rule;
import org.icatproject.User;
import org.icatproject.UserGroup;

/**
 * Add default rules to ICAT server.
 *
 * @author lanjian
 */
public class DataPolicyManager {

    private ICAT icat;
    private String sessionId;
    private Properties properties;

    public DataPolicyManager() {
    }

    private void init() {
        properties = AppHelper.getProperties();
        try {
            URL url = new URL(properties.getProperty("icat.url"));
            QName qName = new QName("http://icatproject.org", "ICATService");
            ICATService service = new ICATService(url, qName);
            icat = service.getICATPort();
        } catch (MalformedURLException ex) {
            Logger.getLogger(DataPolicyManager.class.getName())
                    .log(Level.SEVERE, ex.getLocalizedMessage(), ex);
            throw new RuntimeException(ex.getLocalizedMessage(), ex);
        }
    }

    private void login(String plugin, String username, String password) {
        Credentials credentials = new Credentials();
        List<Entry> entries = credentials.getEntry();
        Entry entry;
        entry = new Entry();
        entry.setKey("username");
        entry.setValue(username);
        entries.add(entry);
        entry = new Entry();
        entry.setKey("password");
        entry.setValue(password);
        entries.add(entry);
        try {
            sessionId = icat.login(plugin, credentials);
            Logger.getLogger(DataPolicyManager.class.getName())
                    .log(Level.INFO, "User {0} logged in.", icat.getUserName(sessionId));
        } catch (IcatException_Exception ex) {
            Logger.getLogger(DataPolicyManager.class.getName())
                    .log(Level.SEVERE, ex.getLocalizedMessage(), ex);
            throw new RuntimeException(ex.getLocalizedMessage(), ex);
        }
    }

    private void logout() {
        try {
            icat.logout(sessionId);
            Logger.getLogger(DataPolicyManager.class.getName())
                    .log(Level.INFO, "User {0} logged out.", icat.getUserName(sessionId));
        } catch (IcatException_Exception ex) {
            Logger.getLogger(DataPolicyManager.class.getName())
                    .log(Level.SEVERE, ex.getLocalizedMessage(), ex);
        }
    }

    private void clearAllRules() throws IcatException_Exception {
        List<EntityBaseBean> allGroups = icat.search(sessionId, "Grouping")
                .stream()
                .filter(EntityBaseBean.class::isInstance)
                .map(EntityBaseBean.class::cast)
                .collect(Collectors.toList());
        icat.deleteMany(sessionId, allGroups);
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Clear all Groupings.");

        List<EntityBaseBean> allRules = icat.search(sessionId, "Rule")
                .stream()
                .filter(EntityBaseBean.class::isInstance)
                .map(EntityBaseBean.class::cast)
                .collect(Collectors.toList());
        icat.deleteMany(sessionId, allRules);
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Clear all Rules.");

        List<EntityBaseBean> allPublicSteps = icat.search(sessionId, "PublicStep")
                .stream()
                .filter(EntityBaseBean.class::isInstance)
                .map(EntityBaseBean.class::cast)
                .collect(Collectors.toList());
        icat.deleteMany(sessionId, allPublicSteps);
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Clear all PublicSteps.");
    }

    private List<String> getMultipleValuesFromProperty(String propertyKey) {
        return Arrays.<String>asList(properties.getProperty(propertyKey).trim().split(","))
                .stream()
                .map(String::trim)
                .filter(user -> !user.isEmpty())
                .collect(Collectors.toList());
    }

    private User createUser(String username) throws IcatException_Exception {
        String query = "SELECT u FROM User u WHERE u.name = '" + username + "'";
        List<User> result = icat.search(sessionId, query).stream()
                .filter(User.class::isInstance)
                .map(User.class::cast)
                .collect(Collectors.toList());
        if (result.isEmpty()) {
            User newUser = new User();
            newUser.setName(username);
            newUser.setId(icat.create(sessionId, newUser));
            return newUser;
        }
        return result.get(0);
    }

    private Grouping createGroup(String groupname) throws IcatException_Exception {
        String query = "SELECT g FROM Grouping g WHERE g.name = '" + groupname + "'";
        List<Grouping> result = icat.search(sessionId, query).stream()
                .filter(Grouping.class::isInstance)
                .map(Grouping.class::cast)
                .collect(Collectors.toList());
        if (result.isEmpty()) {
            Grouping newGroup = new Grouping();
            newGroup.setName(groupname);
            newGroup.setId(icat.create(sessionId, newGroup));
            return newGroup;
        }
        return result.get(0);
    }

    private UserGroup createUserGroup(User user, Grouping group) throws IcatException_Exception {
        String query = "SELECT ug1 FROM UserGroup ug1 "
                + "JOIN ug1.user u "
                + "JOIN u.userGroups ug2 "
                + "JOIN ug2.grouping g "
                + "WHERE u.name = '" + user.getName() + "' "
                + "AND g.name = '" + group.getName() + "'";
        List<UserGroup> result = icat.search(sessionId, query).stream()
                .filter(UserGroup.class::isInstance)
                .map(UserGroup.class::cast)
                .collect(Collectors.toList());
        if (result.isEmpty()) {
            UserGroup newUserGroup = new UserGroup();
            newUserGroup.setUser(user);
            newUserGroup.setGrouping(group);
            newUserGroup.setId(icat.create(sessionId, newUserGroup));
            return newUserGroup;
        }
        return result.get(0);
    }

    private void createFacilityAdministratorRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create rules for facility administrators.");
        
        // Create facility admin group
        String facilityAdminGroupName = properties.getProperty("icat.facility.admin.group").trim();
        Grouping facilityAdminGroup = createGroup(facilityAdminGroupName);

        // Create facility admins as needed.
        List<String> facilityAdmins = getMultipleValuesFromProperty("icat.facility.admins");
        List<User> facilityAdminUsers = new ArrayList<>();
        for (String user : facilityAdmins) {
            User doiReaderUser = createUser(user);
            facilityAdminUsers.add(doiReaderUser);
        }
        for (User u : facilityAdminUsers) {
            createUserGroup(u, facilityAdminGroup);
        }
        
        List<String> allTables = icat.getEntityNames();
        for (String table : allTables) {
            Rule rule = new Rule();
            rule.setGrouping(facilityAdminGroup);
            rule.setCrudFlags("CRUD");
            rule.setWhat(table);
            icat.create(sessionId, rule);
        }
    }

    private void createPublicTablesRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create rules for public tables.");

        List<String> publicTables = new ArrayList<>();

        publicTables.add("Application");
        publicTables.add("DatafileFormat");
        publicTables.add("DatasetType");
        publicTables.add("Facility");
        publicTables.add("FacilityCycle");
        publicTables.add("Instrument");
        publicTables.add("InstrumentScientist");
        publicTables.add("InvestigationType");
        publicTables.add("ParameterType");
        publicTables.add("PermissibleStringValue");
        publicTables.add("Publication");
        publicTables.add("Shift");
        publicTables.add("User");

        publicTables.add("DataCollectionDatafile");
        publicTables.add("DataCollectionDataset");
        publicTables.add("InvestigationUser");
        publicTables.add("StudyInvestigation");

        List<EntityBaseBean> publicRules = new ArrayList<>();
        publicTables.stream()
                .forEach(publicTableName -> {
                    Rule publicRule = new Rule();
                    publicRule.setWhat(publicTableName);
                    publicRule.setCrudFlags("R");
                    publicRules.add(publicRule);
                });
        icat.createMany(sessionId, publicRules);
    }

    private void createPublicSteps() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create public steps.");

        List<EntityBaseBean> publicSteps = new ArrayList<>();

        List<String> parameterBasedPublicSteps = new ArrayList<>();
        parameterBasedPublicSteps.add("Investigation");
        parameterBasedPublicSteps.add("Dataset");
        parameterBasedPublicSteps.add("Datafile");
        parameterBasedPublicSteps.add("Sample");
        parameterBasedPublicSteps.stream().forEach(table -> {
            PublicStep parameterPublicStep = new PublicStep();
            parameterPublicStep.setOrigin(table);
            parameterPublicStep.setField("parameters");
            publicSteps.add(parameterPublicStep);
        });

        String[] publicStepsFromInvestigation = {"samples",
            "publications",
            "shifts",
            "investigationUsers",
            "keywords",
            "investigationInstruments"};
        Arrays.<String>asList(publicStepsFromInvestigation)
                .stream()
                .forEach(step -> {
                    PublicStep invPublicStep = new PublicStep();
                    invPublicStep.setOrigin("Investigation");
                    invPublicStep.setField(step);
                    publicSteps.add(invPublicStep);
                });

        String[] publicStepsFromDataset = new String[]{"sample", "datafiles"};
        Arrays.<String>asList(publicStepsFromDataset)
                .stream()
                .forEach(step -> {
                    PublicStep datasetPublicStep = new PublicStep();
                    datasetPublicStep.setOrigin("Dataset");
                    datasetPublicStep.setField(step);
                    publicSteps.add(datasetPublicStep);
                });

        PublicStep sampleToSampleType = new PublicStep();
        sampleToSampleType.setOrigin("Sample");
        sampleToSampleType.setField("type");
        publicSteps.add(sampleToSampleType);

        PublicStep userToUserGroup = new PublicStep();
        userToUserGroup.setOrigin("User");
        userToUserGroup.setField("userGroups");
        publicSteps.add(userToUserGroup);

        PublicStep userGroupToGrouping = new PublicStep();
        userGroupToGrouping.setOrigin("UserGroup");
        userGroupToGrouping.setField("grouping");
        publicSteps.add(userGroupToGrouping);

        icat.createMany(sessionId, publicSteps);
    }

    private void createDataIngestorRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create rules for data ingestors.");
        String dataIngestorGroupString = properties.getProperty("icat.data.ingestor.group").trim();
        Grouping dataIngestorGroup = createGroup(dataIngestorGroupString);

        // Create data investors as needed.
        List<String> dataIngestors = getMultipleValuesFromProperty("icat.data.ingestor.users");
        List<User> dataIngestorUsers = new ArrayList<>();
        for (String user : dataIngestors) {
            User dataIngestorUser = createUser(user);
            dataIngestorUsers.add(dataIngestorUser);
        }
        for (User u : dataIngestorUsers) {
            createUserGroup(u, dataIngestorGroup);
        }

        List<String> ingestorTables = icat.getEntityNames();
        ingestorTables.remove("Facility");
        List<EntityBaseBean> ingestorRules = new ArrayList<>();
        ingestorTables.stream().map((table) -> {
            Rule rule = new Rule();
            rule.setGrouping(dataIngestorGroup);
            rule.setCrudFlags("CRU"); // No delete permission
            rule.setWhat(table);
            return rule;
        }).forEachOrdered((rule) -> {
            ingestorRules.add(rule);
        });
        icat.createMany(sessionId, ingestorRules);
    }

    private void createInstrumentScientistRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create rules for instrument scientists.");
        Rule rule;
        String what;

        // Investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        rule.setWhat(sessionId);
        what = "SELECT i FROM Investigation i "
                + "JOIN i.investigationInstruments ii "
                + "JOIN ii.instrument inst "
                + "JOIN inst.instrumentScientists instSci "
                + "JOIN instSci.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        rule.setGrouping(null); // no special group needed.
        icat.create(sessionId, rule);

        String csnsDatasetNameCriteria = properties.getProperty("csns.dataset.name.pattern");

        // Dataset
        rule = new Rule();
        rule.setCrudFlags("CRU");
        what = "SELECT ds FROM Dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationInstruments ii "
                + "JOIN ii.instrument inst "
                + "JOIN inst.instrumentScientists instSci "
                + "JOIN instSci.user u "
                + "WHERE ds.name " + csnsDatasetNameCriteria + " AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Datafile
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT df FROM Datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationInstruments ii "
                + "JOIN ii.instrument inst "
                + "JOIN inst.instrumentScientists instSci "
                + "JOIN instSci.user u "
                + "WHERE ds.name " + csnsDatasetNameCriteria + " AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Samples - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT s FROM Sample s "
                + "JOIN s.investigation i "
                + "JOIN i.investigationInstruments ii "
                + "JOIN ii.instrument inst "
                + "JOIN inst.instrumentScientists instSci "
                + "JOIN instSci.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Samples - via dataset
        // Experiment samples may differ from investigation samples.
        rule = new Rule();
        rule.setCrudFlags("CR");
        what = "SELECT s FROM Sample AS s "
                + "JOIN s.datasets AS ds "
                + "JOIN ds.investigation AS i "
                + "JOIN i.investigationInstruments AS ii "
                + "JOIN ii.instrument AS inst "
                + "JOIN inst.instrumentScientists AS instSci "
                + "JOIN instSci.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample type - via investigation
        rule = new Rule();
        rule.setCrudFlags("CR");
        what = "SELECT st FROM SampleType st "
                + "JOIN st.samples s "
                + "JOIN s.investigation i "
                + "JOIN i.investigationInstruments ii "
                + "JOIN ii.instrument inst "
                + "JOIN inst.instrumentScientists instSci "
                + "JOIN instSci.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample type - via dataset
        rule = new Rule();
        rule.setCrudFlags("CR");
        what = "SELECT st FROM SampleType st "
                + "JOIN st.samples s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationInstruments ii "
                + "JOIN ii.instrument inst "
                + "JOIN inst.instrumentScientists instSci "
                + "JOIN instSci.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample parameter - via investigation
        rule = new Rule();
        rule.setCrudFlags("CR");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.investigation i "
                + "JOIN i.investigationInstruments ii "
                + "JOIN ii.instrument inst "
                + "JOIN inst.instrumentScientists instSci "
                + "JOIN instSci.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample parameter - via dataset
        rule = new Rule();
        rule.setCrudFlags("CR");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.datasets d "
                + "JOIN d.investigation i "
                + "JOIN i.investigationInstruments ii "
                + "JOIN ii.instrument inst "
                + "JOIN inst.instrumentScientists instSci "
                + "JOIN instSci.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);
    }

    private void createInvestigationUserRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create rules for investigation users.");
        Rule rule;
        String what;

        // Investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT i FROM Investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // InvestigationParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT ip FROM InvestigationParameter ip "
                + "JOIN ip.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT ds FROM Dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatasetParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT dsp FROM DatasetParameter dsp "
                + "JOIN dsp.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Datafile
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT df FROM Datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatafileParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT dfp FROM DatafileParameter dfp "
                + "JOIN dfp.datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT s FROM Sample s "
                + "JOIN s.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT s FROM Sample s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleParameter - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleParameter - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        //********************************************
        //** Operations to non-CSNS experiment data **
        //********************************************
        String userDatasetNameCriteria = properties.getProperty("user.dataset.name.pattern");
        // Dataset 
        rule = new Rule();
        rule.setCrudFlags("CRUD");
        what = "SELECT ds FROM Dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE ds.name " + userDatasetNameCriteria + " AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatasetParameter 
        rule = new Rule();
        rule.setCrudFlags("CRUD");
        what = "SELECT dsp FROM DatasetParameter dsp "
                + "JOIN dsp.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE ds.name " + userDatasetNameCriteria + " AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Datafile 
        rule = new Rule();
        rule.setCrudFlags("CRUD");
        what = "SELECT df FROM Datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE ds.name " + userDatasetNameCriteria + " AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatafileParameter 
        rule = new Rule();
        rule.setCrudFlags("CRUD");
        what = "SELECT dfp FROM DatafileParameter dfp "
                + "JOIN dfp.datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE ds.name " + userDatasetNameCriteria + " AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample - via dataset 
        rule = new Rule();
        rule.setCrudFlags("CRUD");
        what = "SELECT s FROM Sample s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE ds.name " + userDatasetNameCriteria + " AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleParameter - via dataset 
        rule = new Rule();
        rule.setCrudFlags("CRUD");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE ds.name " + userDatasetNameCriteria + " AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);
    }

    /**
     * Pulic access to old data.
     */
    private void createPublicReleasedDataRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create rules for public released data.");

        Rule rule;
        String what;

        // Investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT i FROM Investigation i WHERE i.releaseDate < CURRENT_TIMESTAMP";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // InvestigationParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT ip FROM InvestigationParameter ip "
                + "JOIN ip.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT ds FROM Dataset ds "
                + "JOIN ds.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        //+ " AND ds.name = 'Default'"; // TODO
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatasetParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT dsp FROM DatasetParameter dsp "
                + "JOIN dsp.dataset ds "
                + "JOIN ds.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        //+ " AND ds.name = 'Default'"; // TODO
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Datafile
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT df FROM Datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        //+ " AND ds.name = 'Default'"; // TODO
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatafileParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT dfp FROM DatafileParameter dfp "
                + "JOIN dfp.datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        //+ " AND ds.name = 'Default'"; // TODO
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT s FROM Sample s "
                + "JOIN s.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleParameter - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT s FROM Sample s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        //+ " AND ds.name = 'Default'"; // TODO
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleParameter - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        //+ " AND ds.name = 'Default'"; // TODO
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleType - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT st FROM SampleType st "
                + "JOIN st.samples s "
                + "JOIN s.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleType - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT st FROM SampleType st "
                + "JOIN st.samples s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "WHERE i.releaseDate < CURRENT_TIMESTAMP";
        //+ " AND ds.name = 'Default'"; // TODO
        rule.setWhat(what);
        icat.create(sessionId, rule);
    }

    private void createPublicCalibrationRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create rules for public calibration data.");
        String calibrationInvestigationType = properties
                .getProperty("csns.investigation.calibration.type")
                .trim();
        Rule rule;
        String what;

        // Calibration Investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT i FROM Investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // InvestigationParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT ip FROM InvestigationParameter ip "
                + "JOIN ip.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT ds FROM Dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatasetParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT dsp FROM DatasetParameter dsp "
                + "JOIN dsp.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Datafile
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT df FROM Datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatafileParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT dfp FROM DatafileParameter dfp "
                + "JOIN dfp.datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT s FROM Sample s "
                + "JOIN s.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleParameter - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT s FROM Sample s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleParameter - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleType - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT st FROM SampleType st "
                + "JOIN st.samples s "
                + "JOIN s.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleType - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT st FROM SampleType st "
                + "JOIN st.samples s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t "
                + "WHERE t.name = '" + calibrationInvestigationType + "'";
        rule.setWhat(what);
        icat.create(sessionId, rule);
    }
    
    private void createInstrumentCalibrationRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create rules for instrument calibration data.");
        String calibrationInvestigationType = properties
                .getProperty("csns.investigation.calibration.type")
                .trim();
        Rule rule;
        String what;

        // Calibration Investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT i FROM Investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // InvestigationParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT ip FROM InvestigationParameter ip "
                + "JOIN ip.investigation i "                
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT ds FROM Dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatasetParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT dsp FROM DatasetParameter dsp "
                + "JOIN dsp.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Datafile
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT df FROM Datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatafileParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT dfp FROM DatafileParameter dfp "
                + "JOIN dfp.datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT s FROM Sample s "
                + "JOIN s.investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleParameter - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT s FROM Sample s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleParameter - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleType - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT st FROM SampleType st "
                + "JOIN st.samples s "
                + "JOIN s.investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleType - via dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT st FROM SampleType st "
                + "JOIN st.samples s "
                + "JOIN s.datasets ds "
                + "JOIN ds.investigation i "
                + "JOIN i.type t JOIN i.investigationInstruments ii1 "
                + "JOIN ii1.instrument ins JOIN ins.investigationInstruments ii2 "
                + "JOIN ii2.investigation i2 JOIN i2.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE t.name = '" + calibrationInvestigationType + "' "
                + "AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);
    }

    private void createCoInvestigatorRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create rules for co-investigators.");
        Rule rule;
        String what;

        // Investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT i FROM Investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // InvestigationParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT ip FROM InvestigationParameter ip "
                + "JOIN ip.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Dataset
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT ds FROM Dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatasetParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT dsp FROM DatasetParameter dsp "
                + "JOIN dsp.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Datafile
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT df FROM Datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // DatafileParameter
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT dfp FROM DatafileParameter dfp "
                + "JOIN dfp.datafile df "
                + "JOIN df.dataset ds "
                + "JOIN ds.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Sample - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT s FROM Sample s "
                + "JOIN s.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // SampleParameter - via investigation
        rule = new Rule();
        rule.setCrudFlags("R");
        what = "SELECT sp FROM SampleParameter sp "
                + "JOIN sp.sample s "
                + "JOIN s.investigation i "
                + "JOIN i.investigationUsers iu "
                + "JOIN iu.user u "
                + "WHERE u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        //TODO: Sample[Parameter] - via dataset?
    }

    private void createPIDelegationRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create delegation rules for primary investigator.");

        String piRoleName = properties.getProperty("csns.investigation.pi.role.name").trim();
        Rule rule = new Rule();
        rule.setCrudFlags("C");
        String what = "SELECT iu1 FROM InvestigationUser iu1 "
                + "JOIN iu1.investigation i "
                + "JOIN i.investigationUsers iu2 "
                + "JOIN iu2.user u "
                + "WHERE iu2.role = '" + piRoleName + "' AND u.name = :user";
        rule.setWhat(what);
        icat.create(sessionId, rule);
    }

    /**
     * Rules for referencing Investigation, Dataset, Datafile from DOI.
     *
     * @throws IcatException_Exception
     */
    private void createDOIRules() throws IcatException_Exception {
        Logger.getLogger(DataPolicyManager.class.getName())
                .log(Level.INFO, "Create rules for DOI readers.");

        // Create DOI reader group
        String doiReaderGroupName = properties.getProperty("csns.doi.reader.group.name").trim();
        Grouping doiReaderGroup = createGroup(doiReaderGroupName);

        // Create DOI readers as needed.
        List<String> doiReaders = getMultipleValuesFromProperty("csns.doi.reader.users");
        List<User> doiReaserUsers = new ArrayList<>();
        for (String user : doiReaders) {
            User doiReaderUser = createUser(user);
            doiReaserUsers.add(doiReaderUser);
        }
        for (User u : doiReaserUsers) {
            createUserGroup(u, doiReaderGroup);
        }

        // Rules
        Rule rule;
        String what;

        // Investigation
        rule = new Rule();
        rule.setGrouping(doiReaderGroup);
        rule.setCrudFlags("R");
        what = "SELECT i FROM Investigation i WHERE i.doi IS NOT NULL";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Dataset
        rule = new Rule();
        rule.setGrouping(doiReaderGroup);
        rule.setCrudFlags("R");
        what = "SELECT ds FROM Dataset ds WHERE ds.doi IS NOT NULL";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Investigation with Dataset
        rule = new Rule();
        rule.setGrouping(doiReaderGroup);
        rule.setCrudFlags("R");
        what = "SELECT i FROM Investigation i "
                + "JOIN i.datasets ds WHERE ds.doi IS NOT NULL";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Datafile
        rule = new Rule();
        rule.setGrouping(doiReaderGroup);
        rule.setCrudFlags("R");
        what = "SELECT df FROM Datafile df WHERE df.doi IS NOT NULL";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Dataset with Datafile
        rule = new Rule();
        rule.setGrouping(doiReaderGroup);
        rule.setCrudFlags("R");
        what = "SELECT ds FROM Dataset ds "
                + "JOIN ds.datafiles df WHERE df.doi IS NOT NULL";
        rule.setWhat(what);
        icat.create(sessionId, rule);

        // Investigation with Datafile
        rule = new Rule();
        rule.setGrouping(doiReaderGroup);
        rule.setCrudFlags("R");
        what = "SELECT i FROM Investigation i "
                + "JOIN i.datasets ds "
                + "JOIN ds.datafiles df WHERE df.doi IS NOT NULL";
        rule.setWhat(what);
        icat.create(sessionId, rule);
    }

    public void run() {
        init();
        login(properties.getProperty("icat.plugin"),
                properties.getProperty("icat.user"),
                properties.getProperty("icat.password"));

        try {
            clearAllRules();
            createFacilityAdministratorRules();
            createPublicTablesRules();
            createPublicSteps();
            createDataIngestorRules();
            createInstrumentScientistRules();
            createInvestigationUserRules();
            createPublicReleasedDataRules();
            //createPublicCalibrationRules();  // not used in CSNS
            createInstrumentCalibrationRules();
            createCoInvestigatorRules();
            createPIDelegationRules();
            createDOIRules();
            Logger.getLogger(DataPolicyManager.class.getName())
                    .log(Level.INFO, "All done.");
        } catch (IcatException_Exception ex) {
            Logger.getLogger(DataPolicyManager.class.getName())
                    .log(Level.SEVERE, ex.getLocalizedMessage(), ex);
            throw new RuntimeException(ex.getLocalizedMessage(), ex.getCause());
        }

        //logout();
    }

}
